/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdUtil.h,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.3 $      $Date: 1994/10/05 06:58:16 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Command objects for doing various utilities, such as executing a shell,
 * displaying help, or quitting
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: CmdUtil.h,v $
 * Revision 1.3  1994/10/05  06:58:16  billh
 * Converted to use a ResizeArray instead of a DLinkList in the command queue
 * and ui list.
 *
 * Revision 1.2  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifndef CMDUTIL_H
#define CMDUTIL_H

#include <stdlib.h>
#include <string.h>
#include "Command.h"
#include "Global.h"
#include "UIList.h"
#include "UIText.h"
#include "DisplayDevice.h"
#include "utilities.h"

// The following uses the Cmdtypes "SHELL", "HELP", "RESET", "DEBUG", and
// "QUIT" from the Command class

//////////// Execute an external command (via the system() call)
class CmdShell : public Command {
public:
  // the command to execute
  char *cmd;

private:
  // single command, or just a shell?
  int justShell;

protected:
  virtual int do_execute(void) {
    int retval = FALSE;
    if(cmd) {
      if(justShell)
        msgInfo << "Executing shell (use 'exit' to return) ..." << sendmsg;
      retval = system(cmd);
      if(justShell)
        msgInfo << "\nReturning to program ..." << sendmsg;
    }
    return retval;
  }

public:
  CmdShell(char *newcmd, int newUIid=(-1)) : Command(Command::SHELL, newUIid) {
    char *envtxt;

    // save copy of the command to execute.  Must append shell command
    if((envtxt = getenv("SHELL")) != NULL) {
      justShell = (newcmd == NULL);
      if(newcmd)
        cmd = new char[strlen(newcmd) + strlen(envtxt) + 2];
      else
	cmd = new char[strlen(newcmd) + 1];
      strcpy(cmd,envtxt);
      if(newcmd) {
	strcat(cmd," ");
	strcat(cmd,newcmd);
      }
    } else if(newcmd) {
      cmd = new char[strlen(envtxt) + 4];
      strcpy(cmd,"sh ");
      strcat(cmd,newcmd);
    } else
      cmd = NULL;
      
    *cmdText << "sh ";
    if(newcmd)
      *cmdText << newcmd;
    *cmdText << ends;
  }
    
  virtual ~CmdShell(void) {
    if(cmd) delete [] cmd;
  }
};

//////////// display help, in various formats
class CmdHelp : public Command {
public:
  // ways that help can be displayed (currently, only to console)
  enum HelpLoc { HELP_CONSOLE };
    
  // where to display help
  HelpLoc helpLoc;

protected:
  // do the actual action
  virtual int do_execute(void) {
    int retval = FALSE;
    if(helpLoc == HELP_CONSOLE) {
      uiText->help();
      retval = TRUE;
    }
    return retval;
  }

public:
  CmdHelp(HelpLoc loc, int newUIid=(-1)) : Command(Command::HELP, newUIid) {
    helpLoc = loc;
  }
};


//////////// reset all the UI's
class CmdReset : public Command {
private:
  // which UI to reset.  if < 0, reset all
  int resetUI;
    
protected:
  // do the actual action
  virtual int do_execute(void) {
    int retval = TRUE;
    if(resetUI < 0) {
      if(display)
        display->reshape();
      uiList->reset_UI();
    } else if(resetUI >= 0 && resetUI < uiList->num()) {
      (uiList->item(resetUI))->reset();
    } else {
      retval = FALSE;
    }
    return retval;
  }

public:
  CmdReset(int rUI=(-1), int newUIid=(-1)) : Command(Command::RESET, newUIid) {
    resetUI = rUI;
    *cmdText << "reset ";
    if(resetUI < 0)
      *cmdText << "all";
    else if(resetUI >= 0 && resetUI < uiList->num())
      *cmdText << (uiList->item(resetUI))->name;
    *cmdText << ends;
  }
};


//////////// exit the program, possibly asking for confirmation
class CmdQuit : public Command {
private:
  // ask for confirmation?
  int askConfirm;

protected:
  // do the actual action
  virtual int do_execute(void) {
    int doexit = TRUE;
    if(askConfirm)
      // for now, does nothing
      ;
    if(doexit)
      VMDexit("Exiting normally", 0);

    return FALSE;		// only get here if confirm = no
  }

public:
  CmdQuit(int confirm, int newUIid=(-1)) : Command(Command::QUIT, newUIid) {
    askConfirm = confirm;
    *cmdText << "quit";
    if(askConfirm)
      *cmdText << " confirm";
    else
      *cmdText << " now";
    *cmdText << ends;
  }
};


//////////// set the debug level for the program
class CmdDebug : public Command {
public:
  // are we changing the debug level, or turning on/off?
  enum DebugAction { TOGGLE, LEVEL };
  DebugAction action;

  // new debug level, or whether to turn on/off
  int newVal;
    
protected:
  // do the actual action
  virtual int do_execute(void) {
    int retval = TRUE;
    if(action == TOGGLE) {
      msgDebug.on(newVal);
    } else if(action == LEVEL) {
      msgDebug.output_level(newVal);
    } else {
      retval = FALSE;
    }
    return retval;
  }

public:
  CmdDebug(DebugAction actn, int nval, int newUIid=(-1))
    : Command(Command::DEBUG, newUIid) {
    action = actn;
    newVal = nval;
    *cmdText << "debug ";
    if(action == TOGGLE) {
      *cmdText << (nval ? "on" : "off");
    } else if(action == LEVEL) {
      *cmdText << "level " << nval;
    }
    *cmdText << ends;
  }
};


#endif
