/************************************************************/
/**                                                        **/
/**                   CAVE.H  		                   **/
/**                                                        **/
/**  This files contains the definitions required by       **/
/**  cave.onyx.c                                           **/
/**                                                        **/
/************************************************************/
/*
*	Copyright 1994, Electronic Visualization Laboratory.
*	All rights reserved.
*
*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF EVL.
*	Unless otherwise agreed to in writing by EVL, this code is
*	to be used and accessed only by SIGGRAPH VROOM participants.
*	It is not public domain software.  Call Margaret Rawlings at
*	(312)-996-3002 for details.
*
*/
#ifndef	__CAVE_H__
#define	__CAVE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <math.h>
#include <gl/gl.h>
#include <gl/device.h>

#define	CAVE_MAX_WALLS		6
#define  CAVE_FRONT_WALL  	1
#define  CAVE_FLOOR_WALL  	2
#define  CAVE_RIGHT_WALL  	3
#define  CAVE_LEFT_WALL   	4
#define  CAVE_CEILING_WALL	5
#define  CAVE_BACK_WALL		6
#define CAVE_MAX_WALL_ID	6

#define  ON                1
#define  OFF               0

#define CAVE_LEFT_EYE	1
#define CAVE_RIGHT_EYE	0

#define  CAVEBUTTON1    (CAVEController->button[0])
#define  CAVEBUTTON2    (CAVEController->button[1])
#define  CAVEBUTTON3    (CAVEController->button[2])

#define CAVE_JOYSTICK_X	(CAVEController->valuator[0])
#define CAVE_JOYSTICK_Y	(CAVEController->valuator[1])

#define CAVE_MAX_BUTTONS	8
#define CAVE_MAX_VALUATORS	4

/* Values for CAVEConfig->DisplayMode & CAVEConfig->NonCAVEDisplayMode */
#define CAVE_MONO		0
#define CAVE_STEREO_96		1
#define CAVE_STEREO_120		2
#define CAVE_SETMON		3
#define CAVE_MONO_96		4
#define CAVE_MONO_60		5
#define CAVE_MONO_72		6

/* Values for CAVEConfig->WandType */
#define CAVE_NO_WAND		0
#define CAVE_MOUSE_WAND		1
#define CAVE_PC_WAND		2

/* Values for CAVEConfig->TrackerDevice */
#define CAVE_NO_TRACKING	0
#define CAVE_POLHEMUS		1
#define CAVE_BIRDS		2
#define CAVE_LOGITECH		3
#define CAVE_MOUSETRACKER	4
#define CAVE_SIMTRACKER		5
#define CAVE_SPACEBALL		6

/* Values for CAVEConfig->Units */
#define CAVE_FEET		1
#define CAVE_METERS		2


typedef enum {
		CAVE_HEAD_FRONT,
		CAVE_HEAD_BACK,
		CAVE_HEAD_LEFT,
		CAVE_HEAD_RIGHT,
		CAVE_HEAD_UP,
		CAVE_HEAD_DOWN,
		CAVE_WAND_FRONT,
		CAVE_WAND_BACK,
		CAVE_WAND_LEFT,
		CAVE_WAND_RIGHT,
		CAVE_WAND_UP,
		CAVE_WAND_DOWN
		} CAVEID;

typedef struct {
	int xoff,yoff;
	int xdim,ydim;
	} CAVE_WINDOW_ST;

typedef struct {
	float dx,dy,dz;
	} CAVE_OFFSET_ST;

typedef struct {
	int		ActiveWalls;
	int		DisplayWall[CAVE_MAX_WALLS];
	int		WallPipe[CAVE_MAX_WALL_ID+1];
	CAVE_WINDOW_ST	WallGeom[CAVE_MAX_WALL_ID+1];
	int		UseTracker;
	int		TrackerDevice;  /* CAVE_BIRDS / CAVE_POLHEMUS / etc. */
	char		TrackerSerialPort[2][15];
	int		TrackerBaudRate;
	CAVE_OFFSET_ST	TransmitterOffset;
	CAVE_OFFSET_ST	WandSensorOffset;
	float		WandSensorElevation;
	CAVE_OFFSET_ST	HeadSensorOffset;
	float		InterocularDistance;
	float		HalfInterocularDistance;
	int		UseCalibration;
	char		CalibrationFile[256];
	char		TrackerLogfile[256];
	float		CAVEWidth;
	float		CAVEHeight;
	float		OrigX,OrigY,OrigZ;
	int		Units;
	char		AudioServer[256];
	int		DisplayMode;
	char		DisplayModeSetmon[256];
	int		NonCAVEDisplayMode;
	char		NonCAVEDisplayModeSetmon[256];
	int		WandType;
	int		CPULock;
	int		TimeFrames;
	int		HideCursor;
	}  CAVE_CONFIG_ST;

typedef	struct {
	unsigned int	HostMaster;
	unsigned int	Quit;
	} CAVE_SYNC_ST;

typedef struct {
	float 			x, y, z;
	float 			azi, elev, roll;
	} CAVE_SENSOR_ST;

typedef struct {
	int	num_buttons,num_valuators;
	int	button[CAVE_MAX_BUTTONS];
	float	valuator[CAVE_MAX_VALUATORS];
	} CAVE_CONTROLLER_ST;

#define CAVE_LOG_MAGIC  51902
typedef struct {
		unsigned long magic;
		char cave_version[80];
		char start_time[32];
		int num_sensors, num_buttons, num_valuators;
		char pad[128];  /* Pad to 256 bytes */
		} CAVE_LOG_HEADER_ST;


void	CAVEInit(void);
void	CAVEExit(void);
void	CAVEDisplay(void (*)(), int, ...);
void	CAVEInitApplication( void (*their_draw)(), int argc, ...);
void	*CAVEUserSharedMemory(int);
void	CAVEFrameFunction(void (*)(), int, ...);
void	CAVEStopApplication(void (*)(), int, ...);
void	CAVEGetEyePosition(int eye,float *x,float *y,float *z);
void	CAVEConfigure(int *argc,char **argv,char **appdefaults);
int	CAVEMasterDisplay(void);

int	CAVEButtonChange(int num);
void	CAVEGetVector(CAVEID vecname,float *vec);
float	CAVEGetTime(void);
void	CAVEWallTransform(void);

void	CAVENavigationInit(float, float);
void	CAVESetNavigationParm(float, float);
void	CAVENavigate();
void	CAVECalculateNavigation();


extern	volatile float     	*CAVEFramesPerSecond;
extern	volatile float		*CAVETime;
extern	int                 	CAVEWall;
extern	int			CAVEPipe;
extern	volatile CAVE_CONFIG_ST	*CAVEConfig;
extern	volatile CAVE_SYNC_ST  	*CAVESync;     
extern	volatile CAVE_SENSOR_ST	*CAVESensor1, *CAVESensor2;
extern  float    		CAVENear, CAVEFar;
extern	int			CAVEEye;
extern	char 			*CAVEVersion;
extern	volatile CAVE_CONTROLLER_ST	*CAVEController;


/*************************************************************************/
/**                  SENSOR RELATED MACROS                              **/
/**                                                                     **/
/**    CAVEGetHead(xhead, yhead, zhead)                                 **/
/**    CAVEGetWand(xwand, ywand, zwand)                                 **/
/**    CAVEGetHeadOrientation(azimuth, elevation, wroll)                **/
/**    CAVEGetWandOrientation(azimuth, elevation, wroll)                **/
/**    CAVEWandOrientation	                                        **/
/**    CAVEHeadOrientation	                                        **/
/**    CAVEGetHeadUp(hx, hy, hz)                                        **/
/**    CAVEGetHeadFront(hx, hy, hz)                                     **/
/**    CAVEGetHeadLeft(hx, hy, hz)                                      **/
/*************************************************************************/

#define  CAVEGetHead(xhead, yhead, zhead) \
   {                                   \
   xhead = CAVESensor1 -> x;              \
   yhead = CAVESensor1 -> y;              \
   zhead = CAVESensor1 -> z;               \
   }
#define  CAVEGetWand(xw,  yw,  zw) \
   { \
    xw = CAVESensor2 -> x; \
    yw = CAVESensor2 -> y; \
    zw = CAVESensor2 -> z; \
   }
    
#define  CAVEGetHeadOrientation(azimuth, elevation, wroll) \
   {                                   \
   azimuth = CAVESensor1 -> azi;              \
   elevation = CAVESensor1 -> elev;              \
   wroll = CAVESensor1 -> roll;               \
   }

#define  CAVEGetWandOrientation(azimuth, elevation, wroll) \
   { \
   azimuth = CAVESensor2 -> azi;              \
   elevation = CAVESensor2 -> elev;              \
   wroll = CAVESensor2 -> roll; \
   }


#define  CAVEWandOrientation  \
   { \
   translate( CAVESensor2 -> x,  CAVESensor2 -> y, CAVESensor2 -> z); \
   rot(CAVESensor2->azi ,'y'); \
   rot(CAVESensor2->elev, 'x'); \
   rot(CAVESensor2->roll, 'z'); \
   translate( -CAVESensor2 -> x,  -CAVESensor2 -> y, -CAVESensor2 -> z); \
   }
   
#define  CAVEHeadOrientation  \
   { \
   translate( CAVESensor1 -> x,  CAVESensor1 -> y, CAVESensor1 -> z); \
   rot(CAVESensor1->azi ,'y'); \
   rot(CAVESensor1->elev, 'x'); \
   rot(CAVESensor1->roll, 'z'); \
   translate( -CAVESensor1 -> x, -CAVESensor1 -> y, -CAVESensor1 -> z); \
   }

#define DTOR(r) ((r)*M_PI/180.0)


/*************************************/
/* Wand unit direction vector macros */
/*************************************/

#define CAVEGetWandFront(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor2->elev)) * sinf(DTOR(CAVESensor2->azi));	\
 hy = sinf(DTOR(CAVESensor2->elev));					\
 hz = -cosf(DTOR(CAVESensor2->elev)) * cosf(DTOR(CAVESensor2->azi));	\
}

#define CAVEGetWandUp(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->roll)) +	\
	sinf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	cosf(DTOR(CAVESensor2->roll));					\
 hy = cosf(DTOR(CAVESensor2->roll)) * cosf(DTOR(CAVESensor2->elev));	\
 hz = sinf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->roll)) +	\
	cosf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	cosf(DTOR(CAVESensor2->roll));					\
}

#define CAVEGetWandLeft(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor2->azi)) * cosf(DTOR(CAVESensor2->roll)) -	\
	sinf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	sinf(DTOR(CAVESensor2->roll));					\
 hy = -cosf(DTOR(CAVESensor2->elev)) * sinf(DTOR(CAVESensor2->roll));	\
 hz = sinf(DTOR(CAVESensor2->azi)) * cosf(DTOR(CAVESensor2->roll)) -	\
	cosf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	sinf(DTOR(CAVESensor2->roll));					\
}

/*************************************/
/* Head unit direction vector macros */
/*************************************/

#define CAVEGetHeadFront(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor1->elev)) * sinf(DTOR(CAVESensor1->azi));	\
 hy = sinf(DTOR(CAVESensor1->elev));					\
 hz = -cosf(DTOR(CAVESensor1->elev)) * cosf(DTOR(CAVESensor1->azi));	\
}

#define CAVEGetHeadUp(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->roll)) +	\
	sinf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	cosf(DTOR(CAVESensor1->roll));					\
 hy = cosf(DTOR(CAVESensor1->roll)) * cosf(DTOR(CAVESensor1->elev));	\
 hz = sinf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->roll)) +	\
	cosf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	cosf(DTOR(CAVESensor1->roll));					\
}

#define CAVEGetHeadLeft(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor1->azi)) * cosf(DTOR(CAVESensor1->roll)) -	\
	sinf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	sinf(DTOR(CAVESensor1->roll));					\
 hy = -cosf(DTOR(CAVESensor1->elev)) * sinf(DTOR(CAVESensor1->roll));	\
 hz = sinf(DTOR(CAVESensor1->azi)) * cosf(DTOR(CAVESensor1->roll)) -	\
	cosf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	sinf(DTOR(CAVESensor1->roll));					\
}

#ifdef __cplusplus
}
#endif

#endif
