/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * lattice_weights.c - compute weights for lattice cutoff computation
 *
 * Weights are based on softened pairwise potential described in:
 *
 *   R.D. Skeel, I. Tezcan, D.J. Hardy, "Multiple Grid Methods for
 *   Classical Molecular Dynamics," J. Comput. Chem. 23:673-684, 2002.
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>


/*
 * C2 Taylor softening of 1/r
 */
static float g(float a, float r) {
  if (r < a) {
    float r_a = r/a;
    return (1/a)*(1.875f - 1.25f*r_a*r_a + 0.375f*r_a*r_a*r_a*r_a);
  }
  else {
    return 1/r;
  }
}


/*
 * allocate lattice cube of weights
 *
 * nr - returns radius of cube, i.e. dimension of cube is 2*nr + 1
 * a - cutoff distance, should typically be between 8 and 12 Angstroms
 * h - grid spacing, should typically be between 2 and 3 Angstroms
 */
float *alloc_wt(int *nr, float a, float h)
{
  float *wt, *w;
  int dim, n, i, j, k;
  float r, r2;

  /* compute radius of weight cube */
  n = (int) ceil(2*a / h) - 1;

  /* allocate weight cube */
  dim = 2*n + 1;
  wt = (float *) calloc(dim*dim*dim, sizeof(float));
  if (NULL==wt) {
    fprintf(stderr, "calloc() unable to allocate storage for weights\n");
    abort();
  }

  /* point to center element of cube */
  w = &wt[n*dim*dim + n*dim + n];

  /*
   * fill weights according to softened lattice potential:
   *   g_{a}(r) - g_{2a}(r)
   */
  for (k = -n;  k <= n;  k++) {
    for (j = -n;  j <= n;  j++) {
      for (i = -n;  i <= n;  i++) {
        r2 = (i*i + j*j + k*k) * h*h;
        r = sqrtf(r2);
        w[k*dim*dim + j*dim + i] = g(a, r) - g(2*a, r);
      }
    }
  }

  *nr = n;
  return wt;
}


/*
 * free weights when finished with them
 */
void free_wt(float *wt)
{
  free(wt);
}
