/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * lattice_cutoff.c - lattice cutoff computation for multilevel summation
 *
 * All lattices are stored in column-major order, i.e. loop with:
 *   for (k = 0;  k < nz;  k++) {
 *     for (j = 0;  j < ny;  j++) {
 *       for (i = 0;  i < nx;  i++) {
 *         e[k*nj*ni + j*ni + i] = ...
 *       }
 *     }
 *   }
 * to access the array elements in consecutive order.
 */

#include <stdlib.h>
#include <stdio.h>


/*
 * e  - 3D lattice of potentials to be computed
 * q  - 3D lattice of charges
 * nx, ny, nz  - dimensions of e and q lattices
 *
 * wt  - lattice of precomputed weights
 * nr  - "radius" of wt, i.e. wt is cube of dimension 2*nr + 1
 *
 * returns potential lattice computed as weighted sums of surrounding charges
 * (for nonperiodic boundaries, weighted sum stops at lattice edges)
 */
void nonperiodic_lattice_cutoff(float *e, const float *q,
    int nx, int ny, int nz, const float *wt, int nr)
{
  /*
   * point to center of wt cube and reference x, y, z dimensions
   * using range:  -nr..nr
   */
  int wdim = 2*nr + 1;
  const float *w = &wt[nr*wdim*wdim + nr*wdim + nr];

  int i, ia, ib, iw;
  int j, ja, jb, jw;
  int k, ka, kb, kw;

  float esum;

#ifdef OPCOUNT
  unsigned long long int opcount = 0;
#endif

  /* loop through e lattice */
  for (k = 0;  k < nz;  k++) {
    for (j = 0;  j < ny;  j++) {
      for (i = 0;  i < nx;  i++) {

        /* trim w centered at i,j,k to fit within lattice */
        ia = (i - nr < 0 ? -i : -nr);
        ib = (i + nr >= nx ? nx-1-i : nr);
        ja = (j - nr < 0 ? -j : -nr);
        jb = (j + nr >= ny ? ny-1-j : nr);
        ka = (k - nr < 0 ? -k : -nr);
        kb = (k + nr >= nz ? nz-1-k : nr);

#ifdef OPCOUNT
        opcount += (ib-ia+1)*(jb-ja+1)*(kb-ka+1);
#endif

        /* loop over trimmed w cube, sum weighted q contributions */
        esum = 0;
        for (kw = ka;  kw <= kb;  kw++) {
          for (jw = ja;  jw <= jb;  jw++) {
            for (iw = ia;  iw <= ib;  iw++) {
              esum += w[kw*wdim*wdim + jw*wdim + iw] *
                q[(k+kw)*ny*nx + (j+jw)*nx + (i+iw)];
            }
          }
        }
        e[k*ny*nx + j*nx + i] = esum;
      }
    }
  }
#ifdef OPCOUNT
  printf("OPERATION COUNT:  %llu multiply-adds\n", opcount);
#endif

}


/*
 * e  - 3D lattice of potentials to be computed
 * q  - 3D lattice of charges
 * nx, ny, nz  - dimensions of e and q lattices
 *
 * wt  - lattice of precomputed weights
 * nr  - "radius" of wt, i.e. wt is cube of dimension 2*nr + 1
 *
 * returns potential lattice computed as weighted sums of surrounding charges
 * (for periodic boundaries, weighted sum wraps around lattice edges)
 */
void periodic_lattice_cutoff(float *e, const float *q,
    int nx, int ny, int nz, const float *wt, int nr)
{
  /*
   * point to center of wt cube and reference x, y, z dimensions
   * using range:  -nr..nr
   */
  int wdim = 2*nr + 1;
  const float *w = &wt[nr*wdim*wdim + nr*wdim + nr];

  int i, iw, j, jw, k, kw;
  float esum;

#ifdef OPCOUNT
  unsigned long long int opcount = 0;
#endif

  /* loop through e lattice */
  for (k = 0;  k < nz;  k++) {
    for (j = 0;  j < ny;  j++) {
      for (i = 0;  i < nx;  i++) {

#ifdef OPCOUNT
        opcount += (2*nr+1)*(2*nr+1)*(2*nr+1);
#endif

        /* loop over w cube, sum weighted q contributions */
        esum = 0;
        for (kw = -nr;  kw <= nr;  kw++) {
          for (jw = -nr;  jw <= nr;  jw++) {
            for (iw = -nr;  iw <= nr;  iw++) {
              esum += w[kw*wdim*wdim + jw*wdim + iw] *
                q[((k+kw+nz*nr)%nz)*ny*nx + ((j+jw+ny*nr)%ny)*nx
                + (i+iw+nx*nr)%nx];
              /*
               * add nz*nr, etc., to ensure that the modulo operation
               * gives a non-negative result
               */
            }
          }
        }
        e[k*ny*nx + j*nx + i] = esum;
      }
    }
  }
#ifdef OPCOUNT
  printf("OPERATION COUNT:  %llu multiply-adds\n", opcount);
#endif

}
