/* 
 * tcl interface for the gofr_calc plugin 
 * 
 * Copyright (c) 2004 axel.kohlmeyer@theochem.ruhr-uni-bochum.de
 */

#include <stdio.h>
#include <tcl.h>

#include "gofr.h"

/* this is the actual interface to the 'gofr' command.
 * it parses the arguments, calls the calculation subroutine
 * and then passes the result to the interpreter. */
int tcl_gofr(ClientData nodata, Tcl_Interp *interp,
             int objc, Tcl_Obj *const objv[]) {

    Tcl_Obj *resultPtr, **clist_a, **clist_b, **ulist, **rlist;
    Tcl_Obj *getcmd[3], *getlist[3], *resa, *resb;

    int i, count_a, count_b, count_u, count_h, *hlist;
    double delta, rmax;
    coord *alist, *blist, boxby2;
    double a,b,c, alpha,beta,gamma;
    
    /* argument list treatment is a bit tricky. one could make quite a few
     * optional, e.g. we could get the unit cell from the 'molinfo' implicitely,
     * but this will fail for many systems, that do not provide the unit
     * cell information (which are many).
     * 
     * so we require the full five arguments and allow some flexibility
     * with the parsing of the box parameters.
     * 1 parameter:  cubic cell with given length a
     * 3 parameters: orthorhombic cell with given lengths a, b, and c.
     * PBC treatment can be disabled by using a very large cubic box.
     */
    if (objc != 6) {
        Tcl_WrongNumArgs(interp, 1, objv, "?delta? ?rmax? ?coord_a? ?coord_b? ?cell?");
        return TCL_ERROR;
    }
    
    /* parse arguments one by one. */
    if (Tcl_GetDoubleFromObj(interp, objv[1], &delta) != TCL_OK) {
        return TCL_ERROR;
    }
    if (Tcl_GetDoubleFromObj(interp, objv[2], &rmax) != TCL_OK) {
        return TCL_ERROR;
    }

    /* prepare tcl command for retrieval of coordinates from selection */
    getcmd[1] = Tcl_NewStringObj("get", -1);
    getlist[0] = Tcl_NewStringObj("x", -1);
    getlist[1] = Tcl_NewStringObj("y", -1);
    getlist[2] = Tcl_NewStringObj("z", -1);
    getcmd[2] = Tcl_NewListObj(3, getlist);

    /* get the coordinates from the first selection */
    getcmd[0] = objv[3];
    Tcl_EvalObjv(interp, 3, getcmd, 0);
    resa = Tcl_GetObjResult(interp);
    Tcl_IncrRefCount(resa);
    if (Tcl_ListObjGetElements(interp, resa, &count_a, &clist_a) != TCL_OK) {
        return TCL_ERROR;
    }
    
    /* get the coordinates from the second selection */
    getcmd[0] = objv[4];
    Tcl_EvalObjv(interp, 3, getcmd, 0);
    resb = Tcl_GetObjResult(interp);
    Tcl_IncrRefCount(resb);
    if (Tcl_ListObjGetElements(interp, resb, &count_b, &clist_b) != TCL_OK) {
        return TCL_ERROR;
    }

    /* unit cell info */
    if (Tcl_ListObjGetElements(interp, objv[5], &count_u, &ulist) != TCL_OK) {
        return TCL_ERROR;
    }

    /* check arguments */
    /* coordinate lists */
    if (count_a < 1) {
        printf("need at least one element in coordinate group A\n");
        return TCL_ERROR;
    }
    if (count_b < 1) {
        printf("need at least one element in coordinate group B\n");
        return TCL_ERROR;
    }

    /* unit cell. first set defaults. */
    a=b=c=9999999.0;
    alpha=beta=gamma=90.0;
    if ((count_u != 1) && (count_u != 3) && (count_u != 6)) {
        printf("invalid number of unit cell parameters: need either 1, 3, or 6, but got %d\n", count_u);
        return TCL_ERROR;
    }
    if (Tcl_GetDoubleFromObj(interp, ulist[0], &a) != TCL_OK) {
        return TCL_ERROR;
    }
    b=c=a; /* in case of cubic box only a may be given. set b and c.*/

    if (count_u == 3) {
        if (Tcl_GetDoubleFromObj(interp, ulist[1], &b) != TCL_OK) {
            return TCL_ERROR;
        }
        if (Tcl_GetDoubleFromObj(interp, ulist[2], &c) != TCL_OK) {
            return TCL_ERROR;
        }
    }
    
    if (count_u == 6) {
        if (Tcl_GetDoubleFromObj(interp, ulist[3], &alpha) != TCL_OK) {
            return TCL_ERROR;
        }
        if (Tcl_GetDoubleFromObj(interp, ulist[4], &beta) != TCL_OK) {
            return TCL_ERROR;
        }
        if (Tcl_GetDoubleFromObj(interp, ulist[5], &gamma) != TCL_OK) {
            return TCL_ERROR;
        }
    }
    
    if ((alpha != 90.0) || (beta != 90.0) || (gamma != 90.0)) {
        printf("only orthorhombic cells are supported (for now).\n");
        return TCL_ERROR;
    }
    boxby2.x = 0.5*a;
    boxby2.y = 0.5*b;
    boxby2.z = 0.5*c;
    
    
    /* FIXME: we could limit rmax to half the box or something similar here.*/
    count_h = (int)(rmax / delta + 1.0);
    hlist = (int *)Tcl_Alloc(count_h * sizeof(int));
    for(i=0; i<count_h; ++i) {hlist[i]=0;}
    
    /* read and convert coordinates */
    alist = (coord *)Tcl_Alloc(count_a*sizeof(coord));
    blist = (coord *)Tcl_Alloc(count_b*sizeof(coord));
    if (!alist || !blist) {
        printf("memory allocation problem.\n");
        return TCL_ERROR;
    }

    for (i=0; i<count_a; ++i) {
        int num_coords;
        Tcl_Obj **clist;

        if (Tcl_ListObjGetElements(interp, clist_a[i], &num_coords, &clist) != TCL_OK) {
            return TCL_ERROR;
        }
        if (num_coords != 3) {
            printf("need 3 values for coordinate %d in group A, got %d:\n", i, num_coords);
            return TCL_ERROR;
        }
        Tcl_GetDoubleFromObj(interp, clist[0], &(alist[i].x));
        Tcl_GetDoubleFromObj(interp, clist[1], &(alist[i].y));
        Tcl_GetDoubleFromObj(interp, clist[2], &(alist[i].z));
    }

    for (i=0; i<count_b; ++i) {
        int num_coords;
        Tcl_Obj **clist;
        
        if (Tcl_ListObjGetElements(interp, clist_b[i], &num_coords, &clist) != TCL_OK) {
            return TCL_ERROR;
        }
        if (num_coords != 3) {
            printf("need 3 values for coordinate %d in group B, got %d:\n", i, num_coords);
            return TCL_ERROR;
        }
        Tcl_GetDoubleFromObj(interp, clist[0], &(blist[i].x));
        Tcl_GetDoubleFromObj(interp, clist[1], &(blist[i].y));
        Tcl_GetDoubleFromObj(interp, clist[2], &(blist[i].z));
    }
    /* free references on the selection command results.*/
    Tcl_DecrRefCount(resa);
    Tcl_DecrRefCount(resb);
    
    /* do the gofr calculation */
    calc_gofr(count_a, alist, count_b, blist, count_h, hlist, delta, boxby2);

    rlist = (Tcl_Obj **) Tcl_Alloc(count_h * sizeof(Tcl_Obj *));
    for (i=0; i<count_h; ++i) {
        rlist[i] = Tcl_NewIntObj(hlist[i]);
    }
    resultPtr = Tcl_NewListObj(count_h, rlist);
    Tcl_SetObjResult(interp, resultPtr);

    /* free intermediate storage */
    Tcl_Free((char *)alist);
    Tcl_Free((char *)blist);
    Tcl_Free((char *)hlist);
    Tcl_Free((char *)rlist);

    return TCL_OK;
}

/* register the plugin with the tcl interpreters */
#if defined(GOFRCLDLL_EXPORTS) && defined(_WIN32)
#  undef TCL_STORAGE_CLASS
#  define TCL_STORAGE_CLASS DLLEXPORT

#define WIN32_LEAN_AND_MEAN // Exclude rarely-used stuff from Window s headers
#include <windows.h>

BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved )
{
    return TRUE;
}

EXTERN int Gofr_Init(Tcl_Interp *interp)

#else

int Gofr_Init(Tcl_Interp *interp)   

#endif
{
  Tcl_CreateObjCommand(interp,"gofr",tcl_gofr,
        (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);

  Tcl_PkgProvide(interp, "gofr_calc", "1.0.0");

  return TCL_OK;
}
