# Ensure TclOO is available
package require TclOO

# Class representing a Scene, which contains a list of ActionWrapper arrays
oo::class create scene {
    variable action_wrappers
    variable scene_name
    variable scene_params
    variable time
    variable time_offset 0

    constructor { new_scene_name new_scene_params } {
        array set action_wrappers {}
        set scene_name $new_scene_name
        array set scene_params $new_scene_params
        set time_offset 0
    }

    # Add an action to a specific action_wrapper
    method add_action {wrapper_index name params} {
        if {![info exists action_wrappers($wrapper_index)]} {
            set action_wrappers($wrapper_index) {}
        }
        lappend action_wrappers($wrapper_index) [list $name $params]
    }

    # Edit an action in a specific action_wrapper by index
    method edit_action {wrapper_index action_index new_name new_params} {
        set action [lindex $action_wrappers($wrapper_index) $action_index]
        set action [list $new_name $new_params]
        lset action_wrappers($wrapper_index) $action_index $action
    }

    method remove_action {wrapper_index} {
        # Ensure the action wrapper exists
        if {![info exists action_wrappers($wrapper_index)]} {
            error "Action wrapper at index $wrapper_index does not exist"
        }

        # Remove the wrapper from the array by unsetting the key
        unset action_wrappers($wrapper_index)

        set new_wrappers_list {}
        set new_index 0
        foreach old_index [lsort -integer [array names action_wrappers]] {
            lappend new_wrappers_list $new_index $action_wrappers($old_index)
            incr new_index
        }

        # Replace action_wrappers with the renumbered ones
        array unset action_wrappers
        array set action_wrappers $new_wrappers_list
    }

    method move_action_up {wrapper_index} {
        # Ensure the action wrapper exists
        if {![info exists action_wrappers($wrapper_index)]} {
            error "Action wrapper at index $wrapper_index does not exist"
        }

        # Ensure it's not the first item (index 0), which can't be moved up
        if {$wrapper_index == 0} {
            error "Action wrapper at index 0 cannot be moved up"
        }

        # Get the previous index
        set prev_index [expr {$wrapper_index - 1}]

        # Ensure the previous action wrapper exists
        if {![info exists action_wrappers($prev_index)]} {
            error "Action wrapper at index $prev_index does not exist"
        }

        # Swap the action wrappers at wrapper_index and prev_index
        set temp $action_wrappers($wrapper_index)
        set action_wrappers($wrapper_index) $action_wrappers($prev_index)
        set action_wrappers($prev_index) $temp
    }

    method move_action_down {wrapper_index} {
        # Ensure the action wrapper exists
        if {![info exists action_wrappers($wrapper_index)]} {
            error "Action wrapper at index $wrapper_index does not exist"
        }

        # Ensure it's not the first item (index 0), which can't be moved up
        if {$wrapper_index == [expr [llength [my get_all_action_times]] - 1]} {
            error "Action wrapper at last index cannot be moved down"
        }

        # Get the next index
        set next_index [expr {$wrapper_index + 1}]

        # Ensure the previous action wrapper exists
        if {![info exists action_wrappers($next_index)]} {
            error "Action wrapper at index $next_index does not exist"
        }

        # Swap the action wrappers at wrapper_index and next_index
        set temp $action_wrappers($wrapper_index)
        set action_wrappers($wrapper_index) $action_wrappers($next_index)
        set action_wrappers($next_index) $temp
    }

    method merge_actions {wrapper_index_1 wrapper_index_2} {
        foreach action $action_wrappers($wrapper_index_2) {
            lappend action_wrappers($wrapper_index_1) $action
        }
        my remove_action $wrapper_index_2
    }

    method unmerge_action {wrapper_index} {
        set new_wrappers_list {}
        set new_index 0
        foreach old_index [lsort -integer [array names action_wrappers]] {
            if {$old_index == $wrapper_index} {
                foreach action $action_wrappers($wrapper_index) {
                    lassign $action name params
                    set newaction [list $name $params]
                    lappend new_wrappers_list $new_index [list $newaction]
                    incr new_index
                }
            } else {
                lappend new_wrappers_list $new_index $action_wrappers($old_index)
                incr new_index
            }
        }
        array unset action_wrappers
        array set action_wrappers $new_wrappers_list
    }

    method get_all_action_names {} {
        set allnames {}
        foreach wrapper_index [lsort -integer [array names action_wrappers]] {
            set namelist {}
            foreach action $action_wrappers($wrapper_index) {
                lassign $action name params
                lappend namelist $name
            }
            lappend allnames $namelist
        }
        return $allnames
    }

    method get_all_action_params {} {
        set allparams {}
        foreach wrapper_index [lsort -integer [array names action_wrappers]] {
            set paramlist {}
            foreach action $action_wrappers($wrapper_index) {
                lassign $action name params
                set params_without_t [dict filter $params script {key value} {expr {$key ne "t"}}]
                lappend paramlist $params_without_t
            }
            lappend allparams $paramlist
        }
        return $allparams 
    }

    method get_all_action_times {} {
        set alltimes {}
        foreach wrapper_index [lsort -integer [array names action_wrappers]] {
            set time 0
            foreach action $action_wrappers($wrapper_index) {
                lassign $action name params
                if {[dict exists $params t]} {
                    if {[dict get $params t] > 0} {
                    set time [dict get $params t]
                    }
                }
            }
            lappend alltimes $time
        }
        return $alltimes
    }
    
    # Read actions from a text string (assumes a specific format)
    method read_actions_from_text {text} {
        foreach line [split $text "\n"] {
            if {[regexp {^(.*) -> (.*)$} $line -> wrapper_index action_desc]} {
                set name [lindex [split $action_desc ","] 0]
                set params_dict {}
                foreach param [lrange [split $action_desc ","] 1 end] {
                    lassign [split $param "="] key value
                    dict set params_dict $key $value
                }
                my add_action $wrapper_index $name $params_dict
            }
        }
    }

    # Write actions to a string (in the format 'WrapperIndex -> ActionName, key1=value1, key2=value2')
    method write_actions_to_string {} {
        set result "# [my get_scene_name]\n"
        foreach wrapper_index [lsort -integer [array names action_wrappers]] {
            set wraplen [llength $action_wrappers($wrapper_index)]
            if {$wraplen > 1} {
                lassign {"\{" ";" "\}"} copen cmid cend
            } else {
                lassign {"" "" ""} copen cmid cend
            }
            set params_str $copen
            set counter 0
            foreach action $action_wrappers($wrapper_index) {
                incr counter
                lassign $action name params
                # TODO do unpacking here
                set valtypes [dict get [dict get $::molywood::actions $name] types]
                set unpack [dict get [dict get $::molywood::actions $name] unpack]
                if {$unpack == false} {
                    append params_str "$name "
                    foreach {key value} [dict get $params] {
                        if {$value != ""} {
                            if {[regexp {\s} $value]} {
                                set value [string trim $value "\"\'"]
                                set value \"$value\"
                            }
                            if {[dict get $valtypes $key] == "boolean"} {
                                if {$value == 0} {
                                    set value f
                                } elseif {$value == 1} {
                                    set value t
                                }
                            }
                            append params_str "$key=$value "
                        }
                    }
                } else {
                    foreach {up_action dum} [dict get [dict get $::molywood::actions $name] unpack_actions] {
                        append params_str "$up_action "
                        set valtypes [dict get [dict get $::molywood::actions $up_action] types]
                        foreach {key value} [dict get $params] {
                            set testval [dict get [dict get [dict get $::molywood::actions $name] params_assign] $key]
                            if {[string match "${up_action}*" $testval]} {
                                if {$value != ""} {
                                    if {[regexp {^([^-]+)-(.*)$} $testval -> dum2 key]} {
                                        ;;
                                    }
                                    if {[regexp {\s} $value]} {
                                        set value [string trim $value "\"\'"]
                                        set value \"$value\"
                                    }
                                    if {[dict get $valtypes $key] == "boolean"} {
                                        if {$value == 0} {
                                            set value f
                                        } elseif {$value == 1} {
                                            set value t
                                        }
                                    }
                                    append params_str "$key=$value "
                                }
                            }
                        }
                        append params_str "$cmid\n"
                    }
                }

                if {$counter < $wraplen} {
                    append params_str "$cmid\n"
                }
            }
            append result "$params_str $cend \n"
        }
        return $result
    }

    # Method to print the current state of the scene
    method print_scene {} {
        puts [my write_actions_to_string]
    }

    method calc_total_time {} {
        set total_time 0
        foreach acttime [my get_all_action_times] {
            if {$acttime == ""} {
                set realtime 0
            } else {
                set realtime $acttime
            }
            set total_time [expr $total_time + $realtime]
        }
        return $total_time
    }
    
    method get_scene_name {} {
        return $scene_name
    }

    method get_scene_attributes {} {
        set action_count [array size action_wrappers]
        return "$action_count action(s), $scene_params(xres) x $scene_params(yres) px, from $scene_params(srcs) $scene_params(srcf)"
    }

    method set_scene_name {new_scene_name} {
        set scene_name $new_scene_name
    }

    method set_scene_parameters {new_scene_params} {
        array set scene_params $new_scene_params
    }

    method set_scene_parameter {key new_scene_param} {
        array set scene_params [list $key $new_scene_param]
    }

    method get_scene_parameters {} {
        set all_params {}
        foreach scene_param [array names scene_params] {
            dict set all_params $scene_param $scene_params($scene_param)
        }
        return $all_params
    }

    method get_formatted_scene_parameters {} {
        set params_dict {}
        foreach param [array names scene_params] {
            switch $param {
                srcs {
                    dict set params_dict $scene_params(srcs) $scene_params(srcf)
                }
                xres {
                    dict set params_dict "resolution" "$scene_params(xres),$scene_params(yres)"
                }
                rnum {
                    dict set params_dict "position" "$scene_params(rnum),$scene_params(cnum)"
                }
                aftr {
                    if {$scene_params(aftr) != ""} {
                        dict set params_dict "after" "$scene_params(aftr)"
                    }
                }
                aocc {
                    dict set params_dict "ambient_occlusion" "$scene_params(aocc)"
                }
                mxts {
                    dict set params_dict "max_transparent_surfaces" "$scene_params(mxts)"
                }
            }
        }
        return $params_dict
    }

    method get_action_wrapper {index} {
        return $action_wrappers($index)
    }

    method add_action_wrapper {wrapper index} {
        set action_wrappers($index) $wrapper
    }

    method set_time_offset {new_time_offset} {
        set time_offset $new_time_offset
    }

    method get_time_offset {} {
        return $time_offset
    }

    method is_after {} {
        lassign [::molywood::resolve_deps] afters withins
        foreach {key val} $afters {
            if {$key == $scene_name} {
                return 1
            }
        }
        return 0
    }

    method is_within {} {
        lassign [::molywood::resolve_deps] afters withins
        foreach {key val} $withins {
            if {$key == $scene_name} {
                return 1
            }
        }
        return 0
    }

    method is_dep {} {
        return [expr [my is_after] + [my is_within]]
    }

    method depends_on {} {
        lassign [::molywood::resolve_deps] afters withins
        foreach {key val} [dict merge $withins $afters] {
            if {$key == $scene_name} {
                return $val
            }
        }
        return $scene_name
    }

    method sanitize_values {} {
        set allnames [my get_all_action_names]
        set allparams [my get_all_action_params]
        for {set i 0} {$i < [llength $allnames]} {incr i} {
            set currnames [lindex $allnames $i]
            set currparamset [lindex $allparams $i]
            for {set j 0} {$j < [llength $currnames]} {incr j} {
                set currname [lindex $currnames $j]
                # puts "checking $currname"
                set currparams [lindex $currparamset $j]
                set typesdict [dict get [dict get $::molywood::actions $currname] types]
                foreach {parm value} [dict get $currparams] {
                    switch [dict get $typesdict $parm] {
                        float {
                            if {![string is double $value]} {
                                error "parameter $parm in action $currname should be a float, but is $value"
                            }
                        }
                        int {
                            if {![string is entier $value]} {
                                error "parameter $parm in action $currname should be an integer, but is $value"
                            }
                        }
                        boolean {
                            set valid_booleans {true false t f yes no y n 0 1}
                            if {[expr {[lsearch -exact $valid_booleans [string tolower $value]] == -1}]} {
                                error "parameter $parm in action $currname should be a boolean (true/false), but is $value"
                            }
                        }
                    }
                    # if {[dict exists $forb $parm]} {
                    #     if {$value == [dict get $forb $parm]} {
                    #         error "Value $value is not allowed for parameter $parm in action $currname"
                    #     }
                    # }
                }
            }
        }
    }
}