#
# $Id: qwikmd_logText.tcl,v 1.1.1.1 2018/07/31 19:37:29 jribeiro Exp $
#
#==============================================================================
proc QWIKMD::introText {} {
    set systemTime [clock seconds]
    set OS "Linux"
    if {$::tcl_platform(os) == "Darwin"} {
        set OS "MacOS"
    } elseif {[string first "Windows" $::tcl_platform(os)] != -1} {
        set OS "Windows"
    }
    set text \
"File generated by QwikMD (version [package versions qwikmd]) on [clock format $systemTime -format %H:%M-%Z], [clock format $systemTime -format %D]\n
Machine name: [info hostname] \($OS\).

==============================================================================
QwikMD text log file. In this file one can find the steps taken to prepare,
perform and analyze the MD simulation. The file is divided in 3 major sections:\n
    \"Structure Preparation\" lists the operations performed to prepare 
    the structure for simulation, such as atom deletion and residue renaming;\n
    \"MD Protocols\" lists the MD simulation protocols prepared/performed 
    and their specific parameters like temperature, and simulation time;\n 
    \"MD Analysis\" list the analysis performed to the trajectory generated
    by the execution of the previous MD protocols.                              
==============================================================================



"
    return $text
}

proc QWIKMD::structPrepLog {} {
    set molname [molinfo $QWIKMD::topMol get name]


    set text \
"============================== Structure Preparation ===============================

The structure $molname was loaded "


    if {[file isfile $QWIKMD::inputstrct] == 1} {
        append text "from a local folder:\n    [file dirname $QWIKMD::inputstrct].\n"
    } else {
        append text "directly from PDB website.\n"
    }
append text "The original structure can be found at ${QWIKMD::outPath}/setup/[lindex [file root [file tail [lindex [molinfo $QWIKMD::topMol get filename] 0] ] ] 0 ]_original.pdb\n"

    

    return $text
}

proc QWIKMD::deleteLog {} {

    set text ""
    set length [expr [llength [array names QWIKMD::chains]] /3]
    set do 0
    for {set i 0} {$i < $length} {incr i} {
        if {$QWIKMD::chains($i,0) == 0} {
            lappend chainstext $QWIKMD::chains($i,1)
            set do 1
        }
    }
    if {$do == 1} {
        append text "\nThe atom selection defined by the following pairs chain&type were deleted:\n"
        foreach chain $chainstext {
            append text "    $chain\n"
        }
    }
if {[llength $QWIKMD::delete] > 0} {
append text \
"\nThe following residues were deleted:\n
"
    foreach delResid $QWIKMD::delete {
        set residChain [split $delResid "_"]
        set sel_resname [atomselect $QWIKMD::topMol "resid \"[lindex $residChain 0]\" and chain \"[lindex $residChain end]\" "]
        append text "    The Residue name [lindex [$sel_resname get resname] 0], id [lindex $residChain 0] from chain [lindex $residChain end]\n"
        $sel_resname delete
    }
}

    return $text
}

proc QWIKMD::renameLog {} {
set text \
"\nThe following residues were renamed:\n"
    foreach rename $QWIKMD::renameindex {
        set residChain [split $rename "_"]
        set sel_resname [atomselect $QWIKMD::topMol "resid \"[lindex $residChain 0]\" and chain \"[lindex $residChain end]\" "]
        append text "    The Residue name [lindex [$sel_resname get resname] 0], id [lindex $residChain 0] from chain [lindex $residChain end] renamed as $QWIKMD::dorename($rename)\n"
        $sel_resname delete
    }

    return $text
}

proc QWIKMD::renameReorderAtomLog {} {
    set text ""
    if {[llength $QWIKMD::autorenameLog] > 0} {
        append text \
    "\nThe following elements were automatically renamed:\n"
        #QWIKMD::autorenameLog {$oldResidueName $newResidueName} or {$newResidueName {$oldAtomName $newAtomName}}
        foreach atom $QWIKMD::autorenameLog {
            if {[llength [lindex $atom 1] ] == 1} {
                append text "    All the residues with name [lindex $atom 0] were renamed to [lindex $atom 1]\n"
            } else {
                set atomval [lindex $atom 1]
                #foreach atomval [lindex $atom 1] {
                append text "    The atoms of the residue [lindex $atom 0] named [lindex $atomval 0] were renamed to [lindex $atomval 1]\n" 
                #}
            }
        }
    }

    if {[llength $QWIKMD::atmsRenameLog] > 0} {
        append text \
    "\nThe following atoms were renamed:\n"
        #QWIKMD::atmsRenameLog {{$resid $PDBresname $chain $originalname $newname]}}
        foreach atom $QWIKMD::atmsRenameLog {
            append text "    Atom from Residue ID [lindex $atom 0], residue name [lindex $atom 1], chain [lindex $atom 2], originally name [lindex $atom 3] was renamed as [lindex $atom 4]\n"
        }
    }

    if {[llength $QWIKMD::atmsDeleteLog] > 0} {
        append text \
    "\nThe following atoms were deleted:\n"
        #QWIKMD::atmsRenameLog {{$resid $PDBresname $chain $originalname $newname]}}
        foreach atom $QWIKMD::atmsDeleteLog {
            append text "    Atom from Residue ID [lindex $atom 0], residue name [lindex $atom 1], chain [lindex $atom 2] and named name [lindex $atom 3]\n"
        }
    }

    if {[llength $QWIKMD::atmsReorderLog] > 0} {
        append text \
    "\nThe following residues were reorder:\n"
        #QWIKMD::atmsRenameLog {{$resid $PDBresname $chain $originalResid $newResid]}}
        foreach atom $QWIKMD::atmsReorderLog {
            append text "    The Residue ID [lindex $atom 0], residue name [lindex $atom 1], chain [lindex $atom 2], originally ID [lindex $atom 3] was reorder to the ID [lindex $atom 4]\n"
        }
    }

    return $text
}

proc QWIKMD::membraneLog {} {
    set angsChr "Å"
set text \
"\nThe system was embedded on a membrane environment. \nMembrane details:\n
"
    append text "Lipid Composition: $QWIKMD::advGui(membrane,lipid)\n\n"
    append text "Membrane Dimensions (x,y in $angsChr): $QWIKMD::advGui(membrane,xsize), $QWIKMD::advGui(membrane,ysize) \n\n"
    append text "Membrane Center (x,y,z): [format "%.3f,%.3f,%.3f" $QWIKMD::advGui(membrane,center,x) $QWIKMD::advGui(membrane,center,y) $QWIKMD::advGui(membrane,center,z)]\n\n"
    append text "Rotation degrees over x axis: $QWIKMD::advGui(membrane,rotate,x)\n"
    append text "Rotation degrees over y axis: $QWIKMD::advGui(membrane,rotate,y)\n"
    append text "Rotation degrees over z axis: $QWIKMD::advGui(membrane,rotate,z)\n"

    return $text
}

proc QWIKMD::mutateLog {} {
set text \
"\nThe following residues were mutated:\n"
    foreach mutation $QWIKMD::mutindex {
        set residChain [split $mutation "_"]
        set sel_resname [atomselect $QWIKMD::topMol "resid \"[lindex $residChain 0]\" and chain \"[lindex $residChain end]\" "]
        append text "    The Residue name [lindex [$sel_resname get resname] 0], id [lindex $residChain 0] from chain [lindex $residChain end] was mutated to [lindex $QWIKMD::mutate($mutation) 1]\n"
        $sel_resname delete
    }

    return $text
}

proc QWIKMD::protonateLog {} {
set text \
"\nThe system was placed in a water box :\n"

    foreach prot $QWIKMD::protindex {
        set residChain [split $prot "_"]
        set resname ""
        set searchIndex [lsearch $QWIKMD::mutindex $prot]
        if {$searchIndex == -1} {
            set sel_resname [atomselect $QWIKMD::topMol "resid \"[lindex $residChain 0]\" and chain \"[lindex $residChain end]\" "]
            set resname [lindex [$sel_resname get resname] 0]
            $sel_resname delete
        } else {
            set resname [lindex $QWIKMD::mutate([lindex $QWIKMD::mutindex $searchIndex]) 1]
        }
        append text "    The protonation state of the residue $resname, id [lindex $residChain 0], chain [lindex $residChain end] was assigned to [lindex $QWIKMD::protonate($prot) 1]\n"
    }

    return $text
}

proc QWIKMD::patchLog {} {
set text \
"\nThe following modifications (patches) were applied to the structure:\n"

    foreach patch $QWIKMD::patchestr {
        set patchname [lindex $patch 0]
        set chain1 [lindex $patch 1]
        set resid1 [lindex $patch 2]
        set chain2 ""
        set resid2 ""
        if {[llength $patch] > 3} {
            set chain2 [lindex $patch 3]
            set resid2 [lindex $patch 4]
        }
        
        append text "    $patchname on residue $resid1, chain $chain1 "
        if {[llength $patch] > 3} {
            append text "and the residue $resid2, chain $chain2."
        }
        append text "\n"
    }

    return $text
}

proc QWIKMD::solvateLog {membrane dimList} {
    set angsChr "Å"
        if $membrane {
        set text "\nThe system on a membrane environment was solvated in two steps and the distance between\n the maximum and minimum z coordinates and the water box edges in the z-axis was set to $QWIKMD::advGui(solvent,boxbuffer,$QWIKMD::run) $angsChr \n\n"
        set i 0
        foreach dim $dimList {
            if {$i == 0} {
                append text "    First the system was solvate below the membrane plane with:\n"
            } else {
                append text "    And second the system was solvate above the membrane plane with:\n"
            }
            
            set lDim $dim
            set xmin [lindex [lindex $lDim 0] 0]
            set xmax [lindex [lindex $lDim 1] 0]
            set ymin [lindex [lindex $lDim 0] 1]
            set ymax [lindex [lindex $lDim 1] 1]
            set zmin [lindex [lindex $lDim 0] 2]
            set zmax [lindex [lindex $lDim 1] 2]
            set xDim [format %.2f [expr abs($xmax -  $xmin)]]
            set yDim [format %.2f [expr abs($ymax -  $ymin)]]
            set zDim [format %.2f [expr abs($zmax -  $zmin)]]

            append text "        a water box of the dimensions ${xDim},${yDim},${zDim} (x,y,z in $angsChr )\n"
            append text "        placed at \([format %.2f $xmin],[format %.2f $ymin],[format %.2f $zmin]\)\
            and \([format %.2f $xmax],[format %.2f $ymax],[format %.2f $zmax]\) as minimum and maximum coordinates.\n"
            set watersel [atomselect top "segname \"W.*\""]
            
            set residuenumber [llength [lsort -unique [$watersel get residue]]]
            append text "\n     A total of $residuenumber water molecules TIP3\[[QWIKMD::addReference TIP3P]\] model were placed.\n\n"
            $watersel delete
            incr i
        }
    
    } else {
        set tabid [$QWIKMD::topGui.nbinput index current]
        set buffer 7.5
        if {$tabid == 1} {
            set buffer $QWIKMD::advGui(solvent,boxbuffer,$QWIKMD::run)
        }
        if {$QWIKMD::run == "SMD"} {
            set text "\nThe system was solvated a using water box with a $buffer ${angsChr} buffer around the system and then extended $QWIKMD::basicGui(plength) ${angsChr} in the SMD \n \
            pulling axis"
        } else {
            set text "\nThe system was solvated a using"
            if {$QWIKMD::advGui(solvent,minimalbox,$QWIKMD::run) == 0} {
                append text " cubic"
            }
            append text " water box with a $buffer ${angsChr} buffer"
            if {$QWIKMD::advGui(solvent,minimalbox,$QWIKMD::run) == 0} {
             append text " additionally to the system longest dimension"
            }
        }
        
        set lDim $dimList
        set xmin [lindex [lindex $lDim 0] 0]
        set xmax [lindex [lindex $lDim 1] 0]
        set ymin [lindex [lindex $lDim 0] 1]
        set ymax [lindex [lindex $lDim 1] 1]
        set zmin [lindex [lindex $lDim 0] 2]
        set zmax [lindex [lindex $lDim 1] 2]
        set xDim [format %.2f [expr abs($xmax -  $xmin)]]
        set yDim [format %.2f [expr abs($ymax -  $ymin)]]
        set zDim [format %.2f [expr abs($zmax -  $zmin)]]
        append text "\n    a water box of the dimensions ${xDim},${yDim},${zDim} (x,y,z in $angsChr)\n"
        append text "    placed at \([format %.2f $xmin],[format %.2f $ymin],[format %.2f $zmin]\)\
        and \([format %.2f $xmax],[format %.2f $ymax],[format %.2f $zmax]\) as minimum and maximum coordinates.\n"
        set watersel [atomselect top "segname \"W.*\""]
        set residuenumber [llength [lsort -unique [$watersel get residue]]]
        if {$QWIKMD::advGui(solvent,minimalbox,$QWIKMD::run) == 1 && $QWIKMD::run == "MD"} {
            append text " The molecule was also rotated to minimize the solvent box volume."
        }
        append text "\n     A total of $residuenumber water molecules TIP3\[[QWIKMD::addReference TIP3P]\] model were placed.\n\n"
        $watersel delete
    }
    return $text
}

proc QWIKMD::ionizeLog {conc cation anion} {
    set molList [molinfo list]
    set solvIndex [lindex $molList [expr [llength $molList] -2 ]]
    set sel [atomselect $solvIndex "all"]
    set charge [eval "vecadd [$sel get charge]"]
    set text "\nAfter the addition of the water molecules, the system presented a total net charge of [format %.3f $charge].\n "
    $sel delete
    if {[expr int([expr abs($charge)])] > 0} {
        append text "The system was neutralized and in total "
    } else {
        if {$conc == 0} {
            append text "No ions were added to the system.\n\n"
            return $text
        } else {
            append text "    "
        }
    }

    set ionsel [atomselect top "segname \"ION.*\""]
    set ionsList [$ionsel get name]
    set ionames [lsort -unique $ionsList]
    set i 0
    foreach ion $ionames {
        append text "[llength [lsearch -all $ionsList $ion]] $ion ions "
        if {$i == 0} {
            append text "plus "
        }
        incr i
    }
    $ionsel delete
    append text "were added to the system making up a salt concentration of $conc mol/L.\n"
    return $text
}


proc QWIKMD::printMD {} {
    set angsChr "Å"
    set text "================================== MD Protocols ====================================\n\n"
    

    append text "The structure [molinfo $QWIKMD::topMol get name] was prepared using VMD\[[QWIKMD::addReference VMD]\] "
    append text "and the plugin QwikMD\[[QWIKMD::addReference QwikMD]\].\n"

    append text "The MD simulations in the present study were performed employing the NAMD molecular dynamics package\[[QWIKMD::addReference NAMD]\]. " 
    append text "The CHARMM36 force field\[[QWIKMD::addReference CHARMM]\] was used in all MD simulations.\n\n"
    
    set confIndex 0
    set solvent ""
    set tabid [$QWIKMD::topGui.nbinput index current]
    if {$tabid == 0} {
        set solvent $QWIKMD::basicGui(solvent,$QWIKMD::run,0)
    } else {
        set solvent $QWIKMD::advGui(solvent,$QWIKMD::run,0)
    }
    foreach prefix $QWIKMD::confFile {
        set mdProtocol $prefix
        if {[string match "*_equilibration*" $prefix ] > 0} {
            set mdProtocol "Minimization and Constrained equilibration MD Simulation "
            
        } elseif {[string match "*_production_smd*" $prefix  ] > 0  && $QWIKMD::mdProtInfo($prefix,smd) == 1} {
            set mdProtocol "SMD simulation "
        } elseif {[string match "*_production*" $prefix  ] > 0 } {
            set mdProtocol "MD Simulation without constrains "
        }
        
        
        append text "\nThe $mdProtocol "
        if {$QWIKMD::run == "QM/MM" && $QWIKMD::mdProtInfo($prefix,qmmm) == 1} {
            append text "QM/MM protocol "
        }
        if {$solvent == "Explicit"} {
            append text "was performed with explicit solvent using the TIP3 water model\[[QWIKMD::addReference TIP3P]\] in the $QWIKMD::mdProtInfo($prefix,ensemble) ensemble.\n"
        } elseif {$solvent == "Implicit"} {
            append text "was performed with implicit solvent represented by the Generalized Born"
            if {$QWIKMD::mdProtInfo($prefix,sasa) == 1} {
                append text "/solvent-accessible surface area "
            }
            append text "model\[[QWIKMD::addReference GBIS]\].\n"
        } else {
            append text "was performed with no solvent representation model, simulation performed in vacuum.\n"
        }

        if {$QWIKMD::mdProtInfo($prefix,ramp) == 1} {
            set tini [lindex $QWIKMD::mdProtInfo($prefix,rampList) 0]
            set tfinal [lindex $QWIKMD::mdProtInfo($prefix,rampList) 1]
            set total [lindex $QWIKMD::mdProtInfo($prefix,rampList) 2]
            append text "A temperature ramp was performed consisted of $total ns of simulation where the temperature was raised from $tini K to $tfinal K "
        }

        if {$QWIKMD::mdProtInfo($prefix,thermostat) != 0 && $QWIKMD::mdProtInfo($prefix,ramp) == 0 && $QWIKMD::mdProtInfo($prefix,run) > 0} {
            append text "The temperature was maintained at $QWIKMD::mdProtInfo($prefix,temp) K using "
            if {$QWIKMD::mdProtInfo($prefix,thermostat) == "Langevin"} {
                append text " Langevin dynamics. "
            } else {
                append text " Lowe-Andersen dynamics\[[QWIKMD::addReference lowAndersen]\]"
            }
        }

        if {$QWIKMD::mdProtInfo($prefix,barostat) != 0  && $QWIKMD::mdProtInfo($prefix,run) > 0} {
            append text "The pressure was maintained at $QWIKMD::mdProtInfo($prefix,press) atm using "
            if {$QWIKMD::mdProtInfo($prefix,barostat) == "Langevin"} {
                append text "Nosé-Hoover Langevin piston\[[QWIKMD::addReference langevinPiston]\]. "
            } else {
                append text "Berendsen pressure bath coupling. "
            }
        }
        append text "A distance cut-off of $QWIKMD::mdProtInfo($prefix,cutoff) $angsChr was applied to short-range, non-bonded interactions, and $QWIKMD::mdProtInfo($prefix,switch) $angsChr for the smothering functions. " 
        if {$solvent == "Explicit"} {
            append text " Long-range electrostatic interactions were treated using the particle-mesh Ewald (PME)\[[QWIKMD::addReference PME]\] method. "
        }
        if {$QWIKMD::mdProtInfo($prefix,run) > 0} {
            append text "The equations of motion were integrated using the r-RESPA multiple time step scheme\[[QWIKMD::addReference NAMD]\] to update the short-range interactions every $QWIKMD::mdProtInfo($prefix,vdw) steps and long-range electrostatics interactions every $QWIKMD::mdProtInfo($prefix,electro) steps. "
            append text "The time step of integration was chosen to be $QWIKMD::mdProtInfo($prefix,timestep) fs for all simulations."
        }
        
        
        if {$QWIKMD::mdProtInfo($prefix,minimize) > 0} {
            append text " Before the MD simulations all the systems were submitted to an energy minimization protocol for $QWIKMD::mdProtInfo($prefix,minimize) steps. "
        }

        if {$QWIKMD::mdProtInfo($prefix,run) > 0 && $QWIKMD::mdProtInfo($prefix,smd) == 0} {

            append text "In this step consisted of $QWIKMD::mdProtInfo($prefix,run) ns of simulation, "

            if {$QWIKMD::mdProtInfo($prefix,const) == 1} {
                append text "the atoms defined by the selection \"$QWIKMD::mdProtInfo($prefix,constsel)\" were restrained. "
            } else {
                append text "no atoms were constrained. "
            }
        }
        
        if {$QWIKMD::mdProtInfo($prefix,smd) == 1} {
            append text "The SMD simulations\[[QWIKMD::addReference SMD]\] of constant velocity stretching (SMD-CV protocol) employing a pulling speed\
             of $QWIKMD::mdProtInfo($prefix,pspeed) $angsChr/ns and a harmonic constraint force of $QWIKMD::mdProtInfo($prefix,smdk) kcal/mol/Å^2 was performed for $QWIKMD::mdProtInfo($prefix,run) ns. "

            set pullingResnames ""
            set pull [atomselect $QWIKMD::topMol $QWIKMD::pullingRessel]
            set prev ""
            foreach res [$pull get resname] resid [$pull get resid] chain [$pull get chain] {
                
                if {"${res}${resid}$chain" != $prev} {
                    append pullingResnames " ${res}${resid}$chain;"
                    set prev "${res}${resid}$chain"
                }
            }
            set pullingResnames [string trimright $pullingResnames ";"]
            $pull delete

            set anchResnames ""
            set anch [atomselect $QWIKMD::topMol $QWIKMD::anchorRessel]
            set prev ""
            foreach res [$anch get resname] resid [$anch get resid] chain [$anch get chain] {
                
                if {"${res}${resid}$chain" != $prev} {
                    append anchResnames " ${res}${resid}$chain;"
                    set prev "${res}${resid}$chain"
                }
            }
            set anchResnames [string trimright $anchResnames ";"]
            $anch delete
            append text "In this step, SMD was employed by harmonically restraining the position of$anchResnames residues, and moving a second restraint residues$pullingResnames, with constant velocity in the axis defined by the two group of restrained residues. "

        }
        append text "\n\n"
    
        if {$QWIKMD::run == "QM/MM"} {
            append text "The quantum mechanical calculations were executed by $QWIKMD::advGui(qmoptions,soft,$mdProtocol)\[[QWIKMD::addReference $QWIKMD::advGui(qmoptions,soft,$mdProtocol)]\]\
            , which is executed by the NAMD QM/MM interface\[[QWIKMD::addReference QMMM]\].\nThe following command line(s) were executed by the QM package: $QWIKMD::mdProtInfo(qmmm,cmdline,$mdProtocol).\n "
            if {$QWIKMD::advGui(qmoptions,ptcharge,$mdProtocol) == "On"} {
                append text "The electrostatics interactions between the QM and MM atoms were considered (\"Electrostatic Embedding\" scheme). "
                if {$QWIKMD::advGui(qmoptions,switchtype,$mdProtocol) != "Off"} {
                    append text "The smooth function used to scale down point charges sent to QM calculations was $QWIKMD::advGui(qmoptions,switchtype,$mdProtocol). "
                } else {
                    append text "No smooth function was used to scale the electrostatic interactions. An abrupt decay in electrostatic forces may occur. "
                }
                if {$QWIKMD::advGui(qmoptions,ptchrgschm,$mdProtocol) != "None"} {
                    append text "The point charges presented to the QM system were altered using the $QWIKMD::advGui(qmoptions,ptchrgschm,$mdProtocol) scheme. "
                } else {
                    append text "The point charges presented to the QM system were not altered. "
                }
                append text "The partial charges of classical atoms involved in the QM/MM bonds and classical atoms directly connected to them were treated using the $QWIKMD::advGui(qmoptions,qmbondsheme,$mdProtocol) scheme. "
               
                if {$QWIKMD::advGui(qmoptions,cmptcharge,$mdProtocol) == "Off"} {
                    append text "No custom point charges were passed to the QM system.\n "
                } else {
                    set qmNum [$QWIKMD::advGui(qmtable) size]
                    for {set qmID 1} {$qmID <= $qmNum} {incr qmID} {
                        append text "Atoms $QWIKMD::advGui(qmtable,$qmID,pcDist) A distanced from the QM region $qmID were passed as point charges to the QM system.\n "
                    }
                }
            } else {
                append text "The electrostatics interactions between the QM and MM atoms were ignored (\"Mechanical Embedding\" scheme).\n "
            }
            if {$QWIKMD::advGui(qmoptions,lssmode,$mdProtocol) == "Off"} {
                append text "No exchange of solvent molecules between the QM and MM  were performed.\n "
            } else {
                append text "Exchange of solvent molecules between the QM and MM were performed, based on "
                if {$QWIKMD::advGui(qmoptions,lssmode,$mdProtocol) == "Distance"} {
                    append text "minimal distance between a solvent atom and a non-solvent QM atom."
                } else {
                    set qmNum [$QWIKMD::advGui(qmtable) size]
                    append text "the distance to the Center Of Mass of the selection "
                    for {set qmID 1} {$qmID <= $qmNum} {incr qmID} {
                        append text "$QWIKMD::advGui(qmtable,$qmID,qmCOMSel) (for QM region $qmID), " 
                    }
                    append text ". \n"
                }
            }
        }
    }
    append text "\n\nBibliography:\n"
    set index 1
    foreach refText $QWIKMD::references {
        append text "\{$index\} $refText\n\n"
        incr index
    }
    append text "[string repeat "=" 81]\n\n"
    return $text
}

proc QWIKMD::loadDCD {} {
    set systemTime [clock seconds]
    set text "================== Load simulations on [clock format $systemTime -format %H:%M-%Z], [clock format $systemTime -format %D] ==================\n\n"
    if {$QWIKMD::loadinitialstruct == 1} {
        append text "The initial structure [lindex $QWIKMD::inputstrct 1] was loaded.\n" 
    }
    if {[llength $QWIKMD::confFile] > 0} {
        append text "The following structure/trajectories were loaded:\n"
        foreach conf $QWIKMD::confFile {
            append text "    $conf;\n"
        }
        if {$QWIKMD::loadlaststep == 1} {
            append text "Only the last frame of each trajectory was loaded. "
        } else {
            append text "The trajectories were loaded every $QWIKMD::loadstride frames. "
        }
    }
    set textaux ""
    if {$QWIKMD::loadremovewater == 1} {
        append textaux " water molecules"
    }
    if {$QWIKMD::loadremoveions == 1} {
        if {$textaux != ""} {
            append textaux ", "
        }
        append textaux " solvent ion molecules"
    }
    if {$QWIKMD::loadremovehydrogen == 1} {
        if {$textaux != ""} {
            append textaux " and "
        }
        append textaux " hydrogen atoms"
    }

    if {$textaux != ""} {
        append text " The $textaux were not included in the loading process.\n\n"
    } else {
        append text "\n\n"
    }
    return $text

}
proc QWIKMD::loadPrevious {name} {
    set text \
"============================== Structure Preparation ===============================\n\n"

    append text "The simulation was prepared from a previously loaded simulation - ${name}.qwikmd.\n"
    if {$QWIKMD::loadinitialstruct == 1} {
        append text "The initial structure [lindex $QWIKMD::inputstrct 1] was loaded;\n"
    }
    append text "The following structure/trajectories were loaded:\n"
    
    if {[llength $QWIKMD::loadprotlist] > 0} {
        foreach dcd $QWIKMD::loadprotlist {
            append text "    $dcd;\n"
        }
    }
    if {$QWIKMD::loadlaststep == 1} {
            append text "Only the last frame of each trajectory was loaded.\n"
        } else {
            append text "The trajectories were loaded every $QWIKMD::loadstride frames.\n"
        }
}

proc QWIKMD::restartFromPrevious {name frame} {
    set text ""
    append text "The log file (${name}.infoMD) with the information regarding the preparation and execution of the previous simulation\
    is stored in setup folder.\n"
    if {$QWIKMD::curframe > 0} {
        append text "The simulation in preparation will be extended from the last step of the ${QWIKMD::loadprotlist} trajectory.\n"
    } else {
        append text "The simulation in preparation will start from the frame $frame selected by the user.\n"
    }
    
    return $text
}


proc QWIKMD::printQMRegions {} {
    set qmNum [$QWIKMD::advGui(qmtable) size]
    set be " was"
    if {$qmNum > 1} {
        set be "s were"
    }
    set text "$qmNum region$be defined for the QM/MM calculations. Next, each QM region is defined:\n\n"
    append text "    <QM ID> <Number of Atoms> <QM Atom Selection> <Charge> <Multiplicity> <QM Solvent Molecules Radius>\n"
    for {set i 0} {$i < $qmNum} {incr i} {
        set qmID [expr $i +1]
        set numatoms [$QWIKMD::advGui(qmtable) cellcget [expr $qmID -1],1 -text]
        append text "    $qmID $numatoms \"$QWIKMD::advGui(qmtable,$qmID,qmRegionSel)\" $QWIKMD::advGui(qmtable,$qmID,charge) \
        $QWIKMD::advGui(qmtable,$qmID,multi) $QWIKMD::advGui(qmtable,$qmID,solvDist)\n"

    }
    set costpclist ""
    foreach prefix $QWIKMD::confFile {
        if {$QWIKMD::advGui(qmoptions,cmptcharge,$prefix) == "On"} {
            append costpclist "$prefix, "
        }
    }
    if {[llength $costpclist] > 0} {
        set costpclist [string trimright $costpclist ", "]
        append text "\nThe follwing protocols used custom point charges in QM calculation and were defined as follow: "
        append text "    <QM ID> <Custom PC Radius>\n"
        for {set i 0} {$i < $qmNum} {incr i} {
            set qmID [expr $i +1]
            append text "    $qmID $numatoms $QWIKMD::advGui(qmtable,$qmID,pcDist)\n"
        }
    }
    return $text
}

proc QWIKMD::printDiffQMCharge {totdiff replaceIon changeCharge} {
    set text "The user assigned charges to the QM regions, different from the charges defined\
    by the topology files.\n"
    append text "    (<QM ID> <Original Topology Charge> <Assigned Charge>)\n"
    set qmNum [$QWIKMD::advGui(qmtable) size]
    for {set i 0} {$i < $qmNum} {incr i} {
        set qmID [expr $i +1]
        append text "    $qmID $QWIKMD::advGui(qmtable,$qmID,qmTopoCharge) $QWIKMD::advGui(qmtable,$qmID,charge)\n"
    }
    append text "To keep a neutral net charge of the system and compensate the total difference in charges of $totdiff, the following ions were \
    change automatically:\n\n"
    if {[llength $changeCharge] > 0} {
        append text "The charges of the following ions were set to 0:\n\n"
        append text "    (<Atom Name> <Atom Index> <Distance to solute Center of Mass>)\n"
    }
    foreach change $changeCharge {
        append text "    [join $change]\n"
    }
    if {[llength $replaceIon] > 0} { 
        append text "The following ions were replaced by an ion with the opposite charge:\n\n"
        append text "    (<Atom Name> <Atom Index> <New Atom Name> <Distance to solute Center of Mass>)\n"
    }
    foreach replace $replaceIon {
        append text "    [join $replace]\n"
    }
    return $text
}

proc QWIKMD::printTempTopo {} {
    set text "The following residues were described using temporary topologies generated by QwikMD:\n"
    append text "    (<Residue Name> <Topology File>)\n"
    set index [lsearch -index 0 $QWIKMD::userMacros "QM"]
    set macro [lindex $QWIKMD::userMacros $index]
    set residues [lindex $macro 1]
    set files [lindex $macro 3]
    foreach res $residues topo $files {
        append text "    $res [file tail $topo]\n"
    }
    append text "These topology files do not contain any information regarding bonded interactions,\
    for instance, bonds, angles and dihedrals. These topologies should not be used to perform regular MD simulations.\
     The topology files can be found in the setup folder.\n"
}

proc QWIKMD::printQMMacroFixAtoms {} {
    set text "To allow the solvent atoms to relax/equilibrate around the molecules described by the temporary topologies, \
    all the non-solvent atoms within 5 A surrounding the these molecules (inclusive) had their coordinates fixed.\n"    
}
proc QWIKMD::printRMSD {numframes seltext const} {
    
    set text "The Root-Mean-Square Deviation (RMSD) was calculated over $numframes frames corresponding to a total of [QWIKMD::format2Dec $QWIKMD::rmsdprevx] ns.\n"
    set seltextalign ""
    if {$QWIKMD::advGui(analyze,basic,alicheck) == 1} {
        if {$QWIKMD::advGui(analyze,basic,alientry) != "" && $QWIKMD::advGui(analyze,basic,alientry) != "Type Selection"} {
            set seltextalign $QWIKMD::advGui(analyze,basic,alientry)
        } else {
            set seltextalign $QWIKMD::advGui(analyze,basic,alicombo)
        }
        append text "The trajectory frames were aligned against the atom selection \"$seltextalign\" of the first frame. "
    }
    append text "The RMSD was calculated for the atom selection \"$seltext\".\n\n"

    return $text

}

proc QWIKMD::printEnergies {file ttime limit rwindow energyfreq const tot kin elect pot bond angle dihedral vdw} {
    set text ""
    set timeunit ns
    if {$QWIKMD::run == "QM/MM"} {
        set timeunit ps
    }
    set text "The energies of the file $file were evaluated over time for [QWIKMD::format2Dec $ttime] $timeunit every [QWIKMD::format5Dec [expr $limit * $const]] $timeunit ($limit steps), on a running average of [QWIKMD::format4Dec [expr $rwindow * $energyfreq * $const]] $timeunit ([QWIKMD::format0Dec [expr $rwindow * $energyfreq]] steps) average length (the energies output frequency was $energyfreq steps or \
        [QWIKMD::format5Dec  [expr $energyfreq * $const] ] $timeunit).\nThe energies evaluated were:"
    if {$tot ==1} {
        append text " total;"
    }
    if {$kin ==1} {
        append text " kinetic;"
    }
    if {$elect ==1} {
        append text " electrostatic;"
    }
    if {$pot ==1} {
        append text " potential;"
    }
    if {$bond ==1} {
        append text " bonds;"
    }
    if {$angle ==1} {
        append text " angles;"
    }
    if {$dihedral ==1} {
        append text " dihedral angles;"
    }
    if {$vdw ==1} {
        append text " vdw"
    }
    set text [string trimright $text ";"]
    append text ".\n\n"
    return $text
}

proc QWIKMD::printThermo {file ttime limit rwindow energyfreq const temp press vol} {
    set timeunit ns
    if {$QWIKMD::run == "QM/MM"} {
        set timeunit ps
    }
    set text "The thermodynamic properties of the file $file were evaluated over time for [QWIKMD::format2Dec $ttime] $timeunit every [QWIKMD::format5Dec [expr $limit * $const]] $timeunit ($limit steps), on a running average of [QWIKMD::format4Dec [expr $rwindow * $energyfreq * $const]] $timeunit ([QWIKMD::format0Dec [expr $rwindow * $energyfreq]] steps) average length (the energies output frequency was $energyfreq steps or\
    [QWIKMD::format5Dec  [expr $energyfreq * $const] ] $timeunit).\nThe properties evaluated were:"
    if {$temp ==1} {
        append text " temperature;"
    }
    if {$press ==1} {
        append text " pressure;"
    }
    
    if {$vol ==1} {
        append text " volume;"
    }

    set text [string trimright $text ";"]
    append text ".\n\n"
    return $text
}

proc QWIKMD::printQMEnergies {file ttime limit rwindow energyfreq const qmregions} {
    set text ""
    set text "The QM energies of the region\(s\) [join $qmregions] of the file $file were evaluated over time for [QWIKMD::format2Dec $ttime] ps every [QWIKMD::format5Dec [expr $limit * $const]] ps ($limit steps), on a running average of [QWIKMD::format4Dec [expr $rwindow * $energyfreq * $const]] ps ([QWIKMD::format0Dec [expr $rwindow * $energyfreq]] steps) average length (the energies output frequency was $energyfreq steps or \
        [QWIKMD::format5Dec  [expr $energyfreq * $const] ] ps).\n\n"
    return $text
}

proc QWIKMD::printHbonds {numframes} {
    set angsChr "Å"
    set text "The hydrogen bonds "
    set seltextalign ""
    if {$QWIKMD::hbondssel == "intra"} {
        append text "between the solute (everything but water and ions) "
    } elseif {$QWIKMD::hbondssel == "inter"} {
        append text "between the solute and solvent (e.g. protein and water) "
    } else {
        append text "between the polar atoms of selection $QWIKMD::advGui(analyze,advance,hbondsel1entry) and $QWIKMD::advGui(analyze,advance,hbondsel2entry) "
    }
    append text "were evaluated over $numframes frames corresponding to a total of [QWIKMD::format2Dec $QWIKMD::hbondsprevx] ns. An atom donor-hydrogen-acceptor angle of 30 degrees and a\
    donor-acceptor cut-off distance of 3.5 $angsChr were used to calculate the hydrogen bonds.\n\n"

    return $text

}

proc QWIKMD::printSMD {ttime dist limit rwindow smdfreq const} {
    set angsChr "Å"
    set text "The SMD forces were evaluated "
    if {$QWIKMD::smdxunit == "time"} {
        append text "over time for [QWIKMD::format2Dec $ttime] ns every [QWIKMD::format5Dec [expr $limit * $const]] ns ($limit steps), on a running average of [QWIKMD::format4Dec [expr $rwindow * $const]] ns ($rwindow steps) average length (the forces output frequency was $smdfreq steps\
or [QWIKMD::format5Dec  [expr $smdfreq * $const] ] ns). "
    } else {
        append text "over distance for [QWIKMD::format5Dec $dist] $angsChr on a running average of [expr int(1.5 * $limit)] steps or (the smd output frequency was $smdfreq steps). "
    }
    append text "\n\n"
    return $text
}

proc QWIKMD::printRMSF {start end skip seltext} {
    set text "The Root-Mean-Square Fluctuation (RMSF) was calculated over [expr int([expr [expr $end - $start]/$skip]) +1] frames, staring on frame [expr $start +1] to frame [expr $end +1], with a step of $skip frames.\n"
    set seltextalign ""
    if {$QWIKMD::advGui(analyze,advance,rmsfalicheck) == 1} {
        if {$QWIKMD::advGui(analyze,advance,rmsfalignsel) != "" && $QWIKMD::advGui(analyze,advance,rmsfalignsel) != "Type Selection"} {
            set seltextalign $QWIKMD::advGui(analyze,advance,rmsfalignsel)
        } else {
            set seltextalign $QWIKMD::advGui(analyze,advance,rmsfaligncomb)
        }
        append text "The trajectory frames were aligned against the atom selection \"$seltextalign\" of the first frame. "
    }
    append text "The RMSF was calculated for the atom selection \"$seltext\".\n\n"

    return $text
}

proc QWIKMD::printSASA {ttime nframes sel restricsel} {
    set angsChr "Å"
    set text "The Solvent-Accessible Surface Area (SASA) of the atom selection \"$sel\" was calculated over $nframes frames ([QWIKMD::format2Dec $ttime] ns), based on a probe sphere of 1.4 $angsChr (approx. water molecule radius).\n"
    set seltextalign ""
    if {$sel != $restricsel} {
        append text "The output results were restricted to the atom selection \"$restricsel\". "
    }
    append text "SASA values were presented for the all selection (\"$sel\") as well as per residue values.\n\n"

    return $text
}

proc QWIKMD::printContSASA {ttime nframes sel1 sel2} {
    set angsChr "Å"
    set text "The Contacting Surface Area between the atom selection1 \"$sel1\" and the selection2 \"$sel2\" was calculated over $nframes frames ([QWIKMD::format2Dec $ttime] ns), based on a probe sphere of 1.4 $angsChr (approx. water molecule radius).\n"
    append text "The contacting surface areas were evaluated by calculating the Solvent-Accessible Surface Area (SASA) of selection1 in the absence of the selectio2 and subtracted by the SASA of selection1 in the presence of the selectio2. "
    append text "The process was repeated, but this time the SASA values were calculated for the selectio2 in the absence and presence of the selectio1. "
    append text "The contacting regions were formed by the atoms of one selection distanced less or equal than 5 $angsChr of the other selection. "

    append text "SASA values were presented for the total contacting surface area of selection1 and selection2 as well as per residue values.\n\n"

    return $text
}

proc QWIKMD::addReference {ref} {
    set ind [lsearch -index 0 $QWIKMD::refIndex $ref]
    if {$ind == -1} {
        set refText [QWIKMD::getReference $ref]
        set refindexList ""
        set length [llength $QWIKMD::references]
        for {set i 1} {$i <= [llength $refText]} {incr i} {
            append refindexList "[expr $i + $length],"
        }
        set refindexList [string trimright $refindexList ","]
        set QWIKMD::references [concat $QWIKMD::references $refText]
        lappend QWIKMD::refIndex [list $ref $refindexList]
    } else {
        set refindexList [lindex [lindex $QWIKMD::refIndex $ind] 1 ]
    }
    return $refindexList
}


proc QWIKMD::getReference {ref} {

    switch $ref {
        VMD {
            return [list {Humphrey, W., Dalke, A. and Schulten, K., "VMD - Visual Molecular Dynamics", J. Molec. Graphics, 1996, vol. 14, pp. 33-38.}]
        }
        QwikMD {
            return [list {Ribeiro, J. V., Bernardi, R. C., Rudack, T., Stone, J. E., Phillips, J. C., Freddolino, P. L. and Schulten, K.,"QwikMD-integrative molecular dynamics toolkit for novices and experts", Sci. Rep., 2016}]
        }
        NAMD {
            return [list {Phillips J. C., Braun, R. , Wang, W., Gumbart, J. , Tajkhorshid, E., Villa, E. , Chipot, C. , Skeel, R. D., Kale, L., and Schulten, K., "Scalable molecular dynamics with NAMD", J. Comp. Chem, 2005, vol 26, pp. 1781-1802}]
        }
        CHARMM {
            set a {Best, R. B., Zhu, X., Shim, J., Lopes, P. E. M., Mittal, J., Feig, M. and MacKerell, A. D., "Optimization of the additive CHARMM All-atom protein force field targeting improved sampling of the backbone φ, ψ and side-chain χ 1and χ 2dihedral Angles", J. Chem. Theory Comput.,2012, vol. 8, pp. 3257–3273.}
            set b {MacKerell, A. D., Jr., Bashford, D., Bellott, M., R. L. , Jr., Evanseck, J. D., Field, M. J., Fischer, S., Gao J., Guo, H., Ha S., Joseph-McCarthy, D., Kuchnir, L.,\
             Kuczera, K., Lau, F. T. K., Mattos, C., Michnick. S., Ngo, T., Nguyen, D. T., Prodhom, B., Reiher, W. E., Roux, B., Schlenkrich, M., Smith, J. C., Stote, R., Straub, J.,\
              Watanabe, M., Wiórkiewicz-Kuczera, J., Yin, D. and Karplus M., "All-atom empirical potential for molecular modeling and dynamics studies of proteins", J. Phys. Chem. B, 1998 , vol. 102, pp. 3586-3616}
            return [list $a $b]
        }
        TIP3P {
            return [list {Jorgensen, W. L., Chandrasekhar, J., Madura, J. D., Impey, R. W. and Klein, M. L., "Comparison of simple potential functions for simulating liquid water", J. Chem. Phys., 1983, vol 79, 6127-6129.}]
        }
        GBIS {
            set a {Still, W. C., Tempczyk, A., Hawley, R. C.; Hendrickson, T., "Semianalytical treatment of solvation for molecular mechanics and dynamics", J. Am. Chem. Soc., 1990, vol. 112, pp. 3257-3273.}
            set b {Tanner, D. E., Phillips J. C. and Schulten, K., "GPU/CPU Algorithm for Generalized Born/Solvent-Accessible Surface Area Implicit Solvent Calculations", J. Phys. Chem. B,1998 , vol. 102, pp. 3586-3616}
            set c {Tanner, D. E., Chan, K. , Phillips J. C. and Schulten, K., "Parallel generalized Born implicit solvent calculations with NAMD.", J. Chem. Theor. Comp., 2011 , vol. 7, pp. 3635-3642}
            return [list $a $b $c]
        }
        PME {
            return [list {Darden, T., York, D. & Pedersen and L., "Particle mesh Ewald: An Nlog(N) method for Ewald sums in large systems", J. Chem. Phys., 1993, vol. 98, pp. 10089-10092}]
        }
        lowAndersen {
            return [list {Koopman, E. A.  and Lowe, C. P., "Advantages of a Lowe-Andersen thermostat in molecular dynamics simulations", J. Chem. Phys., 2006, vol. 124}]
        }
        langevinPiston {
            set a {Martyna, G. J., Tobias, D. J. and Klein, M. L., "Constant pressure molecular dynamics algorithms", J. Chem. Phys., 1994, vol. 101}
            set b {Feller, S. E., Zhang, Y., Pastor, R. W. and Brooks, B.R., "Constant pressure molecular dynamics simulation: The Langevin piston method", J. Chem. Phys., 1995, vol. 103}
            return [list $a $b]
        }
        SMD {
            return [list {Izrailev, S., Stepaniants, S., Balsera, M., Oono, Y. and Schulten, K.,"Molecular dynamics study of unbinding of the avidin-biotin complex", Biophys. J., 1997, vol 72, pp. 1568–1581}]
        }
        QMMM {
            return [list {Melo, M. C. R.,  Bernardi, R. C., Rudack, T., Scheurer, M., Riplinger, C., Phillips, J. C., Maia, J. D. C., Rocha, G. B.,\
            Ribeiro, J. V., Stone, J. E., Neese, F., Schulten, K. and Luthey-Schulten, Z., "NAMD goes quantum: A new integrative suite for QM/MM simulations",Submitted, 2017}]
        }
        MOPAC {
            set a {Stewart, J. J., "MOPAC: a semiempirical molecular orbital program.", J Comput Aided Mol Des. 1990; vol. 4}
            set b {Maia, J. D. C., Carvalho, G. A. U., Mangueira, Jr. C. P. , Santana, S. R., Cabral, L. A. F., Rocha, G. B.,"Gpu linear algebra \
            libraries and gpgpu programming for accelerating mopac semiempirical quantum chemistry calculations", J. Chem. Theory Comput.,2012, vol 8}
            return [list $a $b]
        }
        ORCA {
            return [list {Nesse, F., "The ORCA program system", Wiley Interdiscip. Rev. Comput. Mol. Sci., 2012 vol. 2}]
        }
        default {}
    }
}

