
# generate water block coordinates (VMD)
# replicate water block (psfgen)
# combine water block and solute in psfgen (psfgen)
# Create pdb with just the water you want (VMD)
# merge the solute and the cutout water (psfen)

# Changes since version 1.0:
#   Fixed a bug in the water overlap code which left waters close to the
#     solute.  Never figured out what was wrong; probably just using the wrong
#     coordinates.
#
#   Added a sanity check to make sure that solvate didn't leave any waters 
#     near the solute.

package require psfgen 1.2
package provide solvate 1.2

proc solvate_usage { } {
  puts "Usage: solvate <psffile> <pdbfile> <option1?> <option2?>..."
  puts "Usage: solvate <option1?> <option2?>...  to just create a water box" 
  puts "Options:"
  puts "    -o <output prefix> (data will be written to output.psf/output.pdb)"
  puts "    -s <segid prefix> (should be either one or two letters; default WT)"
  puts "    -b <boundary> (minimum distance between water and solute, default 2.4)"
  puts "    -minmax {{xmin ymin zmin} {xmax ymax zmax}}"
  puts "    -t <pad in all directions> (override with any of the following)"
  puts "    -x <pad negative x>"
  puts "    -y <pad negative y>"
  puts "    -z <pad negative z>"
  puts "    +x <pad positive x>"
  puts "    +y <pad positive y>"
  puts "    +z <pad positive z>"
  error ""
}

proc solvate {args} {
  global env 
 
  # Print usage information if no arguments are given
  if { ![llength $args] } {
    solvate_usage
  }

  # The first argument that starts with a "-" marks the start of the options.
  # Arguments preceding it, if any, must be the psf and pdb files.
  set arg0 [lindex $args 0]
  if { [llength $args] >= 2 && [string range $arg0 0 0] != "-" } {
    set psffile [lindex $args 0]
    set pdbfile [lindex $args 1]
    set args [lrange $args 2 end]
  }
 
  foreach elem { -b +x +y +z -x -y -z -minmax -t -o} {
    set bounds($elem) 0
  }
  set bounds(-s) WT
  set bounds(-b) 2.4

  set n [llength $args]
  # check for even number of args
  if { [expr fmod($n,2)] } { solvate_usage }
    
  #
  # Get all command line options
  #
  for { set i 0 } { $i < $n } { incr i 2 } {
    set key [lindex $args $i]
    set val [lindex $args [expr $i + 1]]
    if { ! [info exists bounds($key)] } {
      solvate_usage 
    }
    set cmdline($key) $val 
  }

  # 
  # Get minmax if specified, or use minmax of solute
  #
  if { [info exists cmdline(-minmax) ] } {
    set bounds(-minmax) $cmdline(-minmax)
  } else {
    if { [info exists psffile] } {  
      mol load psf $psffile pdb $pdbfile
      set sel [atomselect top all]
      set bounds(-minmax) [measure minmax $sel]
      mol delete top
    } else {
      error "No psf/pdb, so minmax must be specified."
    }
  }

  # 
  # If -t was specified, use it for all pads
  #
  if { [info exists cmdline(-t)] } {
    foreach elem { -x -y -z +x +y +z } {
      set bounds($elem) $cmdline(-t)
    }
  }

  # 
  # Fill in all other specified options
  #  
  set outputname solvate
  if { [info exists cmdline(-o)] } {
    set outputname $cmdline(-o)
  }
  foreach elem [array names cmdline] {
    set bounds($elem) $cmdline($elem)
  }

  set env(SOLVATEPREFIX) $bounds(-s)
  set prefix $bounds(-s)

  foreach {min max} $bounds(-minmax) {} 
  set min [vecsub $min [list $bounds(-x) $bounds(-y) $bounds(-z)]]
  set max [vecadd $max [list $bounds(+x) $bounds(+y) $bounds(+z)]]

  #
  # generate combined psf/pdb containing solute and one replica of water
  # VMD can't do multi-molecule atom selections...
  #

  set old [psfcontext new]
  # If I were really paranoid I would wrap the rest of the script in a catch
  # block to make sure that the context got switched back to its original 
  # state.  I suppose the caller could ensure this with 
  # "catch psfcontext 0 delete"

  if { [info exists psffile] } {
    readpsf $psffile
    coordpdb $pdbfile
  }
  readpsf $env(SOLVATEDIR)/wat.psf
  coordpdb $env(SOLVATEDIR)/wat.pdb
  writepsf combine.psf
  writepdb combine.pdb
 
  delatom QQQ

  set watsize 65.4195

  #
  # Extract info about where to put the water
  #
  foreach {xmin ymin zmin} $min {}
  foreach {xmax ymax zmax} $max {}

  set dx [expr $xmax - $xmin]
  set dy [expr $ymax - $ymin]
  set dz [expr $zmax - $zmin]

  set nx [expr int($dx/$watsize) + 1]
  set ny [expr int($dy/$watsize) + 1]
  set nz [expr int($dz/$watsize) + 1]

  puts "replicating $nx by $ny by $nz"

  #
  # Read combined structure back in and generate a new psf/pdb file with just
  # the waters we want.
  #
  mol load psf combine.psf pdb combine.pdb
  set wat [atomselect top "segid QQQ"]
  set wat_unique_res [lsort -unique -integer [$wat get resid]]
  set watres [$wat get resid]
  set watname [$wat get name] 

  topology $env(SOLVATEDIR)/wat.top
  set n 0
  set rwat $bounds(-b)
  set seglist {}

  for { set i 0 } { $i < $nx } { incr i } {
    set movex [expr $xmin + $i * $watsize]
    for { set j 0 } { $j < $ny } { incr j } {
      set movey [expr $ymin + $j * $watsize]
      for { set k 0 } { $k < $nz } { incr k } {
        set movez [expr $zmin + $k * $watsize]
        set vec [list $movex $movey $movez]

        $wat moveby $vec 

        # Create new water replica... 
        incr n
        segment ${prefix}$n {
          first NONE
          last NONE
          foreach res $wat_unique_res {
            residue $res TIP3
          }
        }
        lappend seglist ${prefix}$n
        foreach resid $watres name $watname pos [$wat get {x y z}] {
          coord ${prefix}$n $resid $name $pos
        }
        # ... and delete overlapping waters and those outside the box.
        set sel [atomselect top "segid QQQ and name OH2 and same residue as \
	  (x < $xmin or x > $xmax or y < $ymin or y > $ymax or \
	  z < $zmin or z > $zmax or within $rwat of (not segid QQQ))"]
        foreach resid [$sel get resid] {
          # Use catch because the atom might have already been deleted 
          catch { delatom ${prefix}$n $resid }
        }
        unset upproc_var_$sel 
    
        $wat moveby [vecinvert $vec] 
      } 
    }
  }
  writepsf $outputname.psf
  writepdb $outputname.pdb

  # delete the current context and restore the old one
  psfcontext $old delete

  # Test to make sure we didn't miss any waters.  Add a fudge factor 
  # of sqrt(3 * .001^2) to the distance check because of limited precision
  # in the PDB file.
  mol delete top 
  mol load psf $outputname.psf pdb $outputname.pdb
  set rwat [expr $rwat - .001732]
  set sel [atomselect top "segid $seglist and within $rwat of (not segid $seglist)"]
  set num [$sel num]
  mol delete top
  if { $num != 0 } {
    puts "Found $num water atoms near the solute!  Please report this bug to"
    puts "vmd@ks.uiuc.edu, including, if possible, your psf and pdb file."
    error "Solvate 1.1 failed."  
  }
  puts "Solvate 1.1 completed successfully."
  return [list $min $max]
}
  
proc simple { psffile pdbfile watpsf watpdb } {
	# Create psf/pdb files that contain both our protein as well as
	# a box of equilibrated water.  The water box should be large enough
	# to easily contain our protein.
	resetpsf
	readpsf $psffile
	readpsf $watpsf
	coordpdb $pdbfile
	coordpdb $watpdb

	writepsf combine.psf
	writepdb combine.pdb

	mol load psf combine.psf pdb combine.pdb

	# Assume that the segid of the water in watpsf is QQQ
	# We want to delete waters outside of a box ten Angstroms
	# bigger than the extent of the protein. 
	set protein [atomselect top "not segid QQQ"]
	set minmax [measure minmax $protein]
	foreach {min max} $minmax { break }
	foreach {xmin ymin zmin} $min { break }
	foreach {xmax ymax zmax} $max { break }
    set xmin [expr $xmin - 10]
    set ymin [expr $ymin - 10]
    set zmin [expr $zmin - 10]
    set xmax [expr $xmax + 10]
    set ymax [expr $ymax + 10]
    set zmax [expr $zmax + 10]

	# Center the water on the protein.  Also update the coordinates held
	# by psfgen.
	set wat [atomselect top "segid QQQ"]
	$wat moveby [vecsub [measure center $protein] [measure center $wat]]
	foreach atom [$wat get {segid resid name x y z}] {
		foreach {segid resid name x y z} $atom { break }
		coord $segid $resid $name [list $x $y $z]
	}

	# Select waters that we don't want in the final structure.
	set outsidebox [atomselect top "segid QQQ and (x <= $xmin or y <= $ymin \
		or z <= $zmin or x >= $xmax or y >= $ymax or z >= $zmax)"]
	set overlap [atomselect top "segid QQQ and within 2.4 of (not segid QQQ)"]

	# Get a list of all the residues that are in the two selections, and delete
	# those residues from the structure.
	set reslist [concat [$outsidebox get resid] [$overlap get resid]]
	set reslist [lsort -unique -integer $reslist]

	foreach resid $reslist {
		delatom QQQ $resid
	}

	# That should do it - write out the new psf and pdb file. 
	writepsf solvate.psf 
	writepdb solvate.pdb

	# Delete the combined water/protein molecule and load the system that
	# has excess water removec.
	mol delete top
	mol load psf solvate.psf pdb solvate.pdb

	# Return the size of the water box
	return [list [list $xmin $ymin $zmin] [list $xmax $ymax $zmax]]
}

proc solvategui {} {
  if { [winfo exists .solvategui] } {
    wm deiconify .solvategui
    return
  }
  set w [toplevel ".solvategui"]
  wm title $w "Solvate"
  label $w.blurb -text "The solvate package has been loaded.\n\
    To get a list of available options, type 'solvate' at the\n\
    VMD command line."
  pack $w.blurb -side left
  return $w
}

