/***************************************************************************
 *cr
 *cr            (C) Copyright 1995-2003 The Board of Trustees of the
 *cr                        University of Illinois
 *cr                         All Rights Reserved
 *cr
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: corplugin.c,v $
 *      $Author: johns $       $Locker:  $             $State: Exp $
 *      $Revision: 1.6 $       $Date: 2003/08/13 19:58:11 $
 *
 ***************************************************************************/

#include "molfile_plugin.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#define COR_RECORD_LENGTH   80

/* Remove leading and trailing whitespace from the string str of size n */
void strip(char *str, int n)
{
  char *beg, *end;
  beg = str;
  end = str + (n-2); /* Point to the last non-null character in the string */

  /* Remove leading whitespace */
  while(*beg == ' ') {
    beg++;
  }

  /* Remove trailing whitespace and null-terminate */
  while(*end == ' ') {
    end--;
  }
  *(end+1) = '\0';

  /* Shift the string */
  while(*str != '\0') {
    *str = *beg;
    str++;
    beg++;
  }

  return;
}


/* Open the .cor file and skip past the remarks to the first data section.
 * Returns the file pointer, or NULL if error.  Also puts the number of
 * atoms in the molecule into the given integer.  
 */
static FILE *open_cor_file(const char *fname, int *natom) {
  char inbuf[COR_RECORD_LENGTH+2], header[6];
  FILE *f;

  *natom = MOLFILE_NUMATOMS_NONE;

  if (!fname)
    return NULL;

  if ((f = fopen(fname, "r")) == NULL)
    return NULL;

  /* Read and discard the header */
  do {
    if (fgets(inbuf, COR_RECORD_LENGTH+1, f) == NULL) {
      fclose(f);
      return NULL;
    }

    if (sscanf(inbuf, " %5c", &header) != 1) {
      fclose(f);
      return NULL;
    }

  } while (header[0]=='*');

  header[5] = '\0';
  *natom = atoi(header);

  return f;
}

/* Read in the next atom info into the given storage areas; this assumes
   that file has already been moved to the beginning of the atom records.
   Returns the serial number of the atom. If there is an error, returns -1.*/
static int get_cor_atom(FILE *f, char *atomName, char *atomType, char
    *resName, char *segName, int *resId) {
  char inbuf[COR_RECORD_LENGTH+2], numAtomStr[6], resIdStr[6];
  int numAtom;

  if (feof(f) || ferror(f)) {
    return -1;
  }

  if (inbuf != fgets(inbuf, COR_RECORD_LENGTH+1, f)) {
    return -1;
  }

  if (strlen(inbuf) < 60) {
    fprintf(stderr, "Line too short in cor file: \n%s\n", inbuf);
    return -1;
  }

  numAtomStr[5] = '\0';
  resIdStr[5] = '\0';
  resName[5] = '\0';
  atomName[5] = '\0';
  segName[5] = '\0';

  if (sscanf(inbuf, "%5c%5c%5c%5c%*10c%*10c%*10c%5c", 
             numAtomStr, resIdStr, resName, atomName, segName) != 5) {
    fprintf(stderr, "Improperly formatted line in cor file: \n%s\n", inbuf);
    return -1;
  }

  numAtom = atoi(numAtomStr);
  *resId = atoi(resIdStr);
  strip(resName, 6);
  strip(atomName, 6);
  strip(segName, 6);
  strcpy(atomType, atomName);

  return numAtom;
}


/*
 * API functions
 */

typedef struct {
  FILE *file;
  int numatoms;
} cordata;

static void *open_cor_read(const char *path, const char *filetype, 
    int *natoms) {
  FILE *fd;
  cordata *cor;
  assert(path);
  
  if (!(fd = open_cor_file(path, natoms))) {
    fprintf(stderr, "Couldn't open cor file %s\n", path);
    return NULL;
  } 
  cor = (cordata *) malloc(sizeof(cordata));
  memset(cor, 0, sizeof(cordata));
  cor->numatoms = *natoms;
  cor->file = fd;
  return cor;
}

static int read_cor_structure(void *v, int *optflags, molfile_atom_t *atoms) {
  cordata *data = (cordata *)v;
  int i;
  assert(data);
  assert(data->file);
  
  /* we don't read any optional data */
  *optflags = MOLFILE_NOOPTIONS;

  for (i=0; i<data->numatoms; i++) {
    molfile_atom_t *atom = atoms+i; 
    if (get_cor_atom(data->file, atom->name, atom->type, 
                     atom->resname, atom->segid, 
                     &atom->resid) < 0) {
      fprintf(stderr, "couldn't read atom %d\n", i);
      return MOLFILE_ERROR;
    }
    atom->chain[0] = atom->segid[0];
    atom->chain[1] = '\0';
  }

  rewind(data->file);
  return MOLFILE_SUCCESS;
}

static int read_cor_timestep(void *v, int natoms, molfile_timestep_t *ts) {
  cordata *cor = (cordata *)v;
  char inbuf[COR_RECORD_LENGTH+2], header[6];
  char xStr[11], yStr[11], zStr[11];
  int i;

  xStr[10] = '\0';
  yStr[10] = '\0';
  zStr[10] = '\0';

  /* Skip the header */
  do {
    if (feof(cor->file) || ferror(cor->file)) {
      return MOLFILE_ERROR;
    }

    if (fgets(inbuf, COR_RECORD_LENGTH+1, cor->file) == NULL) {
      return MOLFILE_ERROR;
    }

    if (sscanf(inbuf, " %5c", &header) != 1) {
      return MOLFILE_ERROR;
    }

  } while (header[0]=='*');


  /* read the coordinates */
  for (i = 0; i < natoms; i++) {
    if (feof(cor->file) || ferror(cor->file)) {
      return MOLFILE_ERROR;
    }

    if (fgets(inbuf, COR_RECORD_LENGTH+1, cor->file) == NULL) {
      return MOLFILE_ERROR;
    }
    
    if (sscanf(inbuf, "%*5c%*5c%*5c%*5c%10c%10c%10c%*5c", 
               xStr, yStr, zStr) != 3) {
      return MOLFILE_ERROR;
    }
    else if (ts != NULL) {
      /* We have a timestep -- save the coordinates */
      ts->coords[3*i  ] = atof(xStr);
      ts->coords[3*i+1] = atof(yStr);
      ts->coords[3*i+2] = atof(zStr);
    }
  }

  return MOLFILE_SUCCESS;
}

static void close_cor_read(void *mydata) {
  cordata *data = (cordata *)mydata;
  if (data) {
    if (data->file) fclose(data->file);
    free(data);
  }
}  

/*
 * Initialization stuff down here
 */

static molfile_plugin_t plugin = {
  vmdplugin_ABIVERSION,   /* ABI version */
  MOLFILE_PLUGIN_TYPE,    /* type */
  "cor",                  /* name */
  "Eamon Caddigan",       /* author */
  0,                      /* major version */
  1,                      /* minor version */
  VMDPLUGIN_THREADSAFE,   /* is_reentrant  */
  "cor",                  /* filename extension */
  open_cor_read,          /* open_file_read     */
  read_cor_structure,     /* read_structure     */
  0,                      /* read bond list     */
  read_cor_timestep,      /* read_next_timestep */
  close_cor_read          /* close_file_read    */
};

int VMDPLUGIN_init() {
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_register(void *v, vmdplugin_register_cb cb) {
  (*cb)(v, (vmdplugin_t *)&plugin);
  return VMDPLUGIN_SUCCESS;
}

int VMDPLUGIN_fini() {
  return VMDPLUGIN_SUCCESS;
}
