#define Regular 0
#define PMF     1

class VolMap {
public:
  char *refname;   // name to call this map in stdout
    
  double origin[3];        /// origin of volume (x=0, y=0, z=0 corner)
  double xaxis[3];         /// direction and length for X axis (non-unit)
  double yaxis[3];         /// direction and length for Y axis (non-unit)
  double zaxis[3];         /// direction and length for Z axis (non-unit)
  double xdelta[3];        /// x axis of unit cell
  double ydelta[3];        /// y axis of unit cell
  double zdelta[3];        /// z axis of unit cell
  int xsize, ysize, zsize; /// number of samples along each axis
  double *data;             /// raw data, total of xsize*ysize*zsize voxels
  
  double weight;
  
  /* INITIALIZATION */    
       
  VolMap();
  ~VolMap();
  VolMap(const VolMap *src);

  void clone(const VolMap *src);
  void zero();
    

/* UNARY OPERATIONS */
  
  void convert_pmf_to_density();
  void convert_density_to_pmf();
  
  void trim(int trimxm, int trimxp, int trimym, int trimyp, int trimzm, int trimzp);
  void downsample(int ops=Regular);
  void collapse_onto_z(int ops=Regular);
  
  void average_over_rotations(int optype=Regular);

  void smooth(double radius, int optype=Regular);
  
  void total_occupancy();
  void print_stats();
     
  
  
  /* BINARY OPERATIONS */
  
  void init_from_intersection(VolMap *mapA, VolMap *mapB);
  void init_from_union(VolMap *mapA, VolMap *mapB);
  void init_from_identity(VolMap *mapA);
    
  void chop_B_from_A(VolMap *mapA, VolMap *mapB);
      
  void create_sum(VolMap *mapA, VolMap *mapB);
  void create_multiply(VolMap *mapA, VolMap *mapB);
  void create_diff(VolMap *mapA, VolMap *mapB);
  
  void create_combine(char **pmffiles, int numfiles,int ops=Regular);
  
  void create_mask(VolMap *mapA, VolMap *mapB);
  
  
  
  /* VOXELS */   
  
  /// return index of voxel in map, from x/y/z indices
  inline int voxel_index(int gx, int gy, int gz) const {
    return (gx + gy*xsize + gz*ysize*xsize);
  }
    
  /// return voxel at requested index, no safety checks
  inline double voxel_value(int gx, int gy, int gz) const {
    return data[gx + gy*xsize + gz*ysize*xsize];
  }

  /// return voxel, after safely clamping index to valid range
  double voxel_value_safe(int x, int y, int z) const;
  
  /// return interpolated value from 8 nearest neighbor voxels
  double voxel_value_interpolate(double xv, double yv, double zv) const;
  double voxel_value_interpolate_pmf_exp(double xv, double yv, double zv) const;
    
  int   coord_to_index(double x, double y, double z) const;
  void  index_to_coord(int index, double &x, double &y, double &z) const;
  double voxel_value_from_coord(double xpos, double ypos, double zpos) const;
  
  double voxel_value_interpolate_from_coord(double xpos, double ypos, double zpos, int ops=Regular) const;
  
  
  /* INPUT/OUTPUT (in io.C) */
  
  int   read_dx_file (const char *filename);
  int   write_dx_file (const char *filename) const;
  
};



