
# This package provides a path to a file containing the default CHARMM parameters
package require readcharmmpar


namespace eval PMFUtils {
  set charmm27params $env(CHARMMPARDIR)/par_all27_prot_lipid_na.inp
}

##################################################################################
#
#  The following procs are useful for aligning molecules and trajectories
#
##################################################################################


# ALIGNMOL: This is a quick molecule alignment routine. It aligns frame 0 
# of <moldest> to match frame 0 of <molsrc>, using the best fit obtained by 
# fitting the atoms contained in <seltext> in both molecules <moldest> and 
# <molsrc> (suggested seltext="alpha" for fitting to alpha carbons).
#
# Example: alignmol 0 1 "alpha and chain A"

proc PMFUtils::alignmol {molsrc moldest seltext} {
  puts "alignmol: Aligning frame 0 of molID $molsrc to that of molID $moldest."
	set align2 [atomselect $molsrc $seltext frame 0]
  set align1 [atomselect $moldest $seltext frame 0]
	[atomselect $molsrc all frame 0] move [measure fit $align2 $align1]
}


# ALIGNTRAJ: This is a quick trajectory alignment routine. It aligns all the 
# frames of molecule <molid> to frame 0 of that molecule, using the best fit 
# obtained by fitting the atoms in <seltext> (suggested seltext="alpha" for
# fitting to alpha carbons).
#
# Example: aligntraj 0 "alpha and chain A"

proc PMFUtils::aligntraj { molid alignseltext} {
  puts "aligntraj: Aligning frames of molID $molid relative to frame 0."
  set alignsel [atomselect $molid $alignseltext frame 0]
  set dcdsel   [atomselect $molid $alignseltext frame 0]
  set dcdall   [atomselect $molid all frame 0]
  
  set numframes [molinfo $molid get numframes]
  for {set frame 1} {$frame < $numframes} {incr frame} {
    $dcdsel frame $frame
    $dcdall frame $frame
    $dcdall move [measure fit $dcdsel $alignsel]
  }
}


# ALIGNTRAJ: This is a quick trajectory alignment routine. It aligns all the 
# frames of molecule <molid> such that the z-coordinate of the center of mass 
# of the atom selection <seltext> is shifted to be zero. This is useful for
# aligning lipid membranes instead of proteins.
#
# Example: centertraj_z 0 "lipids"

proc PMFUtils::centertraj_z {molid alignseltext} {
  puts "centertraj: Centering z for all frames of molID $molid."
  set dcdsel   [atomselect $molid $alignseltext frame 0]
  set dcdall   [atomselect $molid all frame 0]
  
  set numframes [molinfo $molid get numframes]
  for {set frame 0} {$frame < $numframes} {incr frame} {
    $dcdsel frame $frame
    $dcdall frame $frame
    set center [measure center $dcdsel]
    $dcdall moveby [list 0 0 [expr -[lindex $center 2]]]
  }
}




##################################################################################
#
#  The following procs are used to get VDW parameters from a list of charmm params 
#  files, for use with the ligand and slow ligand volmap map types.
#
##################################################################################


# READCHARMMPARAMS: This routine parses all the CHARMM NONBONDED (van der Waals/
# Lennard-Jones) parameters from a list of CHARMM parameter files.
#
# Example: readcharmmparams $PMFUtils::defaultcharmmparams customparams1.par customparams2.par

proc PMFUtils::readcharmmparams {args} {
  global nonbonded_table nonbonded_wildcard_list
  set nonbonded_list {}
  set nonbonded_wildcard_list {}
  
  if ![llength $args] {
    set args [list [file join $env(CHARMMPARDIR) par_all27_prot_lipid_na.inp]]
  }
  
  foreach parfile $args {
    set file [open $parfile "r"]

    #Find start of NONBONDED section
    while {[gets $file line] >= 0} {
      if {[lindex [split $line] 0] == "NONBONDED"} {break}
    }
  
    #Read NONBONDED params
    while {[gets $file line] >= 0} {
      if {[lindex [split $line] 0] == "HBOND"} break
      if {[lindex [split $line] 0] == "END"} break
      if {[lindex [split $line] 0] == "BONDS"} break
      if {[lindex [split $line] 0] == "IMPROPER"} break
      if {[lindex [split $line] 0] == "ANGLES"} break
      if {[lindex [split $line] 0] == "DIHEDRALS"} break
            
      if {[scan $line "%s %*f %f %f" type epsilon rmin] == 3} {
        if {[string index $line 0] == "!"} {
          set type [string range $type 1 end]
          if [string match "*\[%/*]*" $type] {
            set replaceindex [string first "%" $type]
            if {$replaceindex >= 0} {set type [string replace $type $replaceindex $replaceindex "?"]}
            #puts "WILDCARD $type $epsilon $rmin"
            set nonbonded_wildcard_list [linsert $nonbonded_wildcard_list 0 "$epsilon $rmin"]
            set nonbonded_wildcard_list [linsert $nonbonded_wildcard_list 0 $type]
          }
        } else {
          #puts "$type $epsilon $rmin"
          lappend nonbonded_list $type
          lappend nonbonded_list "$epsilon $rmin"
        }
      }
    }
  
    close $file
  }

  array unset nonbonded_table
  array unset nonbonded_wc_table
  array set nonbonded_table $nonbonded_list
  
  #puts  $nonbonded_wildcard_list

}



# ASSIGNCHARMMPARAMS: This routine sets the beta and occupancy fields of every atom
# in the specified molecule to what is needed forrunning the implicit ligand sampling 
# analysis. "readcharmmparams" must have been previously run on the desired parameter 
# files.
#
# Example: assigncharmmparams 0

proc PMFUtils::assigncharmmparams {{molid top}} {
  global nonbonded_table nonbonded_wildcard_list
  set atomtypes [[atomselect $molid all] get type]

  set atomradii {}
  set atomepsilon {}
  set atomnotfound {}
  
  foreach type $atomtypes {
    if [catch {
      lappend atomradii [lindex $nonbonded_table($type) 1]
      lappend atomepsilon [lindex $nonbonded_table($type) 0]
    }] {
      set foundmatch false 
      foreach {pattern data} $nonbonded_wildcard_list {
        if [string match $pattern $type] {
          lappend atomradii [lindex $data 1]
          lappend atomepsilon [lindex $data 0]
          set foundmatch true
          break
        }
      }
      
      if !$foundmatch {
        lappend atomradii 0.
        lappend atomepsilon 0.
        lappend atomnotfound $type
      }
    } 
  }
  
  if {[llength $atomnotfound] > 0} {
     set atomnotfound [lsort -unique $atomnotfound]
     foreach type $atomnotfound {
       puts "Could not find parameters for atom type $type. Missing parameters!"
     }
  }
  
  [atomselect $molid all] set radius $atomradii
  [atomselect $molid all] set occupancy $atomepsilon
}




# LOADFILES: Loads multiple files based on a glob pattern (i.e. same 
# as a UNIX shell), calling "mol addfile" for every file. 
# Every element after "--" is considered to be an argument to "mol addfile"
# e.g.: loadfiles file1.psf file*.dcd -- step 10 
# will call:
#   mol addfile file1.psf step 10
#   mol addfile file1.dcd step 10
#   mol addfile file2.dcd step 10
#   mol addfile file3.dcd step 10
#   etc...
#
# e.g.: loadfiles file1.psf file\[2-4\].dcd
# will load: file1.psf file2.dcd file3.dcd file4.dcd


proc PMFUtils::loadfiles {args} {
  set arg_divider [lsearch $args "--"]
  if {$arg_divider < 0} {
    set patternlist $args
    set arglist {}
  } else {
    set patternlist [lrange $args 0 [expr $arg_divider-1]]
    set arglist [lrange $args [expr $arg_divider+1] end]
    puts $arglist
  }
  
  set filelist {}
  foreach pattern $patternlist {
    set newfiles [lsort [glob -nocomplain $pattern]]
    puts "Found [llength $newfiles] files for: $pattern"
    eval lappend filelist $newfiles
  }
  
  foreach file $filelist {
    puts "mol addfile $file $arglist"
    eval mol addfile $file $arglist
  }
}


