$VERSION = "1.00";
package TCB::System::Room;
our $VERSION = "1.00";

# -*- Perl -*- 		Tue May 11 13:14:10 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Room - TCB Group Room Database

=head1 SYNOPSIS

  use TCB::System::Room;

See B<TCB::System::Misc> or B<TCB::System> for more information on how to use
this module.

=head1 DESCRIPTION

The 'Room' table lists possible rooms for various items to be located in
for the TCB system database.  It is a part of the 'TCB::System::Misc'
package, which is used by the rest of TCB::System to offer shared tables
of information.

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Number	Room number within the building.  (INT)
  Phone		Phone number of the room.
  Building	Name of the building the room is un.
  RoomType	Room type - machine room, office, cubicle, etc.
 
Key fields: 	Building, Number

List items: 	Address (Number + Building)

Required: 	Same as Key fields.

Default order: 	Building, Number

Admin Fields:	None

The 'Machine' and 'Admins' tables depend on this table directly.

Depends on the 'Building' table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT @TYPE @BUILDING );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Room";
$FIELDS = {
  'ID'          =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Number'      =>  'INT NOT NULL',  'Phone' => 'TINYTEXT',
  'Building'    =>  'TINYTEXT',    'RoomType' => 'TINYTEXT'
         };
$KEYS  = [ 'Building', 'Number' ];
$LIST  = [ { 'Address'       => '$$Number$$ $$Building$$' } ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ 'Building', 'Number' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	 $entry ||= {};

  my @buildings = (sort $self->select_fieldlist('Building', 'Name') );
  if (lc $type eq 'search') { unshift @buildings, '' };

  my @return = <<HTML;
<div class="basetable">
 <div class="row3">
  <span class="label">Room Number</span>
  <span class="formw"> @{[ $cgi->textfield('Number',
				$$entry{Number} || "", 6, 255) ]} </span>
  <span class="label">Building</span>
  <span class="formw"> 
    @{[ $cgi->popup_menu('Building', \@buildings, $$entry{Building} || "") ]}
  </span>
  <span class="label">Phone Number</span>
  <span class="formw"> @{[ $cgi->textfield('Phone',
				$$entry{Phone} || "", 12, 255) ]} </span>
 <div class="row1">
  <span class="label">Room Type</span>
  <span class="formw"> @{[ $cgi->textfield('RoomType',
				$$entry{RoomType} || "", 50, 255) ]} </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This table was originally put together to try to make an automatic floor
plan.  We never actually did this, which means that we never actually used
the table very much either.  Still, it turned out to be handy occasionally
in other contexts, so we haven't put it into the 'Obsolete' category.

The tables themselves were designed by the author, Tim Skirvin.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

It would have been better if Number hadn't been an INT.  

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::System::Misc>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-system>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-system/license.html> for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Tue May 11 13:14:10 CDT 2004 
### Ready to be a part of DBIx::Frame.
