$VERSION = "1.00";
package TCB::System::OS;
our $VERSION = "1.00";

# -*- Perl -*- 		Tue May 11 13:14:10 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::OS - TCB Group OS Database

=head1 SYNOPSIS

  use TCB::System::OS;

See B<TCB::System::Misc> or B<TCB::System> for more information on how to use
this module.

=head1 DESCRIPTION

The 'OS' table lists the operating systems in use in the TCB Group.  It is
a part of the 'TCB::System::Misc' package, which is used by the rest of
TCB::System to offer shared tables of information.

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Name		Name of the Operating System
  UsrLocal	/usr/local tree used by this OS.  Taken from UsrLocal 
		  table.
 
Key fields: 	Name

List items: 	Name, UsrLocal

Required: 	Same as Key fields.

Default order: 	Name

Admin Fields:	None

The 'Package', 'License', and 'Installed' tables depend on this table
directly.

Depends on the 'UsrLocal' table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT @TYPE @BUILDING );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "OS";
$FIELDS = {
  'ID'   => 'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Name' => 'TINYTEXT',      'UsrLocal' => 'TINYTEXT',
         };
$KEYS  = [ 'Name' ];
$LIST  = [ 'Name', 'UsrLocal' ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ 'Name' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};

  my @usrlocal =  ( sort $self->select_fieldlist('UsrLocal', 'Name') );
  unshift @usrlocal, 'None';
  if (lc $type eq 'search') { unshift @usrlocal, '' };

  my @return = <<HTML;
<div class="basetable">
 <div class="row2">
  <span class="label">Name</span>
  <span class="formw"> @{[ $cgi->textfield('Name',
				$$entry{Name} || "", 30, 255) ]} </span>
  <span class="label">/usr/local tree</span>
  <span class="formw"> 
    @{[ $cgi->popup_menu('UsrLocal', \@usrlocal, $$entry{UsrLocal} || "") ]} 
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This is a table mostly used by other tables; most users therefore won't
make it very interesting.

The tables themselves were designed by the author, Tim Skirvin.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::System::Misc>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-system>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-system/license.html> for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Tue May 11 13:14:10 CDT 2004 
### Ready to be a part of DBIx::Frame.
