$VERSION = "0.95";
package TCB::Seminar::Lecture;
our $VERSION = "0.95";

# -*- Perl -*- 		Fri May 21 10:01:23 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2002-2004, Tim Skirvin and UIUC Board of Trustees.
###############################################################################

=head1 NAME

TCB::Seminar::Lecture - lectures table for TCB::Seminar

=head1 SYNOPSIS

  use TCB::Seminar::Lecture;

See TCB::Seminar for more information.

=head1 DESCRIPTION

The 'Lecture' table is the heart of the seminar database; it contains
information on the actual lecture from the seminar, including speaker,
topic, abstract, times, and so forth.  Most of this information is shared
publicly.

This table contains the following fields:
  
 Internal Information (TINYTEXT fields, unless noted)
  ID            Unique numeric ID - auto-generated (INT)
  CreatedBy     Username of person who created entry     \   Auto-created using 
  CreateDate    Date that this entry was created (DATE)   \  $ENV{REMOTE_USER} 
  ModifiedBy    Username who last modified this entry     /  and current time
  ModifyDate    Date that this entry was modified (DATE) /   in html() 

 Speaker Information (TINYTEXT fields, unless noted)
  Name		Speaker's full name   
  SpeakTitle    Speaker title (eg Professor, Doctor, etc)
  Affiliation   Speaker's affiliation (eg University of Illinois)
  AffilDept     Department of the affiliation (eg Beckman Institute)
  AffilLoc      Location of the affiliation (eg Urbana, Illinois)
  ExtraLine     Any extra information that should be printed

 Lecture Information (TINYTEXT fields, unless noted)
  Title		Title of the lecture
  Location      Location of the lecture (building + room)
  Date          Date of the lecture (DATE)
  Time		Time of the lecture (TIME)
  Hosts		People that are hosting the lecturer
  Status 	Abnormal lecture status - informal, cancelled, etc
  Abstract      The abstract of the lecture  (TEXT)
  ExtraInfo	Any extra information that should be printed (TEXT)

 Private Information (TEXT fields, unless otherwise stated)
  ContactInfo	Contact information about the speaker
  Reimbursements What reimbursements are outstanding/completed?
  Agenda	Speaker agenda  
  HotelReserve  Hotel reservations
  AirlineReserve Airline reservations
  GroundReserve Ground transport reservations (rental cars, etc)
  OtherReserve  Any other reservations
  Notes         Other notes

Key fields:     Name, Title, Date

List items:     Date (converted), Name, Affiliation (full), Title

Required:       Name, Title, Date

Default order:  Date (reverse), Name

Admin Fields:   None

The 'CheckList' table depends on this table.

Doesn't depend on any other table.

=head2 Variables

The following variables are set in the module, and are used as defaults
for creation and searching.

=over 4

=item @TCB::Seminar::Lecture::STATUS

Possible status types for the seminar.  Default possibilities: Cancelled,
Postponed, or Informal.

=back

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $ORDER $ADMIN $REQUIRED @STATUS ); 

use strict;
use warnings;
use CGI;
use TCB::Seminar;
use Time::Local;

unshift @ISA, "TCB::Seminar";

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Lecture";	
$FIELDS = {
  # Non User-Modified Information
  'ID'          => 'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'CreatedBy'   => 'TINYTEXT',  'ModifiedBy'    => 'TINYTEXT',
  'CreateDate'  => 'DATE',      'ModifyDate'    => 'DATE',

  # Speaker Information - public info #1
  'Name'        => 'TINYTEXT',     'SpeakTitle'  => 'TINYTEXT',
  'Affiliation' => 'TINYTEXT', 	   'AffilDept'   => 'TINYTEXT',	   
  'AffilLoc'    => 'TINYTEXT',     'ExtraLine'   => 'TINYTEXT',

  # Lecture Information - public info #2
  'Title'   	=> 'TINYTEXT',     'Location' 	 => 'TINYTEXT',     
  'Date'        => 'DATE', 	   'Time'        => 'TIME',
  'Hosts'       => 'TINYTEXT', 	   'Status'      => 'TINYTEXT',
  'Abstract'  	=> 'TEXT', 	   'ExtraInfo'   => 'TEXT',	   

  # Freeform Info - the stuff that will be private
  'ContactInfo'    => 'TEXT',	   'Reimbursements' => 'TEXT',
  'Agenda'         => 'TEXT',	   'HotelReserve'  => 'TEXT',
  'AirlineReserve' => 'TEXT', 	   'GroundReserve' => 'TEXT',
  'OtherReserve'   => 'TEXT',	   'Notes'         => 'TEXT',

	 };
$KEYS  = [ 'Name', 'Title', 'Date' ];
$LIST  = [ { 'Date' => [ \&xlatedateshort, '$$Date: 2004/05/21 18:25:49 $$' ] },       'Name', 
	   { 'Affiliation' => '$$Affiliation$$, $$AffilLoc$$' }, 'Title' ];
$ORDER = [ '-Date', 'Name' ];
$REQUIRED = [ 'Name', 'Title', 'Date' ];
$ADMIN = [];  
@STATUS = ( '', 'Cancelled', 'Postponed', 'Informal' );

###############################################################################
### Functions #################################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;  $entry ||= {};  $options ||= {};
  my $admin = $$options{'admin'} ? 1 : 0;

  if (lc $type eq 'create') {
    $$entry{'CreatedBy'}  ||= $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'CreateDate'} ||= $self->_date_mysql(time);
    $$entry{'Date'}       ||= $self->_date_mysql(time);
    $$entry{'Time'}       ||= "15:00:00";
    $$entry{'Location'}   ||= "3269 Beckman Institute";
    $$entry{'ExtraInfo'}  ||= 
	"Tea and coffee will be served in R3151 Beckman Institute at 2:15pm.";
    $$entry{'Status'}     ||= '';
  } elsif (lc $type eq 'search') {
  } elsif (lc $type eq 'edit' || lc $type eq 'update') {
    $$entry{'ModifiedBy'} = $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'ModifyDate'} = $self->_date_mysql(time);
  } else { }

  my %trips;
  foreach my $entry ($self->select('Trip', {})) { 
    my $id = $$entry{ID} || next;
    $trips{$id} = $self->trip_name($entry) || next;
  }

  my @status = @STATUS;

  push @status, '' if (lc $type eq 'search');

  my @return = <<HTML;
<div class="basetable">
 <h3 class="tablehead"> Speaker Information </h3>

 <div class="row2">
  <span class="label">Name</span>
  <span class="formw">
   @{[ $cgi->textfield('Name', $$entry{Name} || "", 30, 255) ]}
  </span>
  <span class="label">Title</span>
  <span class="formw">
   @{[ $cgi->textfield('SpeakTitle', $$entry{SpeakTitle} || "", 15, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Department</span>
  <span class="formw">
   @{[ $cgi->textfield('AffilDept', $$entry{AffilDept} || "", 70, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Affiliation</span>
  <span class="formw">
   @{[ $cgi->textfield('Affiliation', $$entry{Affiliation} || "", 70, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Inst Location (City, State) </span>
  <span class="formw">
   @{[ $cgi->textfield('AffilLoc', $$entry{AffilLoc} || "", 70, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Extra Info</span>
  <span class="formw">
   @{[ $cgi->textfield('ExtraLine', $$entry{ExtraLine} || "", 70, 255) ]}
  </span>
 </div>

 <p> &nbsp; </p>

 <h3 class="tablehead"> Lecture Information </h3>

 <div class="row1">
  <span class="label">Title (Book Caps)</span>
  <span class="formw">
   @{[ $cgi->textfield('Title', $$entry{Title} || "", 70, 255) ]}
  </span>
 </div>

 <div class="row3">
  <span class="label">Location</span>
  <span class="formw">
   @{[ $cgi->textfield('Location', $$entry{Location}, 25, 255) ]}
  </span>
  <span class="label">Date</span>
  <span class="formw">
   @{[ $cgi->textfield('Date', $$entry{Date} || "", 11, 10)]} 
  </span>
  <span class="label">Time</span>
  <span class="formw">
   @{[ $cgi->textfield('Time', $$entry{Time} || "", 9, 8)]} 
  </span>
 </div>

 <div class="row2">
  <span class="label">Hosts</span>
  <span class="formw">
   @{[ $cgi->textfield('Hosts', $$entry{Hosts} || "", 30, 255) ]}
  </span>
  <span class="label">Status (if unusual)</span>
  <span class="formw">
   @{[ $cgi->popup_menu('Status', \@status, $$entry{Status}) || "" ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Abstract</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Abstract', -default=>$$entry{Abstract} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Extra Info</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'ExtraInfo', -default=>$$entry{ExtraInfo} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <p> &nbsp; </p>
 
 <h3 class="tablehead"> Internal Information for Seminar Preparation </h3>

 <div class="row1">
  <span class="label">Contact Information</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'ContactInfo', 
			-default=>$$entry{ContactInfo} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Agenda</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Agenda', -default=>$$entry{Agenda} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Airline Reservations & Costs</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'AirlineReserve',
			-default=>$$entry{AirlineReserve} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Hotel Reservations & Costs</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'HotelReserve', 
			-default=>$$entry{HotelReserve} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Ground Transport Reservattions & Costs</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'GroundReserve',
			-default=>$$entry{GroundReserve} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Other Reservations & Costs</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'OtherReserve', 
			-default=>$$entry{OtherReserve} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Reimbursements</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Reimburse', -default=>$$entry{Reimburse} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">General/Other Information</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "",
                        -rows=>4, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row2">
  <div class="modbox">
   @{[ $self->created_html( $$entry{CreateDate} || "", 
                            $$entry{ModifyDate} || "",
                            $$entry{CreatedBy}  || "", 
                            $$entry{ModifiedBy} || "", !$admin ) ]}
  </div>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

sub text { }

=item xlate ( DATAHASH )

xlate() takes a datahash (such as the output of a DBIx::Frame->select()
command) containing Lecture information and translates it into a
single-line summary.  This is distinct from the list() in as much as it is
only one string, not an array of information.  Returns "" if no
B<DATAHASH> is passed.

=cut

sub xlate {
  my ($self, $hash) = @_;
  return "" unless ($hash && ref $hash);
  my $speaker = $$hash{Name} || "";
  my $aff     = $$hash{Affiliation} || "";
  my $loc     = $$hash{AffilLoc} || "";
  my $title   = $$hash{Title} || "";
  my $url     = "seminar.cgi?ID=$$hash{ID}";
  my $status  = $$hash{Status} ? "-- $$hash{Status}" : "";
  sprintf("<b>%s</b>,\n%s<br />\n%s\n%s", 
	$speaker, join(", ", $aff, $loc), "<a href='$url'>$title</a>", $status);
}

=item xlatedate ( DATE ) 

Translates the B<DATE> string (from a MySQL 'Date' entry) into a
human-readable string. 

=cut

sub xlatedate {
  my ($self, $date) = @_;  return "" unless $date;
  my ($year, $month, $day) = split('-', $date);
  my @MONTH = qw( January February March April May June July August 
		  September October November December );
  my @DAYS  = qw( Sunday Monday Tuesday Wednesday Thursday Friday Saturday );
  my $time = timelocal( 12, 12, 12, $day, $month - 1, $year );
  sprintf("%s, %s %2d, %d", $DAYS[(localtime($time))[6]], 
					$MONTH[$month-1], $day, $year);
}

=item xlatedateshort ( DATE ) 

Just like C<xlate()>, except that the string is abbreviated ('Mon' instead
of 'Monday', 'Jan' instead of 'January', etc).

=cut

sub xlatedateshort {
  my ($self, $date) = @_;  return "" unless $date;
  return "(not set)" if ($date eq '0000-00-00');
  my ($year, $month, $day) = split('-', $date);
  my @MONTH = qw( Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec );
  my @DAYS  = qw( Sun Mon Tue Wed Thu Fri Sat );
  my $time = timelocal( 12, 12, 12, $day, $month - 1, $year );
  sprintf("%s, %s %2d, %d", $DAYS[(localtime($time))[6]], 
					$MONTH[$month-1], $day, $year);
}

=item xlatetime ( TIME )

Translates the B<TIME> string (from a MySQL 'Time' entry) into a
human-readable string. 

=cut

sub xlatetime { 
  my ($self, $time) = @_;  return "" unless $time;
  my ($hour, $minute, $second) = split(':', $time);
  my $fixhour = $hour % 12;  $fixhour = 12 if ($fixhour eq '0');
  sprintf("%2d:%02d %s", $hour % 12, $minute, $hour > 12 ? "pm" : "am");
}

=item public_seminar ( DATAHASH )

Takes B<DATAHASH> and returns a formatted HTML page containing all of the
public information of the associated data.  Meant for publicly showing off
seminar information; used by B<seminar.cgi>.

=cut

sub public_seminar {
  my ($self, $data) = @_;
  my @return;
  push @return, "<div class=\"seminar\">";
  push @return, $$data{Title} ? "<h2 class=\"title\">$$data{Title}</h2>"
                              : "<h2 class=\"title\">Unknown Title</h2>";

  # Speaker Information
  push @return, "<p class=\"speaker\">";
  push @return, $$data{SpeakTitle} ? "$$data{SpeakTitle} " : "";
  push @return, $$data{Name} ? "$$data{Name}" : "Unknown Speaker", "<br />";
  push @return, $$data{AffilDept} ? "$$data{AffilDept}<br />" : "";
  push @return, $$data{Affiliation} ? "$$data{Affiliation}"
                                    : "Unknown Institution", "<br />";
  push @return, $$data{AffilLoc} ? "$$data{AffilLoc}"
                                 : "Unknown Location", "<br />";
  push @return, $$data{ExtraLine} ? $$data{ExtraLine} : "";
  push @return, "</p>";

  # Status of talk - cancelled?  Informal lecture?  Etc.
  push @return, "<p class=\"status\"> $$data{Status} </p>" if $$data{Status};

  # Date, Time, Location
  push @return, "<p class=\"timedate\">";
  push @return, $$data{Date} 
		? TCB::Seminar::Lecture->xlatedate($$data{Date}) . "<br />"
                : "Unknown Date <br />";
  push @return, $$data{Time}
        ? TCB::Seminar::Lecture->xlatetime($$data{Time}) . "<br />" : "";
  push @return, $$data{Location} ? "$$data{Location}<br />" : "<br />";
  push @return, "</p>";

  # Abstract
  push @return, "<h4 class=\"title\">Abstract</h4>", 
		"<p class==\"abstract\">$$data{Abstract}</p>" 
						     if ($$data{Abstract});
  # Extra Info
  push @return, "<hr />", "<p class=\"extrainfo\"> $$data{ExtraInfo} </p>" 
						     if ($$data{ExtraInfo});

  wantarray ? @return : join("\n", @return);
}

=back

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::Seminar->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
                        $REQUIRED, \&html, \&text);

=head1 NOTEES

=head1 TODO

=head1 REQUIREMENTS

Perl 5.6.0 or better, DBIx::Frame, TCB::Seminar.

=head1 SEE ALSO

B<TCB::Seminar>, B<DBIx::Frame>, B<seminar.cgi>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-seminar/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-seminar/license.html>
for details.

=head1 COPYRIGHT

Copyright 2002-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.1 	Fri Mar 22 14:09:27 CST 2002
### First functional version, not approved by anybody yet, this will likely 
### still need a lot of work.
# v0.2  Tue Aug  6 10:10:51 CDT 2002
### Added some more fields, and public_seminar(). 
# v0.9	Wed Aug 14 15:00:13 CDT 2002
### Getting ready for publishing.  Documentation complete.
# v0.91 Mon Oct  7 16:43:23 CDT 2002
### Slight fix with 'MODIFIED_BY'.
# v0.92 Thu Jan 16 16:46:18 CST 2003
### Updated to work with DBI::Frame 1.05.01 and better.  This means that
### ordering works and such (only ordering is currently being used.)
# v0.92.1	Tue Oct 14 10:00:46 CDT 2003 
### Fixed a small bug in xlatedateshort()
# v0.95		Fri May 21 10:01:23 CDT 2004 
### Updated for DBIx::Frame.  Getting ready for stylesheets.
