package TCB::Seminar::Functions;
# -*- Perl -*- Wed Aug 14 13:53:51 CDT 2002
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2002, Tim Skirvin and UIUC Board of Trustees.
###############################################################################

use vars qw( $VERSION );  
$VERSION = 0.9;

package TCB::Seminar;

=head1 NAME

TCB::Seminar::Functions - additional functions for TCB::Seminar

=head1 SYNOPSIS

  use TCB::Seminar::Functions;

See TCB::Seminar for general information on this database.

=head1 DESCRIPTION

TCB::Seminar::Functions contains the functions specific to TCB::Seminar.
These are a combination of specific functions for each table and general
extensions to DBIx::Frame.  

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

use TCB::Seminar;
use vars qw( @EXPORT_OK @EXPORT );
use strict;
use CGI;

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Functions

The following methods are available within this object, and are exported
by default:

=over 4

=item seminar_name ( ENTRY )

Returns a formatted version of the seminar's "name" - the date, speaker
name, and speaker affiliation.  Uses B<ENTRY>, which is a hash reference
(as returned by DBIx::Frame).

=cut 

sub seminar_name {
  my ($self, $entry) = @_;
  sprintf("%s - %s, %s", $$entry{Date} || "UNKNOWN", 
			 $$entry{Name} || "UNKNOWN SPEAKER",
			 $$entry{Affiliation} || "NO AFFILIATION" );
}
push @EXPORT_OK, qw( seminar_name );

=item seminar_name_short ( ENTRY )

As with B<seminar_name()>, except just returns the date and the speaker.
Uses B<ENTRY>, which is a hash reference (as returned by DBIx::Frame).

=cut

sub seminar_name_short {
  my ($self, $entry) = @_;
  sprintf("%s - %s", $$entry{Date} || "UNKNOWN", 
			 $$entry{Name} || "UNKNOWN SPEAKER" );
}
push @EXPORT_OK, qw( seminar_name_short );

=item seminar_name_by_id ( SeminarID )

Returns the seminar's name by B<SeminarID> from the B<Seminar> table.  Uses
B<seminar_name()> and requires an active database connection.

=cut

sub seminar_name_by_id {
  my ($self, $id) = @_;
  return undef unless $id;
  foreach my $entry ($self->select( 'Lecture', { 'ID' => $id } )) {
    my $return = $self->seminar_name($entry);
    return $return if $return;
  }
  ""
}
push @EXPORT_OK, qw( seminar_name_by_id );

=item seminar_summary ( SeminarID )

Returns the number of CheckList items that have been completed for the
given SeminarID, out of the total number of items.  Requires a database
connection.

=cut

sub seminar_summary {
  my ($self, $id) = @_;
  return undef unless $id;

  my $fields = $self->fields( 'CheckList' );
  my ($done, $total, $updated);
  foreach my $entry ($self->select( 'CheckList', { 'ID' => $id } )) {
    foreach my $key (sort keys %{$fields}) { 
      next unless $$fields{$key} eq 'DATE';
      next if $key =~ /^(Modify|Create)Date$/;
      $done++ if $$entry{$key} && $$entry{$key} ne '0000-00-00';  
      $total++;
    } 
    last;
  }
  $done ||= 0;
  $total ? "$done/$total done" : "No matching checklist";
}
push @EXPORT_OK, qw( seminar_summary );

=item dates ( DATE1, DATE2 )

Returns the following string, based on whether DATE1 and DATE2 are real
dates (ie not "0000-00-00", the default NULL for mysql's date field):

  DATE1	  DATE2	       Returned String
    Y	    Y		DATE1 - DATE2
    Y       N		DATE1 - 
    N 	    Y		- DATE2
    N       N           No Date

Also, if DATE1 and DATE2 are the same, it just returns DATE1 (the entire
event was on that day).  

=cut

sub dates {
  my ($self, $date1, $date2) = @_;
  map { s/0000-00-00/0/ } $date1, $date2;
  return "No Date" if !$date1 && !$date2;
  return "$date1" if $date1 eq $date2;
  $date1 ? $date2 ? "$date1 - $date2" : "$date1 - "
	 : $date2 ? "- $date2" : "No Date";
}
push @EXPORT_OK, qw( dates );

=item updated ( DATE1, DATE2 )

Takes B<DATE1> (meant to be the B<ModifyDate> field) and B<DATE2> (meant
to be the B<CreateDate> field), and returns a formatted date of last
update.  Used by B<TCB::Seminar::Checklist>'s list() function.

=cut

sub updated {
  my ($self, $date1, $date2) = @_;
  my $date = $date1 eq '0000-00-00' ? $date2 : $date1;
  my ($year, $month, $day) = split('-', $date);
  my @MONTHS = qw( Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec );
  my $monthname = $MONTHS[$month-1] || "UNKNOWN";
  "$day $monthname $year";
}
push @EXPORT_OK, qw( updated );

=item url ( URL, TEXT )

Returns the following string, based on the existance of URL and TEXT:

   URL	   TEXT	        Returned String
    Y	    Y		"<a href='URL'>TEXT</a>"
    Y       N           "<a href='URL'>URL</a>"
    N 	    Y		"TEXT"
    N       N		"No URL"

=cut

sub url {
  my ($self, $url, $text) = @_;
  $url ? $text ? "<a href='$url'>$text</a>" : "<a href='$url'>$url</a>"
       : $text ? $text : "No URL";
}
push @EXPORT_OK, qw( url );

=item created_html ( CreateDate, ModifyDate, CreatedBy, ModifiedBy, NOSHOW )

Returns the standard little table for Date and Person vs Created and
Modified, along with the 'Submit' button.  Used by both the B<Lecture> and
B<CheckList> tables, this may very well get moved into DBIx::Frame itself
B<NOSHOW>, if set, says "don't show them the table".

=cut

sub created_html {
  my ($self, $createdate, $modifydate, $createdby, $modifiedby, $noshow) = @_;
  $createdate ||= "";  $modifydate ||= "";
  $createdby  ||= "";  $modifiedby ||= "";
  my @return;
  push @return, CGI->hidden('CreateDate', $createdate),
     		CGI->hidden('ModifyDate', $modifydate),
     		CGI->hidden('CreatedBy',  $createdby ),
     		CGI->hidden('ModifiedBy', $modifiedby);
  return @return if $noshow;
  push @return, <<END;

     <table class="modbox" border>
      <tr> <th> </th> <th>Created</th> <th>Modified</th> </tr>
      <tr> <td>Date</td>
           <td align=center>@{[ $createdate ]}</td>
           <td align=center>@{[ $modifydate ]}</td> </tr>
      <tr> <td>Person</td>
           <td align=center>@{[ $createdby  ]}</td>
           <td align=center>@{[ $modifiedby ]}</td>  </tr>
     </table>

END
  @return;
}

### _date_mysql( [TIME] )
# Returns the date based on TIME for use with mysql.  
sub _date_mysql {
  my ($self, $time) = @_;  $time ||= time;
  sprintf("%04d-%02d-%02d", (localtime($time))[5] + 1900,
                        (localtime($time))[4] + 1, (localtime($time))[3]);
}

=back

=cut

1;

=head1 TODO

Probably want to add some better error handling functionality.

=head1 REQUIREMENTS

Perl 5.6.0 or better, DBIx::Frame, TCB::Seminar.

=head1 SEE ALSO

B<TCB::Seminar>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-seminar/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-seminar/license.html>
for details.

=head1 COPYRIGHT

Copyright 2002-2004 by the University of Illinois Board of Trustees and Tim
Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.1 	Thu Mar 28 13:13:00 CST 2002
### First functional version, not approved by anybody yet, this will likely 
### still need a lot of work.
# v0.9  Wed Aug 14 13:51:37 CDT 2002
### Getting ready for publishing.  Documentation complete.  There's now a
### separate TB::Seminar::Functions class so we can enter the version
### somewhere.
# v0.95	Fri May 21 11:07:21 CDT 2004 
### Updated for DBIx::Frame.  Getting ready for stylesheets.
