#!/usr/local/bin/perl -Tw
use vars qw( $DEBUG $CLASS $HTML $TITLE $HTMLHEAD $HTMLFOOT $VERSION );
$VERSION = "1.02a";

=head1 NAME

pubs.cgi - easy access to the publications database with SSI

=head1 SYNOPSIS
 
  <!--#include virtual="/cgi-bin/pubs.cgi?ZHU2002,JENS2002"-->   

=head1 DESCRIPTION

It is oftentimes helpful to be able to directly cite papers from within an
HTML file.  With this script, the TCB::Publications database, and SSI,
this is fairly easy; just include directives such as the above.

=head1 REQUIREMENTS

TCB::Publications, DBIx::Frame

=head1 SEE ALSO

B<TCB::Publications>

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Configuration + Private Data ##############################################
###############################################################################

## This is the perl class that you will be using in this script.  

$CLASS   = "TCB::Publications";			# Database class

## Modify and uncomment this to use user code instead of just system-wide
## modules.  Note that this path must be set up as a standard Perl tree;
## I'd personally recommend just installing things system-wide unless
## you're a developer.

# use lib '/PATH/TO/CODE';
# use lib '/home/tskirvin/dev/mdtools/tcb-publications';

## Document title - set this as appropriate.

$TITLE   = "Publications Database";		

## This variable records how much debugging information you want in the
## HTML footers.  It works similarly to Unix permissions, by OR-ing the 
## appropriate options:
## 
##   	1	Print CGI parameters
##      2	Print environment variables
##
## ie, '3' would print CGI and environment variables, and '1' would print 
## just CGI parameters.  '0' will print nothing.

$DEBUG   = 0;					

## These are references to code that will output the headers and footers
## for the messages.  If you want to change these, you can either modify
## the code (which is below) or create a new set of functions and change 
## the below code references appropriately.

$HTMLHEAD = \&html_head;	
$HTMLFOOT = \&html_foot;

###############################################################################
### main() ####################################################################
###############################################################################

use CGI;
use strict;
use TCB::TeX2HTML;

# Load the appropriate class module
{ local $@; eval "use $CLASS";  die "$@\n" if $@; }

$0 =~ s%.*/|\?.*%%g;		# Lose the annoying path information
$|++;				

my $cgi = new CGI || die "Couldn't open CGI";

my $params = {};
foreach ($cgi->param) { $$params{$_} = $cgi->param($_); }

my $hash = {};
my @codes = split(',', $$params{keywords});
Error("No 'keywords' offered") unless scalar @codes;

( print $cgi->header(), &$HTMLHEAD($TITLE), "\n" ) && $HTML++;
foreach (@codes) { 
  print join("\n", 
    '<p> ' . listbib($_, '/Publications/Papers/abstract.cgi') . ' </p>', 
  		'', '' );
}
print &$HTMLFOOT($DEBUG);
exit(0);

###############################################################################
### Subroutines ###############################################################
###############################################################################

## Error ( PROBLEM [, PROBLEM [...]] )
# Prints an error message based on PROBLEM and exits.

sub Error {
  print CGI->header(), &$HTMLHEAD("Error in '$0'") unless $HTML;

  print "This script failed for the following reasons: <p>\n<ul>\n";
  foreach (@_) { next unless $_; print "<li>", canon($_), "<br>\n"; }
  print "</ul>\n";

  print &$HTMLFOOT($DEBUG);
  exit 0;
}

## canon ( ITEM )
# Returns a printable version of whatever it's passed.  Used by Error().

sub canon {
  my $item = shift;
  if    ( ref($item) eq "ARRAY" )   { join(' ', @$item) }
  elsif ( ref($item) eq "HASH" )    { join(' ', %$item) }
  elsif ( ref($item) eq "" )        { $item }
  else                              { $item }
}

## html_head ( TITLE [, OPTIONS] )
# Prints off a basic HTML header, with debugging information.  Extra
# options are passed through to start_html.

sub html_head { 
  my $title = shift || $TITLE;
  my $cgi = new CGI;
  $cgi->start_html(-title => $title, @_) . "\n";
}

## html_foot ( DEBUG [, OPTIONS] )
# Prints off a basic HTML footer, with debugging information.

sub html_foot { 
  my $debug = shift || $DEBUG;
  my $cgi = new CGI;
  my @return = debuginfo($debug);
  push @return, $cgi->end_html(@_);
  join("\n", '', @return, "");
}

## debuginfo ( LEVEL ) 
# Takes care of printing debugging information, as described above

sub debuginfo {
  my $debug = shift || 0;

  my @return;

  if ($debug & 1) {
    push @return,  "Parameters: <p>\n<ul>\n";
    foreach ($cgi->param) { push @return,  " <li>$_: ", $cgi->param($_); }
    push @return,  "</ul>";
  }

  if ($debug & 2) {
    push @return,  "Environment Variables: <p>\n<ul>";
    foreach (sort keys %ENV) { push @return, " <li>$_: $ENV{$_}"; }
    push @return,  "</ul>";
  }

  wantarray ? @return : join("\n", @return);
}

sub listbib {
  my ($tbcode, $program) = @_;
  return "" unless $tbcode; $program ||= '';

  # Split out the title, so we don't print it twice (this algorithm was
  # from the old publications database; I don't really understand it, or
  # at least I haven't looked at it enough to really get it.)
  my ($author, $title, @rest) = split(/\\newblock/,
		      TCB::Publications::Papers->getbib($tbcode) || "" );

  $title ||= ""; $title =~ s/^\s+|\s+$//g;
  $title = tex2html($title);  

  my $rest = tex2html("$author @rest");

  # Trim whitespace appropriately
  map { s/\s+/ /g } $title, $rest;
  map { s/^\s+|\s+$//g } $title, $rest;
  return "" unless $title;

  my @return = ("<a href='$program?tbcode=$tbcode'>$title</a>", $rest);
  map { s/Damjanovi\\'c/Damjanovic/g } @return;   # Special case for Ana
  wantarray ? @return : join("\n", @return);
}

###############################################################################
### Version History ###########################################################
###############################################################################
# v1.00a	Tue Nov 26 15:47:43 CST 2002
### First real version done.  Fairly easy to use, I'll find a good place
### to put it.
# v1.01a	Tue Nov 26 16:07:39 CST 2002
### Took out a lot of the extraneous database information.
# v1.02		Mon May 17 11:44:29 CDT 2004 
### Commented this more reasonably, making it better for public release. 
