#!/usr/local/bin/perl -T
use vars qw( $DBHOST $DBTYPE $DATABASE $DBUSER $DBPASS $DEBUG $CLASS $DB $HTML 
             $OPTIONS $TITLE $HTMLHEAD $HTMLFOOT $VERSION $ROOTCLASS 
	     $TITLEFILE $INTROFILE $OUTROFILE $PAPERSFILE );
$VERSION = "1.5";

=head1 NAME

subject.cgi - List papers by paper number 

=head1 SYNOPSIS

  http://www.ks.uiuc.edu/Publications/Papers/SMD/

=head1 DESCRIPTION

subject.cgi is used to allow for easy lists of papers based on a single
subject, as decided by someone that knows what the papers are about, using
TCB::Publications.  Information is loaded from severan files in the same
directory as the script: 

  header	Header of the page - any text you want to print befoe
  title		Title of the page
  numbers	Paper numbers to load; one per line, number only. 
  footer	Footer of the page

Output is much like list.cgi.  This was designed to be used in a structure
like this:

  subject.cgi
  SMD/
    index.cgi -> ../subject.cgi
    title
    numbers
    header
  IMD/
    index.cgi -> ../subject.cgi
    title
    numbers
    header
    [...]

=head1 REQUIREMENTS

B<TCB::Publications>, B<CGI::SHTML>

=head1 SEE ALSO

B<TCB::Publications>, B<DBIx::Frame::CGI>

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### CONFIGURATION + PRIVATE DATA ##############################################
###############################################################################

## Load shared configurations and/or private data using 'do' commands, as
## seen below.  Note that several 'do's can be run if necessary.  
# do '/FULL/PATH/TO/CODE/TO/RUN';	
use TCB::Internal;			# Load tbhtml_* commands

## This is the perl class that you will be using in this script.  

$CLASS   = "TCB::Publications";			# Database class

## This is the root class of the above class.  Essentially a hack to let
## there be multiple modules using the same database.

$ROOTCLASS = "TCB::Publications";               # Class of the database class

## Modify and uncomment this to use user code instead of just system-wide
## modules.  Note that this path must be set up as a standard Perl tree;
## I'd personally recommend just installing things system-wide unless
## you're a developer.

# use lib '/PATH/TO/CODE';
use lib '/home/tskirvin/dev/mdtools/tcb-publications';

## Document title - set this as appropriate.

$TITLE   = "Publications Database";		

## Set these options to modify the behaviour of the HTML creation code.

$OPTIONS = {
        'admin'    => 0,        # Offer 'edit' and 'delete' functions?
        'nodetail' => 1,        # Don't offer 'view'?
        'nocount'  => 1,        # Don't use 'next 25 entries' buttons
        'count'    => 400,      # How many entries should we offer at a time?
        'tdopts' => 'align=left', # Align the table rows to the left
        'nomenu'   => 1,        # Don't offer the bottom menu
           };

## Database Information
## You may want to set these with a common config file, using 'do FILE'.
## Also, defaults may already be set within the class; only set these if
## you want to override the defaults.

# $DBHOST   = "";		# System that hosts the database
# $DBTYPE   = "";		# The type of database that we're working on
# $DATABASE = "";		# Name of the database we're connecting to
# $DBUSER   = "";		# Username to connect to the database
# $DBPASS   = "";		# Password to connect to the database
do '/home/webserver/dbaccess/user.pubdb';     # Populate DBUSER, DBPASS, etc

## This variable records how much debugging information you want in the
## HTML footers.  It works similarly to Unix permissions, by OR-ing the 
## appropriate options:
## 
##    	1	Print SQL queries
##   	2	Print CGI parameters
##      4	Print environment variables
##
## ie, '6' would print CGI and environment variables, and '5' would print 
## environment variables and SQL queries.  '0' will print nothing.

$DEBUG   = 0;					

## These are references to code that will output the headers and footers
## for the messages.  If you want to change these, you can either modify
## the code (which is below) or create a new set of functions and change 
## the below code references appropriately.

$HTMLHEAD = \&tbhtml_head_pub;	
$HTMLFOOT = \&tbhtml_foot_pub;

## Filename that contains the title

$TITLEFILE = "title";

## Filename that contains header information

$INTROFILE = "header";

## Filename that contains footer information

$OUTROFILE = "footer";

## Filename that contains the papers we want to list, one per line.

$PAPERSFILE = "numbers";

###############################################################################
### main() ####################################################################
###############################################################################

use CGI;
use DBIx::Frame::CGI;
use strict;
use TCB::TeX2HTML;
use warnings;

# Load the appropriate class module
{ local $@; eval "use $CLASS";  die "$@\n" if $@; }
$0 =~ s%.*/%%g;         # Lose the annoying path information

if (-r $TITLEFILE) { 
  open (TITLE, $TITLEFILE) or Error("Couldn't open title file: $!");
  $TITLE = <TITLE>; chomp $TITLE;
  close TITLE;
}

Error("No papers to list") unless (-r $PAPERSFILE);

my @papers;
open(PAPERS, $PAPERSFILE) or Error("Couldn't open papers file: $!");
while (<PAPERS>) { 
  chomp;  
  s/#.*$//;  # Trim comments
  s/^\s+|\s*//g;  # Trim whitespace
  next if /^$/; 
  next unless /^[\d\s]+$/;
  foreach my $papers (split /\s+/, $_) { push @papers, $papers; }
}
close PAPERS;

$0 =~ s%.*/|\?.*%%g;		# Lose the annoying path information

my $cgi = new CGI || die "Couldn't open CGI";
$DB = $CLASS->connect( $DATABASE, $DBUSER, $DBPASS, $DBHOST, $DBTYPE )
	or Error("Couldn't connect to $DBHOST: $DBI::errstr\n");
my $error = $DBI::errstr;	# Avoid a warning, otherwise unnecessary

my $params = {};
foreach ($cgi->param) { $$params{$_} = $cgi->param($_); }
my $action = $cgi->param('action') || "";
my $type   = $cgi->param('table') || "";

$DB->set_list( 'Papers', [ 
  	{ '' => [ \&html_listbib, '/Publications/Papers/abstract.cgi', 
		'$$TBRef$$', '$$TBCode$$' ] } 
			 ] );
$DB->set_html( 'Papers', \&html_search );

my (%codes, @data, @codes);
my $search = { 'TBRef' => join(" OR TBRef = ", @papers) };
  
if (@papers) { 
  foreach my $data ($DB->select('Papers', $search)) {
    next unless ref $data;
    my $tbref = $$data{TBRef};  next unless $tbref;
    next unless $tbref;
    next if TCB::Publications::Papers->is_rejected($data);
    next if TCB::Publications::Papers->is_submitted($data);
    $$data{TBRef} = $tbref;
    $codes{$tbref} = $data;
  }
}

my $sort = $$params{'sort'} || 'reverse';
if ($sort eq 'reverse') { 
  foreach my $tbref (sort {$b<=>$a} keys %codes) { push @data, $codes{$tbref} }
} elsif ($sort eq 'forward') { 
  foreach my $tbref (sort {$a<=>$b} keys %codes) { push @data, $codes{$tbref} }
} elsif ($sort eq 'inorder') {
  foreach my $tbref (@papers) { push @data, $codes{$tbref} if $codes{$tbref} }
} else { } 

# Actually print everything off as appropriate
( print $cgi->header(), &$HTMLHEAD($TITLE), "\n" ) && $HTML++;

if ($INTROFILE && -r $INTROFILE) { printfile($INTROFILE) }
if (@data) { 
  print $DB->html_list_nosearch('Papers', $params, $OPTIONS, @data );
} else { print "<p> <i>No papers returned</i> </p> \n"; }
if ($OUTROFILE && -r $OUTROFILE) { printfile($OUTROFILE) }

print &$HTMLFOOT($DEBUG);
exit(0);

###############################################################################
### Subroutines ###############################################################
###############################################################################

## Error ( PROBLEM [, PROBLEM [...]] )
# Prints an error message based on PROBLEM and exits.

sub Error {
  print CGI->header(), &$HTMLHEAD("Error in '$0'") unless $HTML;

  print "This script failed for the following reasons: <p>\n<ul>\n";
  foreach (@_) { next unless $_; print "<li>", canon($_), "<br>\n"; }
  print "</ul>\n";

  print &$HTMLFOOT($DEBUG);
  exit 0;
}

## canon ( ITEM )
# Returns a printable version of whatever it's passed.  Used by Error().

sub canon {
  my $item = shift;
  if    ( ref($item) eq "ARRAY" )   { join(' ', @$item) }
  elsif ( ref($item) eq "HASH" )    { join(' ', %$item) }
  elsif ( ref($item) eq "" )        { $item }
  else                              { $item }
}

## html_head ( TITLE [, OPTIONS] )
# Prints off a basic HTML header, with debugging information.  Extra
# options are passed through to start_html.

sub html_head { 
  my $title = shift || $TITLE || "";
  use CGI;   my $cgi = new CGI;
  $cgi->start_html( -title => $title, @_ );
}

## html_foot ( DEBUG [, OPTIONS] )
# Prints off a basic HTML footer, with debugging information.

sub html_foot { 
  my $debug = shift || $DEBUG;
  use CGI;   my $cgi = new CGI;
  my @return = debuginfo($debug);
  push @return, $cgi->end_html(@_);
  join("\n", @return, "");
}

## debuginfo ( LEVEL ) 
# Takes care of printing debugging information, as described above

sub debuginfo {
  my $debug = shift || 0;

  my @return;

  if ($debug & 1) {
    push @return, "SQL Queries: <p>\n<ul>";
    foreach ($DB->queries) { push @return, " <li>$_" }
    push @return, "</ul>";
  }

  if ($debug & 2) {
    push @return,  "Parameters: <p>\n<ul>\n";
    foreach ($cgi->param) { push @return,  " <li>$_: ", $cgi->param($_); }
    push @return,  "</ul>";
  }

  if ($debug & 4) {
    push @return,  "Environment Variables: <p>\n<ul>";
    foreach (sort keys %ENV) { push @return, " <li>$_: $ENV{$_}"; }
    push @return,  "</ul>";
  }

  wantarray ? @return : join("\n", @return);
}

sub html_listbib {
  my ($DB, $program, $tbref, $tbcode) = @_;
  return "" unless $tbcode; return "" unless $tbref;
  $program ||= '';

  my @return = "";

  # Split out the title, so we don't print it twice (this algorithm was
  # from the old publications database; I don't really understand it, or
  # at least I haven't looked at it enough to really get it.)
  my ($author, $title, @rest) = split(/\\newblock/,
			     TCB::Publications::Papers->getbib($tbcode) || "" );

  
  $title =~ s/^\s+|\s+$//g;	
  $title = tex2html($title);  $title =~ s/^\s+|\s+$//g;    # Trim whitespace
  return "" unless $title;
  push @return, "$tbref. <a href='$program?tbcode=$tbcode'>$title</a>\n";
  push @return, tex2html("$author @rest");

  map { s/Damjanovi\\'c/Damjanovic/g } @return;   # Special case for Ana
  wantarray ? @return : join("\n", @return);
}

sub printfile {
  my ($file) = @_;
  return 0 unless (-r $file);
  my @text;
  open(FILE, $file) or (warn "Couldn't open $file: $!\n" && return 0); 
  @text = <FILE>; close FILE; 
  print @text; 
  1;
}

sub html_search { 
  return <<ENDL;
No search possible 
ENDL
}

###############################################################################
### Version History ###########################################################
###############################################################################
# v1.1a		Wed Oct  9 10:59:24 CDT 2002
### Now uses DBI::Frame properly.  More generic than before, too; uses
### defaults from the TB::Publications code.
# v1.2a         Fri Oct 11 10:40:37 CDT 2002
### Trims whitespace from the title.
# v1.3a		Fri Nov 15 14:05:22 CST 2002
### Now uses the idea of 'startcode' to let us logically map several sets
### of data into the same database.  Still not documented.  This is, of
### course, a huge hack and I don't like it.  
### Doesn't throw up a bad HTML box when no results are offered.
### Various bugfixes. 
