$VERSION = "1.01";
package TCB::Publications::Papers;
our $VERSION = "1.01";

# -*- Perl -*- 		Mon May 17 19:15:17 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME

TCB::Publications::Papers - published papers

=head1 SYNOPSIS

  use TCB::Publications::Papers;

See TCB::Publications for more information.

=head1 DESCRIPTION

The 'Papers' table is the heart of the publications database; it holds all
of the bibliographic and historical information regarding every paper that
goes through the TCB system.  

This table contains the following fields:

 Internal Information (TINYTEXT fields, unless noted)
  ID            Unique numeric ID - auto-generated (INT)
  CreatedBy     Person who created this entry       \  Auto-created using
  ModifiedBy    Person who last modified this entry  \ $ENV{REMOTE_USER}
  CreateDate    Time this entry was created          / and current time in
  ModifyDate    Time this entry was last modified   /  html()

 Publication Information (TINYTEXT fields, unless noted)
  PaperType     The type of paper (Journal Article, Proceedings, etc; 
                possibilities are stored in @PUBTYPES; see below)
  PubDate       Date of publication (freeform, not a DATE field)
  Pages         The pages that the paper is printed on (journal or book)
  PubStatus     The publication status of the paper (Complete, Rejected,
                etc; possibilities are stored in @PUBSTATUS; see below)

 Journal Information (TINYTEXT fields, unless noted)
  Journal       The Journal that the paper was published in; more info
                on the journals can be found in the B<Journals> table, 
                though the two tables are not directly tied together.
  JournalVolume The volume of the journal
  JournalManNum The manuscript # of the journal

 Book Information (TINYTEXT fields, unless noted)
  BookTitle     Title of the book
  BookEditors   Editors of the book
  BookPublishers Publishers of the book
  BookYear      Year that the book was published
 
 Administrative Information (TINYTEXT fields, unless noted)
  TBRef         The 'reference number' of the paper (INT) 
                (More information about this is below)
  TBCode        The 'name' of the paper 
                (More information about this field is below)
  TechRpt       Tech Report Number 
  AccountNum    The account number associated with this paper
  Grant1        The primary grant that this paper was supported by
  Grant2        The secondary grant that this paper was supported by
  Grant3        The tertiary grant that this paper was supported by
  Copies        The number of printed copies of this paper (INT) 
  PDF           Where a PDF copy of the paper is stored.  This field has 
                been mostly replaced by the B<Files> table, and can be 
                considered obsolete (though someone may be using it for 
                something else, so leave it there!).
  Notes         Any remaining notes about the paper (primarily Jo's 
                scratchpad)

Key fields:     TBRef, TBCode

List items:     TBRef, TBCode, Authors, Title

Required:       Same as Key fields.

Default order:  TBRef (reverse)

Admin Fields:   Grant1 Grant2 Grant3 Notes TechRpt PDF Copies AccountNum

The 'Papers' table depends on this table.

Depends on the 'Journals' table.

This table was designed by Gila Budescu, Jo Miller, and Charlie Brown.  It
was converted to MySQL by Tim Skirvin in 2000, and fully converted to
DBI::Frame in Oct 2002.  

More information about TBRef and TBCode are probably necessary.  

=over 4

=item TBRef

The TBRef field is an INT, and is chosen based on the publication number
within the Bibfile ($TB::Publications::BIBFILE).  The number is
*extremely* 
political, and should not be handled by the sysadmin team.  Take it as
holy writ.

Special codes within TBRef:

  666666                Rejected paper
  999999                Submitted paper

=item TBCode

The TBCode field is generated by Jo, and consists of the first 3-4
letters of the the primary author's last name, the year of publication,
and an additional letter if there have been multiple papers in the same 
year.  To whit:
  
  First Paper           SKIR2002
  Second Paper          SKIR2002A
  Third Paper           SKIR2002B

=back

=head1 USAGE

=head2 Variables

The following variables are set in the module, and are used as defaults
for creation and searching.

=over 4

=item @TCB::Publications::Papers::PUBTYPES

Possible types of publication.  Default possibilities: Journal Article,
Proceedings, Book Article, Other, Technical Report.

=item @TCB::Publications::Papers::PUBSTATUS

Possible states of publication.  Default possibilities: Complete,
Rejected, In Press, Submitted.

=item $TCB::Publications::Papers::WEBROOT

The location of the web root; used by several functions to get more
information.  Default location: /Common/WebRoot .

=item $TCB::Publications::Papers::BIBFILE

Location of the .bbl bibliography file, used to get printable
bioliographic information.  Default location:
/home/papers/group_database/bib.bbl .

=item $TCB::Publications::Papers::ABSTRACTS

Directory structure where abstract information is stored.  Default
location: $WEBROOT/Publications/Papers/abstracts .

  
=back

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $REQUIRED $ADMIN $ORDER 
	     $ABSTRACTS $BIBFILE $WEBROOT @PUBTYPES @PUBSTATUS %BIB );
use strict;
use warnings;
use CGI;
use TCB::Publications;
use TCB::Publications::Functions;

push @ISA, qw( TCB::Publications );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Papers";
$FIELDS = {

  # Non User-Modified Information
  'ID'          =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'CreatedBy'   =>  'TINYTEXT',    'ModifiedBy'  =>  'TINYTEXT',
  'CreateDate'  =>  'DATE',        'ModifyDate'  =>  'DATE',

  # Basic Information
  'Title'       =>  'TINYTEXT',    'Authors'     =>  'TINYTEXT',
  'CorrAuth'    =>  'TINYTEXT',    'Abstract'    =>  'TEXT',       

  # Publication Information
  'PubStatus'   =>  'TINYTEXT',    'PaperType'   =>  'TINYTEXT',    
  'PubDate'     =>  'TINYTEXT',    'Pages' => 'TINYTEXT',    

  # Journal Information
  'Journal'     =>  'TINYTEXT',    'JournalVolume'  =>  'TINYTEXT',
  'JournalManNum'  =>  'TINYTEXT',
  
  # Book information
  'BookTitle'   =>  'TINYTEXT',    'BookEditors' =>  'TINYTEXT',
  'BookPublishers' =>  'TINYTEXT', 'BookYear'   =>   'TINYTEXT', 

  # TB Information
  'TBRef'       =>  'INT',         'TBCode'      =>  'TINYTEXT',
  'TechRpt'     =>  'TINYTEXT',    'Notes'       =>  'TEXT',
  'AccountNum'  =>  'TINYTEXT',    'Grant1'      =>  'TINYTEXT',
  'Grant2'      =>  'TINYTEXT',    'Grant3'      =>  'TINYTEXT',
  'PDF'         =>  'TINYTEXT',    'Copies'      =>  'INT',         
          };

$KEYS  = [ 'TBRef', 'TBCode' ];
$LIST  = [ 'TBRef', 'TBCode', 'Authors', 'Title' ];
$REQUIRED = $KEYS;
$ADMIN = [ qw( Grant1 Grant2 Grant3 Notes TechRpt PDF Copies AccountNum ) ];
$ORDER = [ '-TBRef' ];

$WEBROOT   = "/Common/WebRoot";
$BIBFILE   = "/home/papers/group_database/bib.bbl";
$ABSTRACTS = "$WEBROOT/Publications/Papers/abstracts";
  
@PUBTYPES  = ( 'Journal Article', 'Proceedings', 'Book Article', 
               'Other', 'Technical Report', '' );
@PUBSTATUS = ( 'Complete', 'Rejected', 'In Press', 'Submitted' );

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};    $options ||= {};
  my $admin = $$options{'admin'} ? 1 : 0;

  if (lc $type eq 'create') {
    $$entry{'PaperType'}  ||= "Journal Article";
    $$entry{'CreatedBy'}  ||= $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'CreateDate'} ||= $self->date_mysql(time);
    $$entry{'PubStatus'}  ||= "Submitted";
  } elsif (lc $type eq 'search') {
  } elsif (lc $type eq 'edit' || lc $type eq 'update') {
    $$entry{'ModifiedBy'} = $ENV{'REMOTE_USER'} || "unknown";
    $$entry{'ModifyDate'} = $self->date_mysql(time);
  } else { }

  my %public = ( 1 => "Yes", 0 => "No" );
  my @status = sort @PUBSTATUS;
  my @types  = sort @PUBTYPES; 

  my @journals = sort $self->select_fieldlist('Journals', 'ShortName'), '';
  map { s/^\s+|\s+$//g } @journals;     # Trim whitespace
  
  if (lc $type eq 'search') {
    unshift @journals, '';
    unshift @status, '';
    unshift @types, '';
    $public{''} = "*";
  } else {
    unshift @types,  $$entry{PaperType};
    unshift @journals, $$entry{Journal};
  }

  my @return = <<HTML;
<div class="basetable">
 <h3 class="tablehead"> Basic Information  </h3>

 <div class="row1">
  <span class="label">Title</span>
  <span class="formw">
   @{[ $cgi->textfield('Title', $$entry{Title} || "", 70, 1024) ]}
  </span>
 </div>
 
 <div class="row1">
  <span class="label">Author</span>
  <span class="formw">
   @{[ $cgi->textfield('Authors', $$entry{Authors} || "", 70, 1024) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Corresponding Author</span>
  <span class="formw">
   @{[ $cgi->textfield('CorrAuth', $$entry{CorrAuth} || "", 70, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Abstract</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Abstract', -default=>$$entry{Abstract} || "",
                        -rows=>7, -cols=>70, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">PDF (Full Text)</span>
  <span class="formw">
   @{[ $cgi->textfield('PDF', $$entry{PDF} || "", 70, 255) ]}
  </span>
 </div>

 <div>
  <div class="modbox">
    @{[ $self->created_html( $$entry{CreateDate} || "", 
			     $$entry{ModifyDate} || "",
                             $$entry{CreatedBy}  || "", 
			     $$entry{ModifiedBy} || "", !$admin ) ]}
  </div>

  <div class="rowfloat">
   <span class="label">TB Code</span>
   <span class="formw">
    @{[ $cgi->textfield('TBCode', $$entry{TBCode} || "", 11, 255) ]}
   </span>
   <span class="label">TB Ref #</span>
   <span class="formw">
    @{[ $cgi->textfield('TBRef', $$entry{TBRef} || "", 11, 255) ]}
   </span>
  <br />
   <span class="label">Status</span>
   <span class="formw">
    @{[$cgi->popup_menu('PubStatus', \@status, $$entry{PubStatus} || "") ]}
   </span>
  </div>
 </div>

 <h3 class="tablehead"> Publication Information  </h3>

 <div class="row3">
  <span class="label">Pub Date</span>
  <span class="formw">
   @{[ $cgi->textfield('PubDate', $$entry{PubDate} || "", 20, 255) ]}
  </span>
  <span class="label">Type</span>
  <span class="formw">
   @{[$cgi->popup_menu('PaperType', \@types, $$entry{PaperType} || "") ]}
  </span>
  <span class="label">Pages</span>
  <span class="formw">
   @{[ $cgi->textfield('Pages', $$entry{Pages} || "", 12, 255) ]}
  </span>
 </div>

 <p> &nbsp; </p>
 <h3 class="tablehead"> Journal Information  </h3>

 <div class="row1">
  <span class="label">Journal Name</span>
  <span class="formw">
   @{[ $cgi->popup_menu('Journal', \@journals, $$entry{Journal} || "") ]}
  </span>
 </div>

 <div class="row2">
  <span class="label">Volume</span>
  <span class="formw">
   @{[ $cgi->textfield('JournalVolume', $$entry{JournalVolume} || "", 8, 255) ]}
  </span>
  <span class="label">Manuscript # </span>
  <span class="formw">
   @{[ $cgi->textfield('JournalManNum', $$entry{JournalManNum} || "", 8, 255) ]}
  </span>
 </div>

 <p> &nbsp; </p>
 <h3 class="tablehead"> Book Information  </h3>

 <div class="row1">
  <span class="label">Title</span>
  <span class="formw">
   @{[ $cgi->textfield('BookTitle', $$entry{BookTitle} || "", 70, 255) ]}
  </span>
 </div>

 <div class="row3">
  <span class="label">Publishers</span>
  <span class="formw">
   @{[ $cgi->textfield('BookPublishers', $$entry{BookPublishers} || "", 
								20, 255) ]}
  </span>
  <span class="label">Editors</span>
  <span class="formw">
   @{[ $cgi->textfield('BookEditors', $$entry{BookEditors} || "", 20, 255) ]}
  </span>
  <span class="label">Year</span>
  <span class="formw">
   @{[ $cgi->textfield('BookYear', $$entry{BookYear} || "", 5, 4) ]}
  </span>
 </div>

HTML
  
  push @return, adminhtml($self, $entry, $type, $options, @rest) if $admin;

  push @return, <<HTML;
 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

=item adminhtml ( ENTRY, TYPE, OPTIONS )

Called within html() only if the 'admin' flag is set in C<OPTIONS>, this
offers the fields that shouldn't show up in all searches.

=cut

sub adminhtml {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};    $options ||= {};

  my @return;
  push @return, <<HTML;
 <p> &nbsp; </p>
 <h3 class="tablehead"> TCB Information  </h3>

 <div class="row2">
  <span class="label">Grant #1</span>
  <span class="formw">
   @{[ $cgi->textfield('Grant1', $$entry{Grant1} || "", 30, 255) ]}
  </span>
  <span class="label">Account Number</span>
  <span class="formw">
   @{[ $cgi->textfield('AccountNum', $$entry{AccountNum} || "", 10, 255) ]}
  </span>
 </div>

 <div class="row2">
  <span class="label">Grant #2</span>
  <span class="formw">
   @{[ $cgi->textfield('Grant2', $$entry{Grant2} || "", 30, 255) ]}
  </span>
  <span class="label">Tech Rpt #</span>
  <span class="formw">
   @{[ $cgi->textfield('TechRpt', $$entry{TechRpt} || "", 10, 255) ]}
  </span>
 </div>

 <div class="row2">
  <span class="label">Grant #3</span>
  <span class="formw">
   @{[ $cgi->textfield('Grant3', $$entry{Grant3} || "", 30, 255) ]}
  </span>
  <span class="label">Copies</span>
  <span class="formw">
   @{[ $cgi->textfield('Copies', $$entry{Copies} || "", 10, 255) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
    @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "",       
                       -rows=>7, -cols=>70, -maxlength=>65535,
                       -wrap=>'physical') ]}
  </span>
 </div>


HTML
  wantarray ? @return : join("\n", @return);
}


=item getbib ( TBREF [, FULL] ) 

Returns the bibliography information for B<TBREF> as a single string (with
integral newlines).  Gets the information from
$TB::Publications::Papers::BIBFILE; if it isn't present, returns undef.

If C<FULL> is set, then we return an empty string if there is no bib
information; otherwise, we will just return the title 

Note that the bibfile is cached internally, so it's loaded once per script
invocation.

=cut

sub getbib {
  my ($self, $tbref, $full) = @_;
  $tbref ||= "none";
  my $count = 0;
  unless ( scalar(keys %BIB) > 0 ) {
    open(BBL, $BIBFILE) or return "Couldn't open $BIBFILE: $!\n";
    my $current = "";  
    while (<BBL>) {
      chomp;            # Remove newline
      s/^\s+|\s+$//;    # Trim whitespace
      if (/^\s*$/) { $count++ if $current;  $current = "";  next; }
      if ( /^\\bibitem{(\S+)}\s*$/ ) { $current ||= $1; $BIB{$current} = ""; } 
      elsif ( /^\[.*\]\s*$/) { $current = undef } 
      elsif ($current) {
        $BIB{$current} ? $BIB{$current} = $_ || ""
                       : $BIB{$current} = "$BIB{$current}\n$_";
      } 
    }
    close (BBL);
    $BIB{$current} =~ s/^\s+|\s+$//g
        if ($current && $BIB{$current});   # Trim leading/trailing whitespace
  }
  $BIB{$tbref} || undef;
}

=item abstract ( TBCODE )

Returns an arrayref (or array, if asked) containing the abstract of the
the given paper (by B<TBCODE>).  If the abstract doesn't exist, returns an
empty array.

Note that this is *not* what is contained in the B<Abstract> row of this
table.  The abstract that's currently being used is formatted (based on
this abstract info) using latex2html to properly display on all systems.
This is a liability, but is still necessary.

=cut

sub abstract {
  my ($self, $tbcode) = @_;
  my $cgi = new CGI::SHTML;

  my $dir = $ABSTRACTS;  $dir =~ s%^$WEBROOT%%g;
  my @return = $cgi->parse_shtml("<!--#include file=\"$ABSTRACTS/$tbcode/index.html\"-->");
  map { s/(SRC|HREF)=['"](.*.(?:gif|css|png))['"]/$1="$dir\/$tbcode\/$2"/g } @return;
  wantarray ? @return : join("\n", @return);
}

=item is_rejected ( ENTRY )

Returns 1 if this entry is a rejected paper, 0 if not, or undef if the
paper (or B<ENTRY>) is invalid.

=cut

sub is_rejected { 
  my ($self, $entry) = @_;  
  return undef unless ref $entry;
  my $tbref = $$entry{TBRef} || return undef;
  $tbref eq '6666666' ? 1 : 0;
}

=item is_submitted ( ENTRY ) 

Returns 1 if this entry is a submitted paper, 0 if not, or undef if the
paper (or B<ENTRY>) is invalid.

=cut

sub is_submitted {
  my ($self, $entry) = @_;  
  return undef unless ref $entry;
  my $tbref = $$entry{TBRef} || return undef;
  $tbref eq '9999999' ? 1 : 0;
}

=item text ( )

Not currently populated.

=cut

sub text { }

=back

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

use CGI::SHTML;
use strict;
use TCB::Publications qw( );
unshift @ISA, "TCB::Publications";

TCB::Publications->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
                        $REQUIRED, \&html, \&text);

=head1 NOTES

This database currently only holds papers co-authored by Prof. Schulten.
It has been proposed that the other Group professors should also contain
information in there, but some hacks will be necessary to allow this.  I
(Tim) proposed the following on 2002/09/10:

  My suggestion: we offer a separate space for TBRefs and Codes for each
  individual professor.  This way, we can continue to use the same database
  while still having somewhat separate information.  My suggestions:

                        TBRef                   TBCode
        Klaus             #                     SKIR2002
        Bob               # + 10000             SKIR2002-S
        Sanjay            # + 20000             SKIR2002-K
        Todd              # + 30000             SKIR2002-M
        Zan               # + 40000             SKIR2002-[LZ]
 
  Each professor would have their own .bib file (this is Deyu's job; there's
  not much I can do to help with this), and I'd have to modify the scripts
  to only show one professor's at a time.  All of this would take a fair
  amount of coding on my part, though it's certainly workable.

=head1 TODO

Start working on intellectual property issues for these papers.

=head1 REQUIREMENTS

Perl 5.6.1 or better, DBIx::Frame, TCB::Publications, CGI::SHTML.

=head1 SEE ALSO

B<TCB::Publications>, B<TCB::Publications::Papers>, B<DBIx::Frame>,
B<DBIx::Frame::CGI>

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2          Tue Oct  8 17:10:40 CDT 2002
### First version using a real version of DBI::Frame.  Documented as well.
### No actual changes were made to the code yet, though.
# v0.3		Thu Nov 14 11:09:10 CST 2002
### Added 'Restricted' field to this table.  Heh.  'ALTER TABLE'.  Neat.
# v1.00		Fri May 14 14:52:39 CDT 2004 
### Ready for DBIx::Frame and TCB::Publications.
# v1.01		Mon May 17 19:14:58 CDT 2004 
### Fixed some HREF-fixing code in abstract().
# v1.02		Wed May 19 11:16:02 CDT 2004 
### More HREF-fixing stuff - this time, set ^M's to newlines.  
# v1.03		Mon Aug 23 15:30:11 CDT 2004 
### Now recognizes PNGs in abstract(), as well as .gif and .css.
