$VERSION = "0.52";
package TCB::Publications;
our $VERSION = "0.52";

# -*- Perl -*- 		Fri May 14 15:28:03 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2002-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME

TCB::Publications::Functions - additional functions for TCB::Publications 

=head1 SYNOPSIS

  use TCB::Publications::Functions;

See TCB::Publications for general information on this database.

=head1 DESCRIPTION

TCB::Publications::Functions contains shared functions specific to
TCB::Publications. These are a combination of specific functions for each
table and general extensions to DBIx::Frame.  

=cut

###############################################################################
### main() ####################################################################
###############################################################################

use TCB::Publications;
use Exporter;

use vars qw( @EXPORT_OK @EXPORT @WEBROOT );
use strict;
use CGI;

@WEBROOT = qw( /Common/WebRoot /home/webserver/WebRoot );

###############################################################################
### Functions #################################################################
###############################################################################

=head2 Public Functions

The following methods are available within this object, and can be
exported if a module wants them.

=over 4

=item dates ( DATE1, DATE2 )

Returns the following string, based on whether DATE1 and DATE2 are real
dates (ie not "0000-00-00", the default NULL for mysql's date field):

  DATE1	  DATE2	       Returned String
    Y	    Y		DATE1 - DATE2
    Y       N		DATE1 - 
    N 	    Y		- DATE2
    N       N           No Date

Also, if DATE1 and DATE2 are the same, it just returns DATE1 (the entire
event was on that day).  

=cut

sub dates {
  my ($self, $date1, $date2) = @_;
  map { s/0000-00-00/0/ } $date1, $date2;
  return "No Date" if !$date1 && !$date2;
  return "$date1" if $date1 eq $date2;
  $date1 ? $date2 ? "$date1 - $date2" : "$date1 - "
	 : $date2 ? "- $date2" : "No Date";
}
push @EXPORT_OK, qw( &dates );

=item url ( URL, TEXT )

Returns the following string, based on the existance of URL and TEXT:

   URL	   TEXT	        Returned String
    Y	    Y		<a href='URL'>TEXT</a>
    Y       N           <a href='URL'>URL</a>
    N 	    Y		TEXT
    N       N		No URL

=cut

sub url {
  my ($self, $url, $text) = @_;
  $url ? $text ? "<a href='$url'>$text</a>" : "<a href='$url'>$url</a>"
       : $text ? $text : "No URL";
}
push @EXPORT_OK, qw( &url );

=back

=head2 Internal Functions 

The following methods are available within this object, but are not
exported.

=over 4

=item created_html ( CREATEDATE, MODIFYDATE, CREATEBY, MODIFYBY, NOSHOW )

Returns the standard little table for Date and Person vs Created and
Modified, along with the 'Submit' button.  Used by several of the tables,
we might as well share the code between them, but this is still private
code.  NOSHOW, if set, says "don't show them the table".

=cut

sub created_html {
  my $cgi = new CGI;
  my ($self, $createdate, $modifydate, $createdby, $modifiedby, $noshow) = @_;
  $createdate ||= "";  $modifydate ||= "";
  $createdby  ||= "";  $modifiedby ||= "";
  my @return;
  push @return, $cgi->hidden('CreateDate', $createdate),
     		$cgi->hidden('ModifyDate', $modifydate),
     		$cgi->hidden('CreatedBy',  $createdby ),
     		$cgi->hidden('ModifiedBy', $modifiedby);
  return @return if $noshow;
  push @return, <<END;
     <table class="modbox" border>
      <tr> <th> </th> <th>Created</th> <th>Modified</th> </tr>
      <tr> <th>Date</th>
           <td align=center>@{[ $createdate ]}</td>
           <td align=center>@{[ $modifydate ]}</td> 
	   </tr>
      <tr> <th>Person</th>
           <td align=center>@{[ $createdby  ]}</td>
           <td align=center>@{[ $modifiedby ]}</td>  
	   </tr>
     </table>
END
  @return;
}

=item date_mysql ( [TIME] )

Returns the date based on B<TIME> for use with mysql.  If B<TIME> is not
passed, just uses the current time.

=cut

sub date_mysql {
  my ($self, $time) = @_;  $time ||= time;
  sprintf("%04d-%02d-%02d", (localtime($time))[5] + 1900,
                        (localtime($time))[4] + 1, (localtime($time))[3]);
}

=item html_fulltext ( TBCODE [, PRIVATE] )

Returns an array of HTML references to the given article, based on the
given TBCODE.  If PRIVATE is set, then it includes 'Restricted' entries;
otherwise it doesn't.

Note that if 'NoRequest' is one of the types of files here, then we just reutnr
the string "NoRequest".

=cut

sub html_fulltext {
  my ($self, $tbcode, $private) = @_;
  return () unless $tbcode;

  my @fulltext;
  foreach my $data ($self->select('Files', { 'TBCode' => $tbcode })) {
    next if (!$private && $$data{'Restricted'} > 0); 	# Restrict entries
    return "NoRequest" if ($$data{Type} eq 'NoRequest');
    my $url = _file_url($$data{'Location'}, $$data{'Type'});
    push @fulltext, $url if $url;
  }
  @fulltext;
}

=item html_fulltext_public ( TBCODE )

As with C<html_fulltext()>, except that it never shows restricted entries.

=cut

sub html_fulltext_public {
  my ($self, $tbcode) = @_;
  return () unless $tbcode;

  my @fulltext;
  foreach my $data ($self->select('Files', { 'TBCode' => $tbcode })) {
    next unless ($$data{'Restricted'} <= 0); 	# Restrict entries
    return "NoRequest" if ($$data{Type} eq 'NoRequest');
    my $url = _file_url($$data{'Location'}, $$data{'Type'});
    push @fulltext, $url if $url;
  }
  @fulltext;

}


=item html_fulltext_acct ( TBCODE [, PROG], [, PRIVATE] )

As with C<html_fulltext()>, except that it returns a link to that paper
using C<PROG> (which defaults to $0).  

=cut

sub html_fulltext_acct {
  my ($self, $tbcode, $prog, $private) = @_;
  return () unless $tbcode;  $prog ||= $0;
  
  my @fulltext;
  foreach my $data ($self->select('Files', { 'TBCode' => $tbcode })) {
    next if (!$private && $$data{'Restricted'} > 0); 	# Restrict entries
    my $type = $$data{'Type'};
       $type =~ s/HTML/Journal/g;       	 # HTML means 'journal';
       $type =~ s/Info/Additional Information/g; # Info means 'additional info';
       $type =~ s/Request/Request a Copy/g;      # Request via email
    my $file = $$data{'Location'};

    # The URL needs to be trimmed if it's a file
    my $url = $file;  
    foreach (@WEBROOT) { $url =~ s%$_%%g; }

    my $fsize = CGI::SHTML->fsize($file) if (-r $file);
    if ($url && $type) {
      ## This should be a form instead; we'll work on it later.
      my $string = "<a href='$prog?tbcode=$tbcode'>$type</a>";
      push @fulltext, $fsize ? "$string ($fsize)" : $string;
    }
  }
  @fulltext;
}

=back

=cut

###############################################################################
### Internal Functions ########################################################
###############################################################################

### _file_url ( FILE, TYPE )
# Gives a URL based on the filename given, as modified by @WEBROOT.  If we
# can get the file size, then include that too.
sub _file_url {
  my ($file, $type) = @_;
  my $url = $file;  
  foreach (@WEBROOT) { $url =~ s%$_%%g; }
  $type =~ s/HTML/Journal/g;                # HTML means 'journal';
  $type =~ s/Info/Additional Information/g; # Info means 'additional info';
  $type =~ s/Request/Request a Copy/g;      # Request via email
  my $cgi = new CGI::SHTML;
  my $fsize = $cgi->parse_shtml( 
	"<!--#fsize file=\"$file\"-->") if (-r $file);
  return "" unless ($url && $type);
  $fsize ? "<a href='$url'>$type</a> ($fsize)"
         : "<a href='$url'>$type</a>";
}

1;

=head1 REQUIREMENTS

Perl 5.6.1 or better, DBIx::Frame 1.05 or better, TCB::Publications.

=head1 SEE ALSO

B<TCB::Publications>, B<DBIx::Frame>

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2		Tue Oct  8 15:33:45 CDT 2002
### Updated for actual DBI::Frame version.
# v0.3		Wed Oct  9 14:39:41 CDT 2002
### Added html_fulltext().
# v0.4		Thu Nov 14 12:35:13 CST 2002
### html_fulltext() now ignores 'Restricted' links.
# v0.5		Fri May 14 15:31:55 CDT 2004
### Converting for TCB::Publications.  
# v0.51		Mon Dec 13 15:00:18 CST 2004 
### In _file_url() and html_fulltext_acct(), added support for 'Info'
### category.
# v0.52		Mon Jan 03 15:41:08 CST 2005 
### In _file_url() and html_fulltext_acct(), added support for 'Request'
### category.
