$VERSION = "1.02";
package TCB::Publications::Files;
our $VERSION = "1.02";

# -*- Perl -*- 		Fri 15 Apr 10:29:11 CDT 2005 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################


=head1 NAME

TCB::Publications::Files - fulltext file locations and accessibility

=head1 SYNOPSIS

  use TCB::Publications::Files;

See TCB::Publications for more information.

=head1 DESCRIPTION

The 'Files' table stores URLs and file names where papers are stored, so
they can be distributed.  

This table contains the following fields:

 Internal Information
  ID            Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  TBCode	The TBCode of the appropriate paper (which is a unique 
		identifier to the B<Papers> table) 
  Type		The type of link - HTML, PDF, etc.  Possibilities are 
		stored in @LINKTYPES; see below.
  Location	The location of the file, either a URL or a local filename 
  Description	A description of the link (TEXT)
  Restricted	Restricted for public use?  1 => private, 0 => public (INT)

Key fields:     TBCode, Location

List items:     TBCode, Type

Required:       Same as Key fields.

Default order:  TBCode

Admin Fields:   None

No other tables depend on this table directly.

Depends on the 'Papers' table.

=head1 USAGE

=head2 Variables

The following variables are set in the module, and are used as defaults
for creation and searching.

=over 4

=item @TCB::Publications::Files::LINKTYPES 

Possibly types of files that can be linked to.  Default possibilities: 
HTML, PS, DVI, PDF, Request, Info, NoRequest.

=back

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $REQUIRED $ADMIN 
	     $ORDER @LINKTYPES );
use strict;
use warnings;
use CGI;
use TCB::Publications;

push @ISA, qw( TCB::Publications );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Files";
$FIELDS = {
  'ID'		=>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'TBCode'	=>  'TINYTEXT', 	'Type'	=>  'TINYTEXT',
  'Location'	=>  'TEXT', 	 	'Description' =>  'TEXT',
  'Restricted'  =>  'INT',
	  };
$KEYS  = [ 'TBCode', 'Location' ];
$LIST  = [ 'TBCode', 'Type' ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ 'TBCode' ];

@LINKTYPES = qw ( HTML PS DVI PDF Request Info NoRequest );

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};

  my %public = ( 0 => "Public", 1 => "Internal Only" );
  if (lc $type eq 'search') { $public{''} = "*" }

  my @types = sort @LINKTYPES;  unshift @types, '';

  my @codes = sort grep { $_ if /\S+/ } 
		$self->select_fieldlist('Papers', 'TBCode');  
  unshift @codes, '';
  
  my @return = <<HTML;
<div class="basetable">
 <div class="row2">
  <span class="label">TBCode</span>
  <span class="formw">
   @{[ $cgi->popup_menu('TBCode', \@codes, $$entry{TBCode} || "") ]} 
  </span>
  <span class="label">File Type</span>
  <span class="formw">
   @{[ $cgi->popup_menu('Type', \@types, $$entry{Type} || "") ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Location</span>
  <span class="formw">
   @{[ $cgi->textfield('Location', $$entry{Location} || "", 70, 1024) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Description</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Description', 
			-default=>$$entry{Description} || "",
                        -rows=>7, -cols=>60, -maxlength=>65535,
                        -wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Restricted?</span>
  <span class="formw">
    @{[ $cgi->popup_menu('Restricted', [sort keys %public], 
		  $$entry{Restricted} || "", \%public) ]}
  </span>
 </div>


 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=item text ( )

Not currently populated.

=cut

sub text { }

=back

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

use strict;
use TCB::Publications qw( );
unshift @ISA, "TCB::Publications";

TCB::Publications->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
                        $REQUIRED, \&html, \&text);

=head1 NOTES

=head1 TODO

Start working on intellectual property issues for these papers.

=head1 REQUIREMENTS

Perl 5.6.1 or better, DBIx::Frame, TCB::Publications.

=head1 SEE ALSO

B<TCB::Publications>, B<TCB::Publications::Papers>, B<DBIx::Frame>,
B<DBIx::Frame::CGI>

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-publications/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2          Tue Oct  8 17:10:40 CDT 2002
### First version using a real version of DBI::Frame.  Documented as well.
### No actual changes were made to the code yet, though.
# v0.3		Thu Nov 14 11:09:10 CST 2002
### Added 'Restricted' field to this table.  Heh.  'ALTER TABLE'.  Neat.
# v1.00		Fri May 14 14:52:39 CDT 2004 
### Ready for DBIx::Frame and TCB::Publications.
# v1.01		Mon Dec 13 14:59:25 CST 2004 
### Added 'Info' category for files.
# v1.02         Fri 15 Apr 10:28:43 CDT 2005    tskirvin
### Added 'NoRequest' category for files.
