$VERSION = "1.00";
package TCB::System::Loan;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 14:08:12 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Loan - TCB Loan Database Table

=head1 SYNOPSIS

  use TCB::System::Loan;

See B<TCB::Loan> or B<TCB::System> for more information on how to use this
module.

=head1 DESCRIPTION

The 'Loan' table is used to list all group equipment loans.  

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Description   One-line description of the borrowed item 
  PAS           Property tag number of the item 
  Serial        Serial number of the item
  Location      The location of the borrowed item for the duration 
  Purpose       The purpose of the loan
  LoanStart     The date of the start of the loan (DATE)
  RequestEnd    The requested end date of the loan (DATE) 
  LoanEnd       The actual end date of the loan (DATE)
  BorrowerName  The email address of the borrower 
  BorrowerType  The type of user that borrowed the equipment (from 
		  @TCB::Loan::TYPES)
  Notes         Borrower notes about the item - longer descriptions,
                terms, etc.  (TEXT)
  History       Administrative notes about the item - where did it 
                go, how long has it been gone, etc. (TEXT)

Key fields: 	Description, LoanStart, LoanEnd

List items: 	BorrowerName, Description, LoanStart, LoanEnd 

Required: 	Same as Key fields.

Default order: 	LoanStart (reverse), BorrowerName

Admin Fields:	None

No other tables depend on this table directly.

Doesn't depend on any other table.

=head2 Variables

The following variables are set in the module, and are used as defaults
for creation and searching.

=over 4

=item @TCB::System::Loan::TYPES

Lists the types of users that can borrow equipment.  Default
possibilities: 'Student', 'Faculty', 'Staff'.

=back

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT @TYPES );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Loan";
$FIELDS = {
  'ID'           => 'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'PAS'          => 'TINYTEXT',  'Serial'      =>  'TINYTEXT', 
  'LoanStart'    => 'DATE',      'LoanEnd'  => 'DATE', 
  'RequestEnd'   => 'DATE', 
  'Description'  => 'TINYTEXT NOT NULL', 
  'Location'     => 'TINYTEXT NOT NULL', 'Purpose'     =>  'TINYTEXT',
  'BorrowerName' => 'TINYTEXT NOT NULL', 'BorrowerType' => 'TINYTEXT',
  'Notes'        =>  'TEXT',     'History' => 'TEXT'
         };
$KEYS  = [ 'Description', 'LoanStart', 'LoanEnd' ];
$LIST  = [ 'BorrowerName', 'Description', 'LoanStart', 'LoanEnd' ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ '-LoanStart', 'BorrowerName' ];

@TYPES = qw ( Faculty Staff Student );          # Types of borrowers

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;
  my $date = sprintf("%04d-%02d-%02d", (localtime)[5] + 1900, 
		  		       (localtime)[4] + 1, (localtime)[3]);
  if (lc $type eq 'create') {
    $entry ||= { 'LoanStart' => $date, 'LoanEnd' => '0000-00-00' } 
  } else { $entry ||= {}; }

  my @type     = sort @TYPES;
  unshift @type, '' if (lc $type eq 'search');
 
  my @return = <<HTML;
<div class="basetable">
 <div class="row2">
  <span class="label">Description</span>
  <span class="formw"> @{[ $cgi->textfield('Description',
				$$entry{Description} || "", 40, 255) ]} </span>
  <span class="label">Borrower Email</span>
  <span class="formw"> @{[ $cgi->textfield('BorrowerName',
				$$entry{BorrowerName} || "", 20, 255) ]} </span>
 </div>

 <div class="row2">
  <span class="label">Purpose</span>
  <span class="formw">
   @{[ $cgi->textfield('Purpose', $$entry{Purpose} || "", 40, 255) ]}
  </span>
  <span class="label">User Type</span>
  <span class="formw">
   @{[ $cgi->popup_menu('BorrowerType', \@type, $$entry{BorrowerType} || "") ]}
  </span>
 </div>

 <div class="row2">
  <span class="label">Location</span>
  <span class="formw">
   @{[ $cgi->textfield('Location', $$entry{Location} || "", 40, 255) ]}
  </span>
  <span class="label">Requested Period</span>
  <span class="formw">
     @{[ $cgi->textfield('LoanStart', $$entry{LoanStart} || "", 11, 10) ]}
        -
     @{[ $cgi->textfield('RequestEnd', $$entry{RequestEnd} || "", 11, 10) ]}
  </span>
 </div>

 <div class="row3">
  <span class="label">PAS</span>
  <span class="formw">
   @{[ $cgi->textfield('PAS', $$entry{PAS} || "", 20, 255) ]}
  </span>
  <span class="label">Serial</span>
  <span class="formw">
   @{[ $cgi->textfield('Serial', $$entry{Serial} || "", 20, 255) ]}
  </span>
  <span class="label">Returned</span>
  <span class="formw">
     @{[ $cgi->textfield('LoanEnd', $$entry{LoanEnd} || "", 11, 10) ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">User Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "", 
		        -rows=>5, -cols=>60, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>
 <div class="row1">
  <span class="label">History</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'History', -default=>$$entry{History} || "", 
		        -rows=>5, -cols=>60, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

The fundamentals of this table were created before Charlie Brown, before
my time.  It was originally an msql table containing the same fields as
above; it's worked out pretty well, I suppose.  

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-loan/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-loan/license.html> for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 14:08:12 CDT 2004
### Ready to be a part of DBIx::Frame.
