$VERSION = "1.1";
package TCB::Library;
our $VERSION = "1.1";

# -*- Perl -*- 		Tue Oct 21 15:00:13 CDT 2003 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2003-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME

TCB::Library::Functions - additional functions for TCB::Library

=head1 SYNOPSIS

  use TCB::Library::Functions;

=head1 DESCRIPTION

This module is actually just a set of additional functions for
TCB::Library, primarily used to make a nice user interface for the tables.  

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( $VERSION @EXPORT @EXPORT_OK @ISA );

use TCB::Library;
use Mail::Send;
use Date::Parse;
use strict;
use CGI;
use Exporter;  
push @ISA, 'Exporter';

###############################################################################
### Functions #################################################################
###############################################################################

=head2 Functions 

The following methods are available within TCB::Library.

=over 4

=item borrowdate ( ENTRY )

From C<INFO>, uses Date::Parse's str2time() to figure out the actual time
that the book was checked out, based on C<ENTRY> (the output of a
DBIx::Frame select() function).  Returns a string with the day, month, and
year (we were never keeping track of more than the date anyway).

=cut

sub borrowdate { 
  my ($self, $info) = @_; $info ||= {};
  return "(unknown)" unless my $date = $$info{'borrowdate'};
  my $time = str2time($date);
  my @months = qw( Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec );
  sprintf("%02d %3s %4d", 
		(localtime($time))[3], 
		$months[(localtime($time))[4]],
		(localtime($time))[5] + 1900 );
}
push @EXPORT_OK, qw( borrowdate );

=item get_username ( ENTRY ) 

Gets the real name of the user, based on the username out C<ENTRY> and
getpwnam().  Returns '(unknown)' if no name is offered, or just the given
username if no additional information is available.

=cut

sub get_username { 
  my ($self, $info) = @_; $info ||= {};
  my $user = $$info{'name'} || "";
  return "(unknown)" unless $user;
  my @info = getpwnam($user);
  $info[5] ? $info[5] : $user;
}
push @EXPORT_OK, qw( get_username );

=item checkout_by_user ( USERNAME )

Lists the books checked out by a given C<USERNAME> in a nice HTML table.
Includes linkback()s to allow users to check the books back in.

=cut

sub checkout_by_user {
  my ($self, $username) = @_;
  return "" unless $username;
  my @return = "<h2>Books checked out by '$username'</h2>";
  push @return, "<table>";
  foreach my $entry ($self->select('Library', { 'name' => $username }) ) {
    my $code = $$entry{code} || next;   my $copy = $$entry{copy} || next;
    push @return, " <tr>\n"; 
    foreach my $book ($self->select('Book', { 'code' => $code }) ) {
      my $cin = $self->linkback("Check In", { 'code' => uc $code, 
				'copy' => $copy, 'action' => 'checkin' }, {} );
      my $date = $self->borrowdate($entry);
      push @return, "  <td width=100%>", $self->linktitle($book), "</td>";
      push @return, "  <td align=right nowrap>$date</td>";
      push @return, "  <td nowrap>$cin</td>";
    }
    push @return, " </tr>";
  }
  push @return, "</table>";
  
  wantarray ? @return : join("\n", @return);
}

=item linktitle ( ENTRY )

Creates a linkback() with the title of C<ENTRY> as the text, which will
later print off information about that book (using booklist_html()).

=cut

sub linktitle {
  my ($self, $entry) = @_;
  return "" unless $entry && ref $entry;
  my $title = $$entry{title} || return "";
  my $code  = $$entry{code} || return "";
  $self->linkback($title, { 'action' => 'print', 'code' => uc $code }, {} );
} 

=item linkback ( TEXT, HASHREF, DEFAULTREF )

Returns an HTML link back to the same program, based on the hash
references C<HASHREF> and C<DEFAULT>.  C<TEXT> is the string that appears
in the link.  The key/value pairs in C<HASHREF> are the options passed in
the URL; however, if the C<DEFAULT> hash matching value matches
C<HASHREF>, then we assume that we don't need that argument (and we should
try to keep the URL short anyway).

This probably needs more refinement, but it more or less works.  

=cut

sub linkback {
  my ($self, $text, $hash, $default) = @_;
  $hash ||= {};  $default ||= {};
  my $url = $0;  $url =~ s%.*/%%g;      # Should be something better
  my @opts;
  foreach (sort keys %{$hash}) { 
    next unless defined $$hash{$_};  
    push @opts, "$_=$$hash{$_}" unless 
			(defined $$default{$_} && $$default{$_} eq $$hash{$_});
  }
  my $opts = CGI::escapeHTML(join('&', @opts));
  "<a href='$url?$opts'>$text</a>";
}

=item booklist_html ( ENTRY ) 

A replacement html() for listing off each book in a very summarized
format.  Includes a check into the Library table to see which copies are
checked out; each copy then has a linkback() to check out or steal that
copy.  

=cut

sub booklist_html {
  my ($self, $entry, $type, $options, @rest) = @_;
  $entry ||= {};

  my @return;
  my $tbcode = $$entry{'code'}   || "unknown code";
  push @return, "<p> $tbcode </p>";

  my $author = $$entry{'author'} || "unknown author";
  my $title  = $$entry{'title'}  || "unknown title";
  my $year   = $$entry{'year'}   || "unknown year";
  push @return, "<p> <i>$title</i> <br /> $author, $year </p>";

  my $comment = $$entry{'kscomment'} || "";
  push @return, "<p>$comment</p>" if $comment;

  my %list;
  my $copies = $$entry{'numcopy'} || 1;
  for my $i (1..$copies) { 
    $list{$i} = 0;
    foreach my $lib ($self->select('Library',
                        { 'code' => $tbcode, 'copy' => $i }, $options ) ) {
        $list{$i} = $$lib{name};
    }
  }
  my @checkout;
  foreach (sort keys %list) {
    
    if ($list{$_}) {
      my $checkout = $self->linkback("Take It", 
		{ 'copy' => $_, 'code' => $tbcode, 'action' => 'steal' } );
      push @checkout, $list{$_} eq $$options{'user'} 
		? "Copy $_ - Checked out by '$$options{'user'}' (you)"  
		: "Copy $_ - Checked out by '$list{$_}' ($checkout)";
    } else {
      push @checkout, "Copy $_ - " .
	$self->linkback("Check it out", { 'copy' => $_, 'code' => $tbcode, 
					  'action' => 'checkout' } );
    }
  }
  push @return, join("<br />", @checkout, '');
  wantarray ? @return : join("\n", @return);
}

=item checkin ( CODE, COPY )

Checks a book into the library - ie, deletes the entry from the Library
table.  Returns the appropriate HTML code indicating that it's been done
(or not).

=cut

sub checkin {
  my ($self, $code, $copy) = @_;
  return "Won't check out without book code" unless $code;
  return "Won't check out without copy number" unless $copy;

  if ($self->delete('Library', { 'code' => $code, 'copy' => $copy } ) ) {
    return "<h1>Checked in $code</h1>\n";
  } else {
    return "<h1>Couldn't check in $code</h1> <h3>@{[ DBI->errstr ]}</h3>";
  }
}

=item checkout ( CODE, COPY, USER )

Checks out a book - ie, creates an entry in the Library table.  Fails if
the book has already been checked out; in those cases, you'll want to use
steal(). Returns the appropriate HTML code indicating that it's been done
(or not).

=cut

sub checkout {
  my ($self, $code, $copy, $user) = @_;
  return "Won't check out without book code" unless $code;
  return "Won't check out without copy number" unless $copy;
  return "Won't check out without a username" unless $user;
  
  if ( $self->select( 'Library', { 'code' => $code, 'copy' => $copy } ) ) {
    return "<h2>'$code' already checked out</h2>\n";
  } elsif ( $self->insert('Library', { 'code' => $code, 'copy' => $copy, 
       'name' => $user, 'borrowdate' => scalar(localtime) } ) ) {
    return "<h1>$user checked out $code</h1>\n";
  } else {
    return "<h1>Couldn't check out $code</h1> <h3>@{[ DBI->errstr ]}</h3>";
  }
}

=item steal ( CODE, COPY, CONFIRM, USER )

Steal an item, moving it from one user to the other.  If C<CONFIRM> is not
set, then it will return an HTML table asking for confirmation (which will
later be passed on to the function); if it is set, then it deletes the old 
checkout entry, creates a new one, and mails the person who originally had
the book (using B<Mail::Send>) to let them know this has been done.  

Returns the necessary text 

=cut

sub steal {
  my ($self, $code, $copy, $confirm, $user) = @_;
  return "Won't check out without book code" unless $code;
  return "Won't check out without copy number" unless $copy;
  return "Won't check out without a username" unless $user;
  $confirm ||= 0;
  
  my $entry = ($self->select( 'Library', {'code'=>$code,'copy'=>$copy}))[0];
  my $from = $$entry{name} || "";
  if ( $from ) {
    unless ($confirm) {
      my @return = "<h2>Really take $code from '$from'?</h2>";
      push @return, "<h3>", 
	$self->linkback('Yes', { 'copy' => $copy, 'confirm' => 1, 
				 'code' => $code, 'action' => 'steal',
				 'user' => $user } ), " | ", 
	$self->linkback('No',  { 'user' => $user } ), "</h3>";
      wantarray ? @return : join("\n", @return);
    } else { 
      
      if ( $self->delete('Library', { 'code' => $code, 'copy' => $copy,
                                                  'confirm' => 1 } ) 
          && $self->insert('Library', { 'code' => $code, 'copy' => $copy, 
                      'name' => $user, 'borrowdate' => scalar(localtime) } ) ) {
        my @return = "<h1>Took and checked out $code</h1>\n";
        _mail($from, "Took your book - $code", $user,
      "I have taken your book $code in the Group Library Database.",
      "If you want it back, to go http://www.ks.uiuc.edu/Group/InfoWeb/Library",
      "                    - $user" )
                && push @return, "<h3> Mail sent to $from </h3>\n";
        wantarray ? @return : join("\n", @return);
      } else {
        return "<h1>Couldn't check out $code</h1> <h3>@{[ DBI->errstr ]}</h3>";
      }
    }
  }
}

=back

=cut

###############################################################################
### Internal Functions ########################################################
###############################################################################

### _mail ( TO, SUBJECT, FROM, BODY )
# Creates an email using Mail::Send, and sends it.  Simple and effective,
# but not particularly robust.
sub _mail {
  my ($to, $subject, $from, @body) = @_;
  return undef unless ($to && $from && $subject && scalar @body);
  my $message = new Mail::Send;    
  $message->to($to);
  $message->subject($subject);
  $message->cc($from);
  $message->set('from', $from);
  my $fh = $message->open();
  print $fh join("\n", @body);
  $fh->close;
  1;
}

1;

=head1 NOTES

Most of these functions are designed to work with B<library_user.cgi>,
included.  

=head1 REQUIREMENTS

Perl 5.6.1 or better, DBIx::Frame 1.05 or better, TCB::Library,
Mail::Send, Date::Parse.

=head1 SEE ALSO

B<TCB::Library>, B<DBIx::Frame>, B<DBIx::Frame::CGI>, B<CGI>.

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE
                                                                                
B<http://www.ks.uiuc.edu/Development/MDTools/tcb-library/>
 
=head1 LICENSE
 
This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-library/license.html> for
details.
 
=head1 COPYRIGHT
 
Copyright 2003-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v1.0		Tue Oct 21 15:03:50 CDT 2003 
### Commented and ready for distribution.
# v1.1		Wed May 19 14:05:33 CDT 2004 
### Mostly documentation fixes, but also modified html() to use <div> tags
### instead of <table> stuff.
