$VERSION = "1.1";
package TCB::Library::Book;
our $VERSION = "1.1";

# -*- Perl -*- 		Sat May 15 07:58:31 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2003-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME

TCB::Library::Book - the table that describes a 'book'

=head1 SYNOPSIS

  use TCB::Library::Book;

See TCB::Library for more information.

=head1 DESCRIPTION

The 'Book' table describes an actual book in the library - that is, a
single item that can be checked out.  There may be several copies of this
single item, however, and it may not necessarily be an actual book.  

This table contains the following fields:
  
 Internal Information 
  ID            Unique numeric ID - auto-generated (INT)
 
 Basic Information (TINYTEXT fields, unless noted)
  code		TB Code of the book - generated by combining the first
		four letters of the author's last name and the year of 
		publication, with collisions handled by adding 'A-Z' 
		at the end.  Hand-generated, this is the unique 
		identifier for a book.  
  author	Author of the book.
  title		Title of the book.
  media		Type of media (book, magazine, filmstrip, etc).
  year		Year of publication.
  keyw		Keywords describing the book.
  numcopy	Number of copies of the book.
  room		Room that the book is stored in.  Note that there is no
		way to deal with two copies being stored in different 
		rooms.
  status	Used to keep track of the book's status - ie, is it 
		missing?  Not generally used.
  publ		Publisher of the book
  kscomment	Comments from Klaus (TEXT).

Key fields:	code

List items:	code, title, author

Required fields: code, title, author, numcopy

Default order:   code, author, title

The 'Library' table depends on this table.

Doesn't depend on any other table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $VERSION $BIBFILE $ABSTRACTS
             $REQUIRED $ADMIN $ORDER $WEBROOT @PUBTYPES @PUBSTATUS ); 
use strict;
use warnings;
use CGI;
use TCB::Library;
 
push @ISA, qw( TCB::Library );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Book";
$FIELDS = {
   
  # Non User-Modified Information
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',

  'code'      =>  'TINYTEXT NOT NULL', 'media' => 'TINYTEXT',
  'author'    =>  'TINYTEXT', 'title'     =>  'TINYTEXT',
  'year'      =>  'TINYTEXT', 'room'      =>  'TINYTEXT',
  'keyw'      =>  'TINYTEXT', 'numcopy'   =>  'TINYTEXT',
  'status'    =>  'TINYTEXT', 'kscomment' =>  'TEXT',
  'publ'      =>  'TINYTEXT'
	 };
$KEYS  =  [ 'code' ];
$LIST  =  [ 
  { 'Code'  => '$$code$$' },
  { 'Title' => '$$title$$' },
  { 'Author' => '$$author$$' },
          ];

# New fields - require the new version of DBIx::Frame
$REQUIRED = [ 'code', 'title', 'author', 'numcopy' ];
$ADMIN    = [ ];
$ORDER    = [ 'code', 'author', 'title' ];

############################################################################### ##### Functions ###############################################################
###############################################################################

=head2 Internal Functions

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;

  $entry ||= {};  $options ||= {};
  if (lc $type eq 'create') {
    $$entry{'year'} ||= (localtime)[5] + 1900;
    $$entry{'room'} ||= 3151;
    $$entry{'status'} ||= 'present';
    $$entry{'numcopy'} ||= 1;
  } elsif (lc $type eq 'search') {
  }

  my $extra = lc $type eq 'search' ? { '*' => "" } : { };

  my @media = qw( book tape journal other );
  my @years_since = years_since( 1900 );
  my %status = ( "" => 'unknown/any', 
		 'present' => 'present',
		 'missing' => 'missing' );

  if (lc $type eq 'search') {
    unshift @media, "";
    unshift @years_since, "";
    $status{''} ||= "";
  }

  my @return = <<HTML;
<div class="basetable">
 <div class="row2">
  <span class="label">Title</span>
  <span class="formw"> @{[ $cgi->textfield('title',
                                $$entry{title} || "", 30, 255) ]} </span>
  <span class="label">Book Code</span>
  <span class="formw">
     @{[ $cgi->textfield('code', $$entry{code} || "", 8, 255) ]} 
  </span>
  <span class="label">Media Type</span>
  <span class="formw">
     @{[ $cgi->popup_menu('media', \@media, $$entry{media} || "") ]}
  </span>
 </div>

 <div class="row2">
  <span class="label">Author</span>
  <span class="formw">
   @{[ $cgi->textfield('author', $$entry{author} || "", 30, 255) ]} 
  </span>
  <span class="label">Publisher</span>
  <span class="formw">
    @{[ $cgi->textfield('publ', $$entry{publ} || "", 25, 255) ]} 
  </span>
 </div>

 <div class="row1">
  <span class="label">Key Words</span>
  <span class="formw">
    @{[ $cgi->textfield('keyw', $$entry{keyw} || "", 70, 255) ]} 
  </span>
 </div>

 <div class="row3">
  <span class="label">Year</span>
  <span class="formw">
    @{[ $cgi->popup_menu('year', \@years_since, $$entry{year} || "") ]}
  </span>
  <span class="label">Room</span>
  <span class="formw">
     @{[ $cgi->textfield('room', $$entry{room} || "", 5, 255) ]}
  </span>
  <span class="label">Copies</span>
  <span class="formw">
     @{[ $cgi->textfield('numcopy', $$entry{numcopy} || "", 2, 255) ]} 
  </span>
 </div>

 <div class="row1">
  <span class="label">Comment</span>
  <span class="formw"> 
   @{[ $cgi->textarea(-name=>'kscomment', -default=>$$entry{kscomment} || "", 
			-rows=>7, -cols=>60, -maxlength=>65535) ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

=item years_since ( START [, END] )

Returns an array of years since C<START>, ending at C<END> (which defaults
to the current year if not set).  Used by html().

=cut

sub years_since {
  my ($start, $end) = @_;
  $end ||= (localtime)[5] + 1900;
  return undef unless ($start <= $end);
  my @return;
  for (my $i = $end ; $i >= $start; $i--) { push (@return, $i); } 
  @return;
}

=back

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::Library->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, 
                          $ADMIN, $REQUIRED, \&html, \&text);

=head1 NOTES

=head1 TODO

=head1 REQUIREMENTS

Perl 5.6.1 or better, DBIx::Frame 1.05 or better, TCB::Library.

=head1 SEE ALSO

B<TCB::Library>, B<TCB::Library::Library>, B<TCB::Library::Functions>,
B<DBIx::Frame>, B<DBIx::Frame::CGI>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-library/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-library/license.html> for
details.

=head1 COPYRIGHT

Copyright 2003-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v1.0		Tue Oct 21 14:51:01 CDT 2003 
### Getting ready for release, now using DBIx::Frame.
# v1.1		Sat May 15 07:58:24 CDT 2004 
### Mostly documentation fixes, but also modified html() to use <div> tags
### instead of <table> stuff.
