$VERSION = "1.1";
package TCB::Library::Library;
our $VERSION = "1.1";

# -*- Perl -*-  	Wed May 19 13:56:18 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2003-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME

TCB::Library::Library - the table that describes what books are checked out

=head1 SYNOPSIS
 
  use TCB::Library::Library;

See TCB::Library for more information.

=head1 DESCRIPTION

The 'Library' table describes which books are checked out by which people.

This table contains the following fields:

 Internal Information 
  ID            Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  code		Same as the 'code' field from Book.  
  copy		The copy number checked out.
  name		Name of the borrower; generally a username.
  borrowdate	Date that the book was borrowed.  Currently free-form, 
		it would be better if it had been a DATE field.

Key fields: 	code, copy

List items: 	code, copy, borrower

Required fields:  code, copy

Default order:	name, borrowdate

Depends on the 'Book' table.

No table depends on this table directly.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $VERSION $BIBFILE $ABSTRACTS
             $REQUIRED $ADMIN $ORDER $WEBROOT @PUBTYPES @PUBSTATUS );
use strict;
use warnings;
use CGI;
use TCB::Library; 
use TCB::Library::Functions qw( &get_username );

push @ISA, qw( TCB::Library );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Library";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'copy' => 'INT',  'code' => 'TINYTEXT',  'name' => 'TINYTEXT',
  'borrowdate' => 'TINYTEXT',
	 };
$KEYS  =  [ 'code', 'copy' ];
$LIST  =  [ 
  { 'Code' => '$$code$$'}, 
  { 'Copy' => '$$copy$$'},
  # { 'Borrower' => \&get_username }, 
  { 'Borrower' => \&TCB::Library::get_username }, 
  { 'Borrow Date' => \&TCB::Library::borrowdate } 
          ];
$REQUIRED = [ 'code', 'copy' ];
$ADMIN = [];
$ORDER = [ 'name', 'borrowdate' ];

############################################################################### 
### Functions #################################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI; $entry ||= {};  $options ||= {};

  if (lc $type eq 'create') {
    $$entry{'borrowdate'} ||= scalar(localtime);
    $$entry{'name'}       ||= 'kschulte';
  } elsif (lc $type eq 'search') { }

  my $extra = lc $type eq 'search' ? { '*' => "" } : { };
  
  my @users;
  while ( my @list = getpwent() ) {
    next unless ($list[3] =~ /^100\d$/);	# actual TCB group members
    push @users, $list[0] unless ($list[1] eq '*');
  }
  @users = sort @users;

  my @copy = (1..9);

  if (lc $type eq 'search') {
    unshift @copy, "";
    unshift @users, "";
  } else { unshift @users, $$entry{name}; }

  my @return = <<HTML;
<div class="basetable">
 <div class="row3">
  <span class="label">TB Code</span>
  <span class="formw"> 
   @{[ $cgi->textfield('code', $$entry{code} || "", 8, 255) ]}
  </span>
  <span class="label">Copy</span>
  <span class="formw">
   @{[ $cgi->popup_menu('copy', \@copy, $$entry{copy} || "") ]}
  </span>
  <span class="label">Borrower</span>
  <span class="formw">
     @{[ $cgi->textfield('name', $$entry{name} || "", 20, 255) ]}
  </span>
 </div>
 <div class="row1">
  <span class="label">Borrow Date</span>
  <span class="formw">
     @{[ $cgi->textfield('borrowdate', $$entry{borrowdate} || "", 40, 255) ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>
</div>
HTML
  wantarray ? @return : join("\n", @return);
}

=back

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::Library->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER,
                          $ADMIN, $REQUIRED, \&html, \&text);

=head1 NOTES

=head1 TODO

=head1 REQUIREMENTS

Perl 5.6.1 or better, DBIx::Frame, TCB::Library.

=head1 SEE ALSO

B<TCB::Library>, B<DBIx::Frame>, B<DBIx::Frame::CGI>

=head1 AUTHOR
                                                                                
Tim Skirvin <tskirvin@ks.uiuc.edu>
                                                                                
=head1 HOMEPAGE
                                                                                
B<http://www.ks.uiuc.edu/Development/MDTools/tcb-library/>
 
=head1 LICENSE
 
This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-library/license.html> for
details.
 
=head1 COPYRIGHT
 
Copyright 2003-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v1.0		Tue Oct 21 15:47:48 CDT 2003 
### Getting ready for release, now using DBIx::Frame.
# v1.1		Wed May 19 13:55:30 CDT 2004 
### Mostly documentation fixes, but also modified html() to use <div> tags
### instead of <table> stuff.
