$VERSION = "1.00";
package TCB::System::Help;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 16:30:17 CDT 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Help - TCB Group Help Database

=head1 SYNOPSIS

  use TCB::System::Help;

See B<TCB::System> for more information on how to use this module.

=head1 DESCRIPTION

The 'Help' table is used to list what technical help members of the TCB
Group have offered to other groups or institutions.  It is used as part of
our annual reporting process.

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Person	Name and email address of the helped person.
  Affiliation	Affiliation of the helped person
  Date		Date of start of the offered help (DATE)
  Summary	Summary of the help
  Helper	Name of the helping person(s).
  Request	Full text of the original request.  (TEXT)
  Offered	Full text of the actual help. (TEXT)
  Notes		Any extra notes. (TEXT)
  
Key fields: 	Person, Date

List items: 	Requesting Person (Person + Affiliation), Summary, Date

Required: 	Same as Key fields.

Default order: 	Date (reverse)

Admin Fields:	None

No other tables depend on this table directly.

Doesn't depend on any other table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Help";
$FIELDS = {
  'ID'          =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Person'   => 'TINYTEXT',       'Affiliation'    => 'TINYTEXT', 
  'Date'     => 'DATE',  'Summary' => 'TINYTEXT', 'Helper' => 'TINYTEXT',
  'Request'  => 'TEXT',  'Offered' => 'TEXT',  '  Notes' => 'TEXT'
         };
$KEYS  = [ 'Person', 'Date' ];
$LIST  = [ { 'Requesting Person' => '$$Person$$, $$Affiliation$$' },
                'Summary', 'Date' ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ '-Date' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;
  my $date = sprintf("%04d-%02d-%02d", (localtime)[5] + 1900, 
		  		       (localtime)[4] + 1, (localtime)[3]);
  if (lc $type eq 'create') {
    $entry ||= { 'Date' => $date };
  } else { $entry ||= {}; }
  
  my @return = <<HTML;
<div class="basetable">
 <div class="row2">
  <span class="label">Helped Person</span>
  <span class="formw"> @{[ $cgi->textfield('Person',
				$$entry{Person} || "", 35, 255) ]} </span>
  <span class="label">Request Summary</span>
  <span class="formw"> @{[ $cgi->textfield('Summary',
				$$entry{Summary} || "", 35, 255) ]} </span>
 </div>

 <div class="row3">
  <span class="label">Affiliation</span>
  <span class="formw"> @{[ $cgi->textfield('Affiliation',
				$$entry{Affiliation} || "", 30, 255) ]} </span>
  <span class="label">Date of Request</span>
  <span class="formw"> @{[ $cgi->textfield('Date',
				$$entry{Date} || "", 11, 255) ]} </span>
  <span class="label">Helper</span>
  <span class="formw"> @{[ $cgi->textfield('Helper',
				$$entry{Helper} || "", 15, 255) ]} </span>
 </div>

 <div class="row1">
  <span class="label">Help Offered</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Offered', -default=>$$entry{Offered} || "", 
		        -rows=>5, -cols=>70, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Request (Detailed)</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Request', -default=>$$entry{Request} || "", 
		        -rows=>5, -cols=>70, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "", 
		        -rows=>5, -cols=>70, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

This table was created by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::Equipment>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-system>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-help/license.html> for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 16:30:17 CDT 2004
### Ready to be a part of DBIx::Frame.
