$VERSION = "1.00";
package TCB::System::Package;
our $VERSION = "1.00";

# -*- Perl -*- 		Thu May 13 15:30:01 CDT 2004
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.
# Redistribution terms are below.
###############################################################################

=head1 NAME 

TCB::System::Package - Licensed Software Packages Table

=head1 SYNOPSIS

  use TCB::System::Package;

See B<TCB::Equipment> or B<TCB::System> for more information on how to use
this module.

=head1 DESCRIPTION

The 'License' table describes the licenses used by the TCB Group, both in
quantities and duration.  It is far more convenient to use this than the
equipment database for these types of information

This table contains the following fields:

 Internal Information
  ID		Unique numeric ID - auto-generated (INT)

 Basic Information (TINYTEXT fields, unless noted)
  Program	Name of the program 
  Version	Version of the program
  OS		Operating system the program runs on
  Description	A short description of the package.
  URL		URL describing the program
  PO		PO or other order number the package was purchased with
  Copies	Number of copies of the program (INT)
  SerialNumber  Serial number of the package.
  Notes		Notes about the package (TEXT)

Key fields: 	Program, Version, OS, Copies

List items: 	Program (Program, Version, OS), Description, Copies

Required: 	Same as Key fields.

Default order:  Program, Version, OS

Admin Fields:	None

No other tables depend on this table directly.

The 'Package' table depends on this table.

=cut

###############################################################################
### Initialization ############################################################
###############################################################################
use vars qw( @ISA $FIELDS $KEYS $NAME $LIST $BIBFILE $ABSTRACTS $REQUIRED 
	     $ADMIN $ORDER $WEBROOT );
use strict;
use warnings;
use CGI;
use TCB::System;

push @ISA, qw( TCB::System );

###############################################################################
### Database Variables ########################################################
###############################################################################
$NAME = "Package";
$FIELDS = {
  'ID'            =>  'INT NOT NULL PRIMARY KEY AUTO_INCREMENT',
  'Program'       =>  'TINYTEXT',   'Version'     => 'TINYTEXT',
  'OS'            =>  'TINYTEXT',   'URL'         => 'TINYTEXT',   
  'Notes'         =>  'TEXT',       'SerialNumber' =>  'TINYTEXT',   
  'PO'            =>  'TINYTEXT',   'Copies'      => 'INT',      
  'Description'   => 'TINYTEXT',    
         };
$KEYS  =  [ 'Program', 'Version', 'OS', 'Copies' ];
$LIST  =  [ { 'Program' => '$$Program$$ $$Version$$ ($$OS$$)' },
  		'Description', 'Copies' ];
$REQUIRED = $KEYS;
$ADMIN = [ ];
$ORDER = [ 'Program', 'Version', 'OS' ];

###############################################################################
##### Functions ###############################################################
###############################################################################

=head2 Internal Functions 

=over 4

=item html ( ENTRY, TYPE, OPTIONS )

Prints the HTML version of the table.  Heavily customized for the TCB
environment, but then again so is the whole program.                            

=cut

sub html {
  my ($self, $entry, $type, $options, @rest) = @_;
  my $cgi = new CGI;	$entry ||= {};

  my @return = <<HTML;
<div class="basetable">
 <div class="row3">
  <span class="label">Program</span>
  <span class="formw"> 
   @{[ $cgi->textfield('Program', $$entry{Program} || "", 20, 255) ]}
  </span>
  <span class="label">Version</span>
  <span class="formw"> 
   @{[ $cgi->textfield('Version', $$entry{Version} || "", 12, 255) ]}
  </span>
  <span class="label">OS</span>
  <span class="formw"> 
   @{[ $cgi->textfield('OS', $$entry{OS} || "", 10, 255) ]}
  </span>
 </div>
 
 <div class="row1">
  <span class="label">Description</span>
  <span class="formw">
    @{[ $cgi->textfield('Description', $$entry{Description} || "", 60, 255) ]}
  </span> 
 </div>

 <div class="row2">
  <span class="label">PO</span>
  <span class="formw">
    @{[ $cgi->textfield('PO', $$entry{PO} || "", 50, 255) ]}
  </span> 
  <span class="label">Copies</span>
  <span class="formw">
    @{[ $cgi->textfield('Copies', $$entry{Copies} || "", 3, 4) ]}
  </span> 
 </div>

 <div class="row2">
  <span class="label">URL</span>
  <span class="formw">
    @{[ $cgi->textfield('URL', $$entry{URL} || "", 30, 255) ]}
  </span> 
  <span class="label">Serial Number</span>
  <span class="formw">
    @{[ $cgi->textfield('SerialNumber', $$entry{SerialNumber} || "", 25, 255) ]}
  </span> 
 </div>

 <div class="row1">
  <span class="label">Notes</span>
  <span class="formw">
     @{[ $cgi->textarea(-name=>'Notes', -default=>$$entry{Notes} || "", 
		        -rows=>10, -cols=>60, -maxlength=>65535,
			-wrap=>'physical') ]}
  </span>
 </div>

 <div class="submitbar"> @{[ $cgi->submit(-name=>"Submit") ]} </div>

</div>

HTML
  wantarray ? @return : join("\n", @return);
}

=back

There is no text() field at present.

=cut

###############################################################################
##### main() ##################################################################
###############################################################################

TCB::System->table_add($NAME, $FIELDS, $KEYS, $LIST, $ORDER, $ADMIN, 
			$REQUIRED, \&html, \&text);

=head1 NOTES

The fundamentals of this table were created before Charlie Brown, before
my time.  It was originally an msql table containing the same fields as
above; it's worked out pretty well, I suppose.  

=head1 TODO

Making a nice user interface, rather than just an administrative
interface, would be helpful.

=head1 REQUIREMENTS

Perl 5.6.1 or greater, DBIx::Frame, TCB::System

=head1 SEE ALSO

B<TCB::Equipment>, B<TCB::System>, B<DBIx::Frame>

=head1 AUTHOR

Tim Skirvin <tskirvin@ks.uiuc.edu>

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-equipment/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-equipment/license.html>
for details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.2  	Tue Apr  2 15:35:25 CST 2002
### First version fully using DBI::Frame and commented.
# v1.00         Thu May 13 15:30:01 CDT 2004
### Ready to be a part of DBIx::Frame.
