$VERSION = "0.03";
package TCB::Backup;
# -*- Perl -*-          Fri Dec 10 10:07:56 CST 2004 
###############################################################################
# Written by Tim Skirvin <tskirvin@ks.uiuc.edu>
# Copyright 2000-2004, Tim Skirvin and UIUC Board of Trustees.  
# Redistribution terms are below.
###############################################################################
our $VERSION = "0.03";

=head1 NAME

TCB::Backup - parent perl package for backup scripts in the TCB Group

=head1 SYNOPSIS

  use TCB::Backup;

See the included scripts to see how it really works.

=head1 DESCRIPTION

TCB::Backup is a package meant to simplify and consolidate backup scripts
used in the TCB Group at the University of Illinois.  These backups
generally consist of two types:

=over 2

=item ufsdump - nightly + monthly

ufsdump is the Solaris 'dump' utility, which is used to simply write all
data that has changed since a given time to a file or tape.  It is used in
the TCB Group to run nightly dumps to disk, as well as monthly to tape.
These backups can then be easily restored using 'ufsrestore'.

=item GNU Tar - twice-annual

These backups, using GNU tar, are used as archives.  The last four
archives are saved, and one of every four is saved indefinitely.

=back

These scripts were originally put together piece-meal, but a more
consistent approach should make maintenance simpler.  

=head2 VARIABLES

=over 2

=item $GZIP

Location of the 'gzip' binary.  Defaults to '/usr/local/bin/gzip'.

=back

=cut

use strict;
use Exporter;
use vars qw( $QUIET $VERBOSE $TESTING $GZIP @ISA @EXPORT @EXPORT_OK );

###############################################################################
### User Variables ############################################################
###############################################################################

## Location of your 'gzip' variable?  Must be set on a per-system basis.

$GZIP = "/usr/local/bin/gzip";

###############################################################################
### Subroutines ###############################################################
###############################################################################

=head2 SUBROUTINES

=over 2

=item rotate ( FILE, TOKEEP )

Rotates C<FILE> by renaming it to C<FILE.1>, after renaming C<FILE.1> to
C<FILE.2>, etc.  Keeps C<TOKEEP> counts of that file; removes extras.
Used to keep previous iterations of the same filename.

=cut

sub rotate   {
  my ($self, $file, $tokeep) = @_;
  for (my $i = $tokeep; $i >= 0; $i--) {
    my $next = $i + 1;
    next unless (-f "$file.$i");
    if ($tokeep <= $i) {
      warn "  Removing $file.$i\n"  unless $QUIET;
      unlink("$file.$i") unless $TESTING;
    } else {
      warn "  Renaming $file.$i to $file.$next\n" unless $QUIET;
      rename("$file.$i", "$file.$next") unless $TESTING;
    }
  }
  if (-f $file) {
    if ($tokeep > 0) {
      warn "  Renaming $file to $file.1\n" unless $QUIET;
      rename( $file, "$file.1" ) unless $TESTING;
    } else {
      warn "  Removing $file\n" unless $QUIET;
      unlink ($file) unless $TESTING;
    }
  }
}

=item rotategz ( FILE, TOKEEP )

Same as C<rotate()>, but compresses the file as well with 'gzip'.
Otherwise acts the same as other system.

=cut

sub rotategz {
  my ($file, $tokeep) = @_;
  for (my $i = $tokeep; $i >= 0; $i--) {
    my $next = $i+1;
    next unless (-f "$file.$i.gz");
    if ($tokeep <= $i) {
      warn "  Removing $file.$i.gz\n" if $VERBOSE;
      unlink("$file.$i") unless $TESTING;
    } else {
      warn "  Renaming $file.$i.gz to $file.$next.gz\n" if $VERBOSE;
      rename("$file.$i.gz", "$file.$next.gz") unless $TESTING;
    }
  }
  if (-f $file) {
    if ($tokeep > 0) {
      warn "  Renaming $file to $file.1\n" if $VERBOSE;
      rename( $file, "$file.1" ) unless $TESTING;
      warn "  Gzipping '$file.1'\n" if $VERBOSE;
      system("$GZIP $file.1") unless $TESTING;
    } else {
      warn "  Removing $file\n" if $VERBOSE;
      unlink ($file) unless $TESTING;
    }
  }
}

=back

There will be additional subroutines over time.

=cut

###############################################################################
### NOT YET COMPLETE ##########################################################
###############################################################################

sub new {
  my ($class, $program, $output, %args) = @_;
  my $self = {};  
#  if (ref $program
  $$self{PROGRAM} = $program || [];
  bless $self, $class;
  $self;
}

sub is_open {
  my ($self, $filesystem, $fssnap, @options) = @_;
  return undef unless (-x $fssnap);
  
  my $cmd = "$fssnap -i @options $filesystem";
  open (CMD, "$cmd |" ); 
  close CMD;
}

sub lock   { 1; }
sub unlock { 1; }
sub locked { 0; }

sub writegz  {
  my ($self, $fh, $file) = @_;
  open (FILE, "| $GZIP > $file") 
		   or return $self->_error("Couldn't write to $file: $!\n");
  while (<$fh>) { print FILE } 
  close FILE;
  return 1;
}
sub write    {}

sub _error {
  my ($self, $line) = @_;
  warn "$line\n" unless $self->quiet; 
  undef;
}

sub quiet { $QUIET }

sub _date {
  my $time = shift || time;
  my $day  = (localtime($time))[3];
  my $monthnum = (localtime($time))[4];
  my $month = (qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec))[$monthnum];
  my $year = (localtime($time))[5] + 1900;
  sprintf("%02d %03s %04d", $day, $month, $year);
}

1;

=head1 NOTES

This package is still a work in progress, but it should grow more useful
over time as all backup techniques are moved to here and the associated
scripts are simplified.  

=head1 SEE ALSO

B<ufsdump.config>, B<tardump.config>, B<snapshot.daily>, B<snapshot.0>,
B<snapshot.1>, B<snapshot.gtar>, B<dailybackup>, B<gtarbackup>

=head1 AUTHOR

Written by Tim Skirvin <tskirvin@ks.uiuc.edu>.

=head1 HOMEPAGE

B<http://www.ks.uiuc.edu/Development/MDTools/tcb-backup/>

=head1 LICENSE

This code is distributed under the University of Illinois Open Source
License.  See
C<http://www.ks.uiuc.edu/Development/MDTools/tcb-backup/license.html> for
details.

=head1 COPYRIGHT

Copyright 2000-2004 by the University of Illinois Board of Trustees and
Tim Skirvin <tskirvin@ks.uiuc.edu>.

=cut

###############################################################################
### Version History ###########################################################
###############################################################################
# v0.03		Fri Dec 10 10:19:56 CST 2004 
### First version with any documentation.
