/*
 * Copyright (C) 2005 by David J. Hardy.  All rights reserved.
 *
 * verlet.c  - Compute velocity Verlet (leapfrog) integration.
 */

#include <stdlib.h>
#include "step/step_defn.h"
#include "debug/debug.h"


int step_init_verlet(Step *s)
{
  /* check options? */
  return STEP_SUCCESS;
}


void step_done_verlet(Step *s)
{
  return;  /* nothing to do! */
}


int step_compute_verlet(Step *s, int32 numsteps)
{
  const double dt = s->param->timestep;
  const double half_dt = 0.5 * dt;
  MD_Dvec *f = s->system->force;
  MD_Dvec *vel = s->system->vel;
  MD_Dvec *pos = s->system->pos;
  const double *scal_inv_mass = s->scal_inv_mass;
  const int32 natoms = s->param->natoms;
  const int32 resultsFreq = s->param->resultsFreq;
  const int32 doRigidWaters = s->doRigidWaters;
  int32 n, i, resultsCounter;

  if (step_output(s, "Running velocity-Verlet for %d steps...\n", numsteps)) {
    return STEP_FAILURE;
  }

  if (doRigidWaters) {
    /* settle startup also computes initial force */
    if (step_settle_startup(s)) return STEP_FAILURE;
  }
  else {
    /* compute initial force */
    if (step_force(s)) return STEP_FAILURE;
  }

  /* results for step 0 */
  if (step_results(s, 0)) return STEP_FAILURE;

  /* perform velocity-Verlet integration for numsteps */
  resultsCounter = 0;
  for (n = 0;  n < numsteps;  n++) {

    /* half kick */
    for (i = 0;  i < natoms;  i++) {
      double dt_2m = half_dt * scal_inv_mass[i];
      vel[i].x += dt_2m * f[i].x;
      vel[i].y += dt_2m * f[i].y;
      vel[i].z += dt_2m * f[i].z;
    }

    if (doRigidWaters && step_settle_prep(s)) return STEP_FAILURE;

    /* drift */
    for (i = 0;  i < natoms;  i++) {
      pos[i].x += dt * vel[i].x;
      pos[i].y += dt * vel[i].y;
      pos[i].z += dt * vel[i].z;
    }

    if (doRigidWaters && step_settle1(s, dt)) return STEP_FAILURE;

    /* compute force */
    if (step_force(s)) return STEP_FAILURE;

    /* half kick */
    for (i = 0;  i < natoms;  i++) {
      double dt_2m = half_dt * scal_inv_mass[i];
      vel[i].x += dt_2m * f[i].x;
      vel[i].y += dt_2m * f[i].y;
      vel[i].z += dt_2m * f[i].z;
    }

    if (doRigidWaters && step_settle2(s, dt)) return STEP_FAILURE;

    /* submit results? */
    resultsCounter++;
    if (resultsFreq == resultsCounter) {
      resultsCounter = 0;
      if (step_results(s, resultsFreq)) return STEP_FAILURE;
    }
  }

  return STEP_SUCCESS;
}
