/*
 * Copyright (C) 2005 by David J. Hardy.  All rights reserved.
 *
 * tempbath.c  - Couple system to an external temperature bath.
 */

#include <stdlib.h>
#include "step/step_defn.h"
#include "debug/debug.h"


int step_init_tempbath(Step *s)
{
  s->dt_tau = s->param->timestep / s->param->relaxation_time;
  s->twoke_bath = s->param->bath_temperature *
    2.0 * MD_ENERGY_CONST / s->tempkonst;
  return STEP_SUCCESS;
}

void step_done_tempbath(Step *s)
{
  return;  /* nothing to do! */
}


int step_compute_tempbath(Step *s, int32 numsteps)
{
  const double dt = s->param->timestep;
  const double half_dt = 0.5 * dt;
  const double dt_tau = s->dt_tau;
  const double twoke_bath = s->twoke_bath;
  double twoke;
  double scaling_factor = 1.0;
  MD_Dvec *f = s->system->force;
  MD_Dvec *vel = s->system->vel;
  MD_Dvec *pos = s->system->pos;
  const double *scal_inv_mass = s->scal_inv_mass;
  const MD_Atom *atom = s->param->atom;
  const int32 natoms = s->param->natoms;
  const int32 resultsFreq = s->param->resultsFreq;
  const int32 doRigidWaters = s->doRigidWaters;
  int32 n, i, resultsCounter;

  if (step_output(s, "Running Berendsen temperature bath for %d steps...\n",
        numsteps)) {
    return STEP_FAILURE;
  }

  if (doRigidWaters) {
    /* settle startup also computes initial force */
    if (step_settle_startup(s)) return STEP_FAILURE;
  }
  else {
    /* compute initial force */
    if (step_force(s)) return STEP_FAILURE;
  }

  /* results for step 0 */
  if (step_results(s, 0)) return STEP_FAILURE;

  /* backward half kick */
  for (i = 0;  i < natoms;  i++) {
    double konst = -half_dt * scal_inv_mass[i];
    vel[i].x = vel[i].x + konst * f[i].x;
    vel[i].y = vel[i].y + konst * f[i].y;
    vel[i].z = vel[i].z + konst * f[i].z;
  }

  /* perform integration for numsteps */
  resultsCounter = 0;
  for (n = 0;  n < numsteps;  n++) {

    /* T(n-1/2) = Temp( v(n-1/2) ) */
    step_results_twoke(&twoke, vel, atom, natoms);
    scaling_factor = sqrt(1.0 + dt_tau * (twoke_bath / twoke - 1.0));

    /* v(n+1/2), full kick using temperature bath coupling */
    for (i = 0;  i < natoms;  i++) {
      double konst = dt * scal_inv_mass[i];
      vel[i].x = scaling_factor * (vel[i].x + konst * f[i].x);
      vel[i].y = scaling_factor * (vel[i].y + konst * f[i].y);
      vel[i].z = scaling_factor * (vel[i].z + konst * f[i].z);
    }

    if (doRigidWaters && step_settle_prep(s)) return STEP_FAILURE;

    /* drift */
    for (i = 0;  i < natoms;  i++) {
      pos[i].x += dt * vel[i].x;
      pos[i].y += dt * vel[i].y;
      pos[i].z += dt * vel[i].z;
    }

    if (doRigidWaters && step_settle1(s, dt)) return STEP_FAILURE;

    /* compute force */
    if (step_force(s)) return STEP_FAILURE;

    /* submit results? */
    resultsCounter++;
    if (resultsFreq == resultsCounter) {
      resultsCounter = 0;

      /* report half-step temperature */
      s->system->kinetic_energy = 0.5 * MD_KCAL_MOL * twoke;
      s->system->temperature = s->tempkonst * s->system->kinetic_energy;

      /* forward half kick */
      for (i = 0;  i < natoms;  i++) {
        double konst = half_dt * scal_inv_mass[i];
        vel[i].x = vel[i].x + konst * f[i].x;
        vel[i].y = vel[i].y + konst * f[i].y;
        vel[i].z = vel[i].z + konst * f[i].z;
      }
      if (doRigidWaters && step_settle2(s, dt)) return STEP_FAILURE;

      if (step_results(s, resultsFreq)) return STEP_FAILURE;

      /* backward half kick */
      for (i = 0;  i < natoms;  i++) {
        double konst = -half_dt * scal_inv_mass[i];
        vel[i].x = vel[i].x + konst * f[i].x;
        vel[i].y = vel[i].y + konst * f[i].y;
        vel[i].z = vel[i].z + konst * f[i].z;
      }
    }
  } /* end loop */

  if (resultsCounter > 0) {
    /* forward half kick */
    for (i = 0;  i < natoms;  i++) {
      double konst = half_dt * scal_inv_mass[i];
      vel[i].x = vel[i].x + konst * f[i].x;
      vel[i].y = vel[i].y + konst * f[i].y;
      vel[i].z = vel[i].z + konst * f[i].z;
    }
    if (doRigidWaters && step_settle2(s, dt)) return STEP_FAILURE;
  }

  return STEP_SUCCESS;
}
