/*
 * step_results.c
 */

#include "step/step_defn.h"


int step_results(Step *s, int32 delta_stepnum)
{
  StepSystem *system = s->system;
  const MD_Dvec *pos = system->pos;
  const MD_Dvec *vel = system->vel;
  const MD_Atom *atom = s->param->atom;
  const int32 natoms = s->param->natoms;
  const int32 options = s->param->options;

  /* results callback */
  void *resultsObjPtr = s->param->resultsObjPtr;
  int (*submit_results)(void *, StepSystem *) = s->param->submit_results;

  double twoke;

  system->stepnum += delta_stepnum;

  /* compute energy reductions */
  if (s->param->method != STEP_TEMPBATH) {
    step_results_twoke(&twoke, vel, atom, natoms);
    system->kinetic_energy = 0.5 * MD_KCAL_MOL * twoke;
    system->temperature = s->tempkonst * system->kinetic_energy;
  }

  if (options & STEP_MOMENTUM) {
    step_results_linmo(&(system->linear_momentum), vel, atom, natoms);
    step_results_angmo(&(system->angular_momentum), pos, vel, atom, natoms);
  }

  if (options & STEP_VIRIAL) {
    const MD_Dvec *f = s->system->force;
    const double *scal_inv_mass = s->scal_inv_mass;
    MD_Dvec *half_vel = s->half_vel;
    const double half_dt = 0.5 * s->param->timestep;
    double kv0[VIRIAL_UPPER_LEN], kv1[VIRIAL_UPPER_LEN];
    int32 i;

    /* assume Verlet for now */

    /* half step back */
    for (i = 0;  i < natoms;  i++) {
      double dt_2m = half_dt * scal_inv_mass[i];
      half_vel[i].x = vel[i].x - dt_2m * f[i].x;
      half_vel[i].y = vel[i].y - dt_2m * f[i].y;
      half_vel[i].z = vel[i].z - dt_2m * f[i].z;
    }
    step_results_kvirial(kv0, half_vel, atom, natoms);

    /* half step forward */
    for (i = 0;  i < natoms;  i++) {
      double dt_2m = half_dt * scal_inv_mass[i];
      half_vel[i].x = vel[i].x + dt_2m * f[i].x;
      half_vel[i].y = vel[i].y + dt_2m * f[i].y;
      half_vel[i].z = vel[i].z + dt_2m * f[i].z;
    }
    step_results_kvirial(kv1, half_vel, atom, natoms);

    /* unit conversion */
    system->kinetic_virial[VIRIAL_XX] = 0.5 * MD_KCAL_MOL *
      (kv0[VIRIAL_UPPER_XX] + kv1[VIRIAL_UPPER_XX]);
    system->kinetic_virial[VIRIAL_XY] = 0.5 * MD_KCAL_MOL *
      (kv0[VIRIAL_UPPER_XY] + kv1[VIRIAL_UPPER_XY]);
    system->kinetic_virial[VIRIAL_XZ] = 0.5 * MD_KCAL_MOL *
      (kv0[VIRIAL_UPPER_XZ] + kv1[VIRIAL_UPPER_XZ]);
    system->kinetic_virial[VIRIAL_YX] = system->kinetic_virial[VIRIAL_XY];
    system->kinetic_virial[VIRIAL_YY] = 0.5 * MD_KCAL_MOL *
      (kv0[VIRIAL_UPPER_YY] + kv1[VIRIAL_UPPER_YY]);
    system->kinetic_virial[VIRIAL_YZ] = 0.5 * MD_KCAL_MOL *
      (kv0[VIRIAL_UPPER_YZ] + kv1[VIRIAL_UPPER_YZ]);
    system->kinetic_virial[VIRIAL_ZX] = system->kinetic_virial[VIRIAL_XZ];
    system->kinetic_virial[VIRIAL_ZY] = system->kinetic_virial[VIRIAL_YZ];
    system->kinetic_virial[VIRIAL_ZZ] = 0.5 * MD_KCAL_MOL *
      (kv0[VIRIAL_UPPER_ZZ] + kv1[VIRIAL_UPPER_ZZ]);
  }

  /* callback */
  if (submit_results(resultsObjPtr, system)) {
    return step_error(s, "submit_results() returned an error");
  }

  return STEP_SUCCESS;
}


/*
 * computes \sum_i m_i v_i^T v_i
 * (twice kinetic energy, no unit conversion)
 */
void step_results_twoke(double *e_sum,
    const MD_Dvec *v, const MD_Atom *atom, int32 natoms)
{
  int32 i;
  *e_sum = 0.0;
  for (i = 0;  i < natoms;  i++) {
    *e_sum += (v[i].x*v[i].x + v[i].y*v[i].y + v[i].z*v[i].z) * atom[i].m;
  }
}


/*
 * computes \sum_i m_i v_i
 * (linear momentum of system)
 */
void step_results_linmo(MD_Dvec *linmo,
    const MD_Dvec *v, const MD_Atom *atom, int32 natoms)
{
  int32 i;
  linmo->x = 0.0;
  linmo->y = 0.0;
  linmo->z = 0.0;
  for (i = 0;  i < natoms;  i++) {
    linmo->x += atom[i].m * v[i].x;
    linmo->y += atom[i].m * v[i].y;
    linmo->z += atom[i].m * v[i].z;
  }
}


/*
 * computes \sum r_i \cross m_i v_i
 * (angular momentum)
 */
void step_results_angmo(MD_Dvec *angmo,
    const MD_Dvec *r, const MD_Dvec *v, const MD_Atom *atom, int32 natoms)
{
  int32 i;
  angmo->x = 0.0;
  angmo->y = 0.0;
  angmo->z = 0.0;
  for (i = 0;  i < natoms;  i++) {
    angmo->x += atom[i].m * (r[i].y*v[i].z - r[i].z*v[i].y);
    angmo->y += atom[i].m * (r[i].z*v[i].x - r[i].x*v[i].z);
    angmo->z += atom[i].m * (r[i].x*v[i].y - r[i].y*v[i].x);
  }
}


/*
 * computes \sum_i m_i v_i v_i^T
 * (kinetic contribution to the virial, no unit conversion)
 */
void step_results_kvirial(double u_kv[VIRIAL_UPPER_LEN],
    const MD_Dvec *v, const MD_Atom *atom, int32 natoms)
{
  int32 i;
  u_kv[VIRIAL_UPPER_XX] = 0.0;
  u_kv[VIRIAL_UPPER_XY] = 0.0;
  u_kv[VIRIAL_UPPER_XZ] = 0.0;
  u_kv[VIRIAL_UPPER_YY] = 0.0;
  u_kv[VIRIAL_UPPER_YZ] = 0.0;
  u_kv[VIRIAL_UPPER_ZZ] = 0.0;
  for (i = 0;  i < natoms;  i++) {
    u_kv[VIRIAL_UPPER_XX] += atom[i].m * v[i].x * v[i].x;
    u_kv[VIRIAL_UPPER_XY] += atom[i].m * v[i].x * v[i].y;
    u_kv[VIRIAL_UPPER_XZ] += atom[i].m * v[i].x * v[i].z;
    u_kv[VIRIAL_UPPER_YY] += atom[i].m * v[i].y * v[i].y;
    u_kv[VIRIAL_UPPER_YZ] += atom[i].m * v[i].y * v[i].z;
    u_kv[VIRIAL_UPPER_ZZ] += atom[i].m * v[i].z * v[i].z;
  }
}
