/*
 * Copyright (C) 2006 by David J. Hardy.  All rights reserved.
 *
 * shadow.c  - Compute shadow Hamiltonian (Engle, Skeel, & Drees, 2005).
 *
 * Initialization and computation routines below modified from
 * source code implementation by Robert Engle.
 *
 * (IMH = interpolated modified Hamiltonian)
 */

#include <stdlib.h>
#include <string.h>
#include "step/step_defn.h"
#undef DEBUG_WATCH
#include "debug/debug.h"


/* begin IMH */

/* 
 * Initialize arrays that will be used by IMH computation routine.
 */
int step_init_shadow(Step *s)
{
  int32 natoms = s->param->natoms;

  /* 
   * Initializations for IMH variables 
   */
  int32 i, j, orderk1 = ORDERK1;

  int sizes[12] = {1, 3, 6, 10, 15, 21, 28, 36, 45, 55, 66, 78};
  /*
   * These data arrays contain the coefficients
   * that define the linear combination of terms
   * that make up each IMH.
   */
  double H2[1] = {1};
  double H4[3] = {1, -1.0/2.0, 2.0/3.0};
  double H6[6] = {1, -1.0, 27.0/20.0, 11.0/60.0, -9.0/20.0, 3.0/10.0};
  double H8[10] = {1, -3.0/2.0, 16.0/7.0, 13.0/21.0, -32.0/21.0, 36.0/35.0, 
		   -5.0/84.0, 22.0/105.0, -9.0/35.0, 4.0/35.0};
  double H10[15] = {1, -2, 125.0/36.0, 47.0/36.0, -125.0/36.0, 500.0/189.0, 
		    -11.0/36.0, 1625.0/1512.0, -250.0/189.0, 25.0/42.0, 
		    137.0/7560.0, -125.0/1512.0, 55.0/378.0, -5.0/42.0, 
		    5.0/126.0};
  double H12[21] = {1, -5.0/2.0, 54.0/11.0, 74.0/33.0, -72.0/11.0, 
		    125.0/22.0, -19.0/22.0, 141.0/44.0, -375.0/88.0, 
		    500.0/231.0, 29.0/220.0, -3.0/5.0, 325.0/308.0, 
		    -200.0/231.0, 45.0/154.0, -7.0/1320.0, 137.0/4620.0, 
		    -125.0/1848.0, 5.0/63.0, -15.0/308.0, 1.0/77.0};
  double H14[28] = {1, -3, 343.0/52.0, 535.0/156.0, -1715.0/156.0, 
		    3087.0/286.0, -145.0/78.0, 12691.0/1716.0, -3087.0/286.0, 
		    42875.0/6864.0, 1377.0/2860.0, -6517.0/2860.0, 
		    48363.0/11440.0, -8575.0/2288.0, 1225.0/858.0, 
		    -223.0/4290.0, 9947.0/34320.0, -343.0/520.0, 
		    1225.0/1584.0, -1225.0/2574.0, 147.0/1144.0, 11.0/7280.0, 
		    -343.0/34320.0, 959.0/34320.0, -875.0/20592.0, 
		    35.0/936.0, -21.0/1144.0, 7.0/1716.0};
  double H16[36] = {1, -7.0/2.0, 128.0/15.0, 73.0/15.0, -256.0/15.0, 
		    10976.0/585.0, -41.0/12.0, 1712.0/117.0, -2744.0/117.0, 
		    10976.0/715.0, 743.0/585.0, -3712.0/585.0, 
		    406112.0/32175.0, -43904.0/3575.0, 6860.0/1287.0, 
		    -31.0/130.0, 4896.0/3575.0, -104272.0/32175.0, 
		    128968.0/32175.0, -3430.0/1287.0, 3136.0/3861.0, 
		    37.0/1925.0, -28544.0/225225.0, 11368.0/32175.0, 
		    -1568.0/2925.0, 140.0/297.0, -896.0/3861.0, 112.0/2145.0, 
		    -761.0/1801800.0, 22.0/6825.0, -343.0/32175.0, 
		    1918.0/96525.0, -175.0/7722.0, 28.0/1755.0, -14.0/2145.0, 
		    8.0/6435.0};
  double H18[45] = {1, -4, 729.0/68.0, 1337.0/204.0, -1701.0/68.0, 
		    2592.0/85.0, -385.0/68.0, 17739.0/680.0, -3888.0/85.0, 
		    37044.0/1105.0, 1879.0/680.0, -9963.0/680.0, 
		    34668.0/1105.0, -37044.0/1105.0, 1000188.0/60775.0, 
		    -193.0/255.0, 20061.0/4420.0, -12528.0/1105.0, 
		    913752.0/60775.0, -666792.0/60775.0, 9261.0/2431.0, 
		    197.0/1820.0, -22599.0/30940.0, 52488.0/25025.0, 
		    -201096.0/60775.0, 186543.0/60775.0, -3969.0/2431.0, 
		    1008.0/2431.0, -419.0/61880.0, 26973.0/523600.0, 
		    -72252.0/425425.0, 38367.0/121550.0, -3969.0/11050.0, 
		    189.0/748.0, -252.0/2431.0, 243.0/12155.0, 
		    7129.0/61261200.0, -6849.0/6806800.0, 297.0/77350.0, 
		    -1029.0/121550.0, 2877.0/243100.0, -105.0/9724.0, 
		    7.0/1105.0, -27.0/12155.0, 9.0/24310.0};
  double H20[55] = {1, -9.0/2.0, 250.0/19.0, 484.0/57.0, -2000.0/57.0, 
		    30375.0/646.0, -497.0/57.0, 167125.0/3876.0, 
		    -212625.0/2584.0, 21600.0/323.0, 34167.0/6460.0, 
		    -9625.0/323.0, 88695.0/1292.0, -25920.0/323.0, 
		    185220.0/4199.0, -14917.0/7752.0, 46975.0/3876.0, 
		    -83025.0/2584.0, 192600.0/4199.0, -154350.0/4199.0, 
		    666792.0/46189.0, 2761.0/6783.0, -19300.0/6783.0, 
		    501525.0/58786.0, -417600.0/29393.0, 652680.0/46189.0, 
		    -381024.0/46189.0, 110250.0/46189.0, -831.0/18088.0, 
		    4925.0/13832.0, -564975.0/470288.0, 43740.0/19019.0, 
		    -6615.0/2431.0, 186543.0/92378.0, -165375.0/184756.0, 
		    9000.0/46189.0, 4861.0/2116296.0, -10475.0/529074.0, 
		    14985.0/198968.0, -53520.0/323323.0, 21315.0/92378.0, 
		    -882.0/4199.0, 875.0/7106.0, -2000.0/46189.0, 
		    675.0/92378.0, -671.0/21162960.0, 7129.0/23279256.0, 
		    -6849.0/5173168.0, 99.0/29393.0, -1029.0/184756.0, 
		    2877.0/461890.0, -875.0/184756.0, 10.0/4199.0, 
		    -135.0/184756.0, 5.0/46189.0};
  double H22[66] = {1, -5, 1331.0/84.0, 299.0/28.0, -1331.0/28.0, 
		    166375.0/2394.0, -89.0/7.0, 161051.0/2394.0, 
		    -166375.0/1197.0, 4492125.0/36176.0, 15797.0/1710.0, 
		    -94501.0/1710.0, 6355525.0/46512.0, -898425.0/5168.0, 
		    239580.0/2261.0, -1201.0/285.0, 2165537.0/77520.0, 
		    -1830125.0/23256.0, 4372335.0/36176.0, -239580.0/2261.0, 
		    195657.0/4199.0, 129805.0/108528.0, -2836361.0/325584.0, 
		    62523725.0/2279088.0, -12278475.0/253232.0, 
		    10681275.0/205751.0, -139755.0/4199.0, 45738.0/4199.0, 
		    -32891.0/162792.0, 3674891.0/2279088.0, 
		    -6422075.0/1139544.0, 74169975.0/6584032.0, 
		    -2894925.0/205751.0, 94017.0/8398.0, -22869.0/4199.0, 
		    45375.0/33592.0, 42131.0/2279088.0, -368687.0/2279088.0, 
		    6555175.0/10456992.0, -9283725.0/6584032.0, 
		    49005.0/24206.0, -847.0/442.0, 39809.0/33592.0, 
		    -15125.0/33592.0, 15125.0/176358.0, -4307.0/5697720.0, 
		    6469991.0/888844320.0, -2788445.0/88884432.0, 
		    40293.0/506464.0, -26983.0/205751.0, 24563.0/167960.0, 
		    -9317.0/83980.0, 3025.0/54264.0, -3025.0/176358.0, 
		    605.0/235144.0, 83711.0/9777287520.0, 
		    -81191.0/888844320.0, 78419.0/177768864.0, 
		    -8371.0/6584032.0, 3993.0/1646008.0, -539.0/167960.0, 
		    1507.0/503880.0, -1375.0/705432.0, 605.0/705432.0, 
		    -55.0/235144.0, 11.0/352716.0};
  double H24[78] = {1, -11.0/2.0, 432.0/23.0, 905.0/69.0, -1440.0/23.0, 
		    15972.0/161.0, -1635.0/92.0, 702.0/7.0, -35937.0/161.0, 
		    665500.0/3059.0, 12126.0/805.0, -76896.0/805.0, 
		    3865224.0/15295.0, -1064800.0/3059.0, 
		    24257475.0/104006.0, -953.0/115.0, 126376.0/2185.0, 
		    -378004.0/2185.0, 6355525.0/22287.0, -8085825.0/29716.0, 
		    6899904.0/52003.0, 45454.0/15295.0, -345888.0/15295.0, 
		    19489833.0/260015.0, -7320500.0/52003.0, 
		    118053045.0/728042.0, -41399424.0/364021.0, 
		    4024944.0/96577.0, -8315.0/12236.0, 1168245.0/208012.0, 
		    -8509083.0/416024.0, 62523725.0/1456084.0, 
		    -331518825.0/5824336.0, 230715540.0/4732273.0, 
		    -2515590.0/96577.0, 705672.0/96577.0, 176005.0/1872108.0, 
		    -131564.0/156009.0, 3674891.0/1092063.0, 
		    -25688300.0/3276189.0, 222509925.0/18929092.0, 
		    -55582560.0/4732273.0, 752136.0/96577.0, 
		    -313632.0/96577.0, 136125.0/193154.0, -2339.0/328440.0, 
		    126393.0/1820105.0, -1106061.0/3640210.0, 
		    1311035.0/1670214.0, -50132115.0/37858184.0, 
		    2117016.0/1391845.0, -30492.0/25415.0, 307098.0/482885.0, 
		    -81675.0/386308.0, 24200.0/676039.0, 58301.0/240253860.0, 
		    -51684.0/20021155.0, 588181.0/47322730.0, 
		    -506990.0/14196819.0, 98901.0/1456084.0, 
		    -2119392.0/23661365.0, 40194.0/482885.0, 
		    -26136.0/482885.0, 2475.0/104006.0, -4400.0/676039.0, 
		    594.0/676039.0, -6617.0/2883046320.0, 
		    83711.0/3123300180.0, -81191.0/567872760.0, 
		    78419.0/170361828.0, -75339.0/75716368.0, 
		    35937.0/23661365.0, -1617.0/965770.0, 4521.0/3380195.0, 
		    -4125.0/5408312.0, 605.0/2028117.0, -99.0/1352078.0, 
		    6.0/676039.0};

  /*
   * Sun compilers complained about an initialization
   * of double *coef using {H2,H4,...,H24}
   */
  double *coef[12];
  coef[0] = H2;
  coef[1] = H4;
  coef[2] = H6;
  coef[3] = H8;
  coef[4] = H10;
  coef[5] = H12;
  coef[6] = H14;
  coef[7] = H16;
  coef[8] = H18;
  coef[9] = H20;
  coef[10] = H22;
  coef[11] = H24;

  if (s->doRigidWaters) {
    return step_error(s,
	"use of bond constraints is incompatible with shadow Hamiltonian");
  }
  
  TEXT("allocating shadow memory");

  /* ORDERK1 is defined in step.h */
  s->orderk = ORDERK1 - 1; 

  s->coef = (double **) calloc( s->orderk, sizeof(double *) );
  if (NULL == s->coef) {
    return step_error(s, "calloc() failed to allocate s->coef");
  }

  for(i=0; i<s->orderk; i++) {
    s->coef[i] = (double *) calloc( sizes[i], sizeof(double) );
    if (NULL == s->coef[i]) {
      return step_error(s, "calloc() failed to allocate s->coef[%d]", i);
    }
  }

  for(i=0; i<s->orderk; i++) {
    for(j=0; j<sizes[i]; j++) {
      s->coef[i][j] = coef[i][j];
    }
  }

  /* use calloc to initialize these to 0's */
  s->sham = (double *)calloc( s->orderk, sizeof(double) );
  if (NULL == s->sham) {
    return step_error(s, "calloc() failed to allocate s->sham");
  }

  s->dif_q = (MD_Dvec **)calloc( orderk1, sizeof(MD_Dvec *) );
  if (NULL == s->dif_q) {
    return step_error(s, "calloc() failed to allocate s->dif_q");
  }

  s->dif_p = (MD_Dvec **)calloc( orderk1, sizeof(MD_Dvec *) );
  if (NULL == s->dif_p) {
    return step_error(s, "calloc() failed to allocate s->dif_p");
  }

  s->dif_beta = (double *)calloc( orderk1, sizeof(double) );
  if (NULL == s->dif_beta) {
    return step_error(s, "calloc() failed to allocate s->dif_beta");
  }

  for(i=1; i<orderk1; i++) { 
    s->dif_q[i] = (MD_Dvec *)calloc( natoms, sizeof(MD_Dvec) );
    if (NULL == s->dif_q[i]) {
      return step_error(s, "calloc() failed to allocate s->dif_q[%d]", i);
    }

    s->dif_p[i] = (MD_Dvec *)calloc( natoms, sizeof(MD_Dvec) );
    if (NULL == s->dif_p[i]) {
      return step_error(s, "calloc() failed to allocate s->dif_p[%d]", i);
    }
  }
  
  s->Atab = (double **)calloc( orderk1, sizeof(double *) );
  if (NULL == s->Atab) {
    return step_error(s, "calloc() failed to allocate s->Atab");
  }


  for(i=0; i<orderk1; i++) {
    s->Atab[i] = (double *)calloc( orderk1, sizeof(double) );
    if (NULL == s->Atab[i]) {
      return step_error(s, "calloc() failed to allocate s->Atab[%d]", i);
    }
  }

  s->old_f = (MD_Dvec *)calloc( natoms, sizeof(MD_Dvec) );
  if (NULL == s->old_f) {
    return step_error(s, "calloc() failed to allocate s->old_f");
  }

  return 0;
}


void step_done_shadow(Step *s)
{
  int32 i, orderk1 = ORDERK1;

  TEXT("freeing shadow memory");

  for (i = 0;  i < s->orderk;  i++) {
    free(s->coef[i]);
  }
  for (i = 1;  i < orderk1;  i++) {
    free(s->dif_q[i]);
    free(s->dif_p[i]);
  }
  for (i = 0;  i < orderk1;  i++) {
    free(s->Atab[i]);
  }
  free(s->coef);
  free(s->sham);
  free(s->dif_q);
  free(s->dif_p);
  free(s->dif_beta);
  free(s->Atab);
  free(s->old_f);
}


int step_compute_shadow(Step *s, int32 numsteps)
{
  const double dt = s->param->timestep;
  const double half_dt = 0.5 * dt;
  MD_Dvec *f = s->system->force;
  MD_Dvec *vel = s->system->vel;
  MD_Dvec *pos = s->system->pos;
  const double *scal_inv_mass = s->scal_inv_mass;
  const int32 natoms = s->param->natoms;
  const int32 resultsFreq = s->param->resultsFreq;
  int32 n, i, resultsCounter;

  /* shadow vars */
  const double half_inv_dt = 0.5 / dt;
  int32 j, k, m, p;
  const MD_Atom *atom = s->param->atom;
  double pe;  /* potential energy */

  int32 coefnum;
  int32 orderk1 = s->orderk+1;
  MD_Dvec u;
  double beta;

  double **Atab = s->Atab;

  MD_Dvec **dif_p = s->dif_p;
  MD_Dvec **dif_q = s->dif_q;
  double *dif_beta = s->dif_beta;
  MD_Dvec *old_f = s->old_f;
  MD_Dvec sum;
  double dot;

  if (step_output(s, "Running shadow Hamiltonian for %d steps...\n",
	numsteps)) {
    return STEP_FAILURE;
  }

  /*
   * The old AND new values of the force are needed 
   * to compute the new first difference of p's.
   */
  memset(old_f, 0, natoms*sizeof(MD_Dvec));

  /* compute initial force */
  if (step_force(s)) return STEP_FAILURE;
  pe = s->system->potential_energy;

  /* 
   * Update backward difference table (p portion only) 
   */
  for (j = 0;  j < natoms;  j++) {
    double konst = half_dt * scal_inv_mass[j];
    u.x = konst * (f[j].x + old_f[j].x);
    u.y = konst * (f[j].y + old_f[j].y);
    u.z = konst * (f[j].z + old_f[j].z);

    for (i = s->orderk;  i > 0;  i--) {
      sum.x = sum.y = sum.z = 0.0;
      for (k = 1;  k < i;  k++) {
        sum.x += dif_p[k][j].x;
        sum.y += dif_p[k][j].y;
        sum.z += dif_p[k][j].z;
      } 
      dif_p[i][j].x = u.x - sum.x;
      dif_p[i][j].y = u.y - sum.y;
      dif_p[i][j].z = u.z - sum.z;
    }
  }

  /* store shadow energies */
  s->system->shadow_energy[STEP_SHADOW_4] = s->sham[SHADOW4];
  s->system->shadow_energy[STEP_SHADOW_8] = s->sham[SHADOW8];
  s->system->shadow_energy[STEP_SHADOW_12] = s->sham[SHADOW12];
  s->system->shadow_energy[STEP_SHADOW_16] = s->sham[SHADOW16];
  s->system->shadow_energy[STEP_SHADOW_20] = s->sham[SHADOW20];
  s->system->shadow_energy[STEP_SHADOW_24] = s->sham[SHADOW24];
  /* results for step 0 */
  if (step_results(s, 0)) return STEP_FAILURE;

  /* perform leapfrog integration of system for numsteps */
  resultsCounter = 0;
  for (n = 0;  n < numsteps;  n++) {

    /* half kick */
    for (i = 0;  i < natoms;  i++) {
      double konst = half_dt * scal_inv_mass[i];
      vel[i].x += konst * f[i].x;
      vel[i].y += konst * f[i].y;
      vel[i].z += konst * f[i].z;
    }

    /* Update backward difference table (q portion only) */
    for (j = 0;  j < natoms;  j++) {
      double konst = dt * dt * scal_inv_mass[j];
      u.x = konst * f[j].x;
      u.y = konst * f[j].y;
      u.z = konst * f[j].z;

      for (i = s->orderk;  i > 1;  i--) {
        sum.x = sum.y = sum.z = 0.0;
        for (k = 2;  k < i;  k++) {
          sum.x += dif_q[k][j].x;
          sum.y += dif_q[k][j].y;
          sum.z += dif_q[k][j].z;
        }
        dif_q[i][j].x = u.x - sum.x;
        dif_q[i][j].y = u.y - sum.y;
        dif_q[i][j].z = u.z - sum.z;
      }

      dif_q[1][j].x = dt * vel[j].x;
      dif_q[1][j].y = dt * vel[j].y;
      dif_q[1][j].z = dt * vel[j].z;
    }

    /* Beta n+1/2*/
    dot = 0.0;
    for (i = 0;  i < natoms;  i++) {
      dot += pos[i].x*f[i].x + pos[i].y*f[i].y + pos[i].z*f[i].z;
    }
    beta = half_dt * MD_FORCE_CONST * (-dot - 2.0 * pe);

    /* drift */
    for (i = 0;  i < natoms;  i++) {
      pos[i].x += dt * vel[i].x;
      pos[i].y += dt * vel[i].y;
      pos[i].z += dt * vel[i].z;
    }

    /*
     * The old AND new values of the force are needed 
     * to compute the new first difference of p's.
     */
    for (j = 0;  j < natoms;  j++) {
      old_f[j] = f[j];
    }

    /* compute force */
    if (step_force(s)) return STEP_FAILURE;
    pe = s->system->potential_energy;

    /* 
     * Update backward difference table (p portion only) 
     */
    for (j = 0;  j < natoms;  j++) {
      double konst = half_dt * scal_inv_mass[j];
      u.x = konst * (f[j].x + old_f[j].x);
      u.y = konst * (f[j].y + old_f[j].y);
      u.z = konst * (f[j].z + old_f[j].z);

      for (i = s->orderk;  i > 0;  i--) {
        sum.x = sum.y = sum.z = 0.0;
        for (k = 1;  k < i;  k++) {
          sum.x += dif_p[k][j].x;
          sum.y += dif_p[k][j].y;
          sum.z += dif_p[k][j].z;
        } 
        dif_p[i][j].x = u.x - sum.x;
        dif_p[i][j].y = u.y - sum.y;
        dif_p[i][j].z = u.z - sum.z;
      }
    }

    /* half kick */
    for (i = 0;  i < natoms;  i++) {
      double konst = half_dt * scal_inv_mass[i];
      vel[i].x += konst * f[i].x;
      vel[i].y += konst * f[i].y;
      vel[i].z += konst * f[i].z;
    }

    /* Beta n+1 */
    dot = 0.0;
    for (i = 0;  i < natoms;  i++) {
      dot += pos[i].x*f[i].x + pos[i].y*f[i].y + pos[i].z*f[i].z; 
    }
    beta += half_dt * MD_FORCE_CONST * (-dot - 2.0 * pe);

    /* Update backward difference table (beta portion only) */
    for (i = s->orderk;  i > 0;  i--) {
      dot = 0.0;
      for (k = 1;  k < i;  k++) {
        dot += dif_beta[k];
      }
      dif_beta[i] = beta - dot;
    }

    /* Calculate Aij table */
    dif_q[0] = pos;
    dif_p[0] = vel;
    dif_beta[0] = 0;
    /* dif_beta[0] should never be used! */

    for (i = 1;  i < orderk1;  i++) {
      for (j = 0;  j < i;  j++) {
        /*
         * difference table stores velocity differences;
         * hence mass must be multiplied in to get momentum.
         */
        dot = 0.0;
        for (k = 0;  k < natoms;  k++) {
          dot += atom[k].m * (  dif_q[i][k].x*dif_p[j][k].x
              + dif_q[i][k].y*dif_p[j][k].y
              + dif_q[i][k].z*dif_p[j][k].z );
        }
        Atab[i][j] = dot;
        dot = 0.0;
        for (k = 0;  k < natoms;  k++) {
          dot += atom[k].m * (  dif_p[i][k].x*dif_q[j][k].x
              + dif_p[i][k].y*dif_q[j][k].y
              + dif_p[i][k].z*dif_q[j][k].z );
        }
        Atab[i][j] -= dot;
      }
      Atab[i][0] -= dif_beta[i];
    }

    for (i = 1;  i < orderk1;  i++) {
      for (j = 0;  j < i;  j++) {
        Atab[i][j] *= half_inv_dt;
      }
    }

    /*
     * Use the Aij table to compute the IMH.
     *
     * Higher order differnces will be smallest,
     * so start the summation with these terms
     * to minimize cancellation error.
     */
    for (i = SHADOW4;  i <= SHADOW24;  i += 2) {
      j = i - SHADOW2;
      s->sham[i] = 0;	
      coefnum = -1;
      for (p = 1;  p <= j+1;  p++) {
        for (m = 0;  m < p;  m++) {
          coefnum++;
        }
      }
      for (p = j+1;  p > 0;  p--) {
        for(m = p-1;  m >= 0;  m--) {
          s->sham[i] += s->coef[j][coefnum--] * Atab[p][m];
        }
      }
      s->sham[i] *= MD_KCAL_MOL;
    }
#ifdef DEBUG_WATCH
    printf("# shadow: %23.16f %23.16f %23.16f %23.16f %23.16f %23.16f\n",
        s->sham[SHADOW4], s->sham[SHADOW8], s->sham[SHADOW12],
        s->sham[SHADOW16], s->sham[SHADOW20], s->sham[SHADOW24]);
#endif

    /* submit results? */
    resultsCounter++;
    if (resultsFreq == resultsCounter) {
      resultsCounter = 0;
      /* store shadow energies */
      s->system->shadow_energy[STEP_SHADOW_4] = s->sham[SHADOW4];
      s->system->shadow_energy[STEP_SHADOW_8] = s->sham[SHADOW8];
      s->system->shadow_energy[STEP_SHADOW_12] = s->sham[SHADOW12];
      s->system->shadow_energy[STEP_SHADOW_16] = s->sham[SHADOW16];
      s->system->shadow_energy[STEP_SHADOW_20] = s->sham[SHADOW20];
      s->system->shadow_energy[STEP_SHADOW_24] = s->sham[SHADOW24];
      if (step_results(s, resultsFreq)) return STEP_FAILURE;
    }
  }

  /* store shadow energies */
  s->system->shadow_energy[STEP_SHADOW_4] = s->sham[SHADOW4];
  s->system->shadow_energy[STEP_SHADOW_8] = s->sham[SHADOW8];
  s->system->shadow_energy[STEP_SHADOW_12] = s->sham[SHADOW12];
  s->system->shadow_energy[STEP_SHADOW_16] = s->sham[SHADOW16];
  s->system->shadow_energy[STEP_SHADOW_20] = s->sham[SHADOW20];
  s->system->shadow_energy[STEP_SHADOW_24] = s->sham[SHADOW24];

  return 0;
}

/* end IMH */
