/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * nosehoover_explicit.c  - Explicit, time-reversible Nose-Hoover integration.
 *   A variant of the generalized leapfrog algorithm.
 */

#include <stdlib.h>
#include "step/step_defn.h"
#undef DEBUG_WATCH
#include "debug/debug.h"


int step_init_nosehoover_explicit(Step *s)
{
  /* check validity of parameters */
  if (s->param->nh_temperature <= 0.0) {
    return step_error(s, "Nose-Hoover theromstat temperature must be positive");
  }
  if (s->param->nh_timescale <= 0.0) {
    return step_error(s, "Nose-Hoover thermostat timescale must be positive");
  }

  /* 
   * 2K = g k_B T
   * here, k_B is modified so that units come out as internal energy units,
   * AMU A^2 / fs^2
   */
  s->twice_desired_ke = MD_BOLTZMAN * MD_ENERGY_CONST * 
    s->system->ndegfreedom * s->param->nh_temperature;
  FLT(s->twice_desired_ke);
  FLT(0.5 * MD_KCAL_MOL * s->twice_desired_ke);
  FLT(s->tempkonst * 0.5 * MD_KCAL_MOL * s->twice_desired_ke);
  
  /*
   * fictitious mass is in units AMU A^2, defined by 2K * (nh_timescale)^2,
   * where nh_timescale = tau/(2 pi) corresponds to the period of a
   * harmonic oscillator; usually set to 50 fs (Gromacs default?)
   */
  s->nh_qmass = (s->param->nh_timescale * s->param->nh_timescale) *
    s->twice_desired_ke;

  s->dt_over_qmass = s->param->timestep / s->nh_qmass;

  /* output */
  if (step_output(s, "initializing explicit Nose-Hoover integration:\n")
      || step_output(s, "  setting thermostat temperature to %g K\n",
        s->param->nh_temperature)
      || step_output(s, "  setting thermostat timescale to %g fs\n",
        s->param->nh_timescale)) {
    return STEP_FAILURE;
  }

  /* reset extended variables */
  s->xi = 0;
  s->eta = 0;

  return STEP_SUCCESS;
}


void step_done_nosehoover_explicit(Step *s)
{
  return;  /* nothing to do! */
}


int step_compute_nosehoover_explicit(Step *s, int32 numsteps)
{
  const double dt = s->param->timestep;
  const double half_dt = 0.5 * dt;
  double pe;
  MD_Dvec *f = s->system->force;
  MD_Dvec *vel = s->system->vel;
  MD_Dvec *pos = s->system->pos;
  const double *scal_inv_mass = s->scal_inv_mass;
  const int32 natoms = s->param->natoms;
  const int32 resultsFreq = s->param->resultsFreq;
  const int32 doRigidWaters = s->doRigidWaters;
  int32 n, i, resultsCounter;
  const MD_Atom *atom = s->param->atom;
  double xi;   /* friction coefficient */
  double eta;  /* log s */
  const double dt_q = s->dt_over_qmass;
  const double twoke_desired = s->twice_desired_ke;
  const double qmass = s->nh_qmass;
  double xi_old, scal;
  double twoke = s->twoke;
#ifdef DEBUG_WATCH
  double extended_energy;
#endif

  if (step_output(s, "Running explicit Nose-Hoover thermostat "
        "for %d steps...\n", numsteps)) {
    return STEP_FAILURE;
  }

  if (doRigidWaters) {
    /* settle startup also computes initial force */
    if (step_settle_startup(s)) return STEP_FAILURE;
  }
  else {
    /* compute initial force */
    if (step_force(s)) return STEP_FAILURE;
  }
  pe = s->system->potential_energy;

  /* restore extended Langrangian variables from previous call */
  xi = s->xi;
  eta = s->eta;

  /* also need twice ke for initial extended energy reduction */
  twoke = 0;
  for (i = 0;  i < natoms;  i++) {
    twoke += (vel[i].x * vel[i].x + vel[i].y * vel[i].y
        + vel[i].z * vel[i].z) * atom[i].m;
  }

  /* save reductions specific to Nose-Hoover */
  s->system->nh_extended_energy = pe + MD_KCAL_MOL * (0.5 * twoke
        + 0.5 * qmass * xi * xi + twoke_desired * eta);
  s->system->nh_friction_coef = xi;
  s->system->nh_log_s = eta;

  /* results for step 0 */
  if (step_results(s, 0)) return STEP_FAILURE;

  /* propagate system forward n steps */
  resultsCounter = 0;
  for (n = 0;  n < numsteps;  n++) {

    /* half kick */
    scal = 1.0 / (1.0 + half_dt * xi);
    for (i = 0;  i < natoms;  i++) {
      double konst = half_dt * scal_inv_mass[i];
      vel[i].x = scal * (vel[i].x + konst * f[i].x);
      vel[i].y = scal * (vel[i].y + konst * f[i].y);
      vel[i].z = scal * (vel[i].z + konst * f[i].z);
    }

    if (doRigidWaters && step_settle_prep(s)) return STEP_FAILURE;

    /* drift */
    for (i = 0;  i < natoms;  i++) {
      pos[i].x += dt * vel[i].x;
      pos[i].y += dt * vel[i].y;
      pos[i].z += dt * vel[i].z;
    }

    if (doRigidWaters && step_settle1(s, dt)) return STEP_FAILURE;

    twoke = 0;
    for (i = 0;  i < natoms;  i++) {
      twoke += (vel[i].x * vel[i].x + vel[i].y * vel[i].y
          + vel[i].z * vel[i].z) * atom[i].m;
    }
    xi_old = xi;
    xi = xi_old + dt_q * (twoke - twoke_desired);
    eta += half_dt * (xi + xi_old);

    /* compute force */
    if (step_force(s)) return STEP_FAILURE;
    pe = s->system->potential_energy;

    /* half kick */
    scal = 1.0 - half_dt * xi;
    for (i = 0;  i < natoms;  i++) {
      double konst = half_dt * scal_inv_mass[i];
      vel[i].x = scal * vel[i].x + konst * f[i].x;
      vel[i].y = scal * vel[i].y + konst * f[i].y;
      vel[i].z = scal * vel[i].z + konst * f[i].z;
    }

    if (doRigidWaters && step_settle2(s, dt)) return STEP_FAILURE;

    twoke = 0;
    for (i = 0;  i < natoms;  i++) {
      twoke += (vel[i].x * vel[i].x + vel[i].y * vel[i].y
          + vel[i].z * vel[i].z) * atom[i].m;
    }

#ifdef DEBUG_WATCH
    /* make sure that extended energy is conserved */    
    extended_energy = pe + MD_KCAL_MOL * (0.5 * twoke
        + 0.5 * qmass * xi * xi + twoke_desired * eta);

    printf("#xi= %.12g\n", xi);
    printf("#eta= %.12g\n", eta);
    printf("#ee= %.12g\n", extended_energy);
#endif

    /* submit results? */
    resultsCounter++;
    if (resultsFreq == resultsCounter) {
      resultsCounter = 0;

      /* save reductions specific to Nose-Hoover */
      s->system->nh_extended_energy = pe + MD_KCAL_MOL * (0.5 * twoke
            + 0.5 * qmass * xi * xi + twoke_desired * eta);
      s->system->nh_friction_coef = xi;
      s->system->nh_log_s = eta;

      if (step_results(s, resultsFreq)) return STEP_FAILURE;
    }
  }
  
  /* save extended Langrangian variables */
  s->xi = xi;
  s->eta = eta;
  s->twoke = twoke;

  /* save reductions specific to Nose-Hoover */
  s->system->nh_extended_energy = pe + MD_KCAL_MOL * (0.5 * twoke
        + 0.5 * qmass * xi * xi + twoke_desired * eta);
  s->system->nh_friction_coef = xi;
  s->system->nh_log_s = eta;

  return 0;
}
