/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * drude_chen.c  - Zhongzhou Chen's test version for thermalized Drude
 *   oscillator using dual Nose-Hoover thermostat implemented using
 *   explicit, time-reversible integration.
 *
 * (see G Lamoureux and B Roux, J Chem Phys, vol 119, num 6, p 3025, 2003)
 */

#include <stdlib.h>
#include <string.h>
#include "step/step_defn.h"
#undef DEBUG_WATCH
#include "debug/debug.h"


/* transformations */
static void atomcoord_to_comdbond(Step *s);
static void f_atomcoord_to_comdbond(Step *s);
static void vel_atomcoord_to_comdbond(Step *s);
static void pos_comdbond_to_atomcoord(Step *s);
static void vel_comdbond_to_atomcoord(Step *s);



int step_init_drude_chen(Step *s)
{
  /*
   * determine Drude bonds and mapping from atoms to (com, dbond)
   * allocate arrays needed for integration of COMs and Drude bonds
   * initialize (com, dbond) mass values
   */

  double m;
  int32 g_com, g_dbond;
  int32 num_com, num_dbond;
  int32 i, j, k;

  const int32 natoms = s->param->natoms;
  const MD_Atom *atom = s->param->atom;
  DrudeBond *drudeBond;
  int32 num_drudeBond;
  int32 *nonPolarized;
  int32 num_nonPolarized;

  s->is_drude = STEP_TRUE;

  /* check validity of parameters */
  if (s->param->drude_com_temperature <= 0) {
    return step_error(s,
        "Drude COM thermostat temperature must be positive\n");
  }
  if (s->param->drude_bond_temperature <= 0) {
    return step_error(s,
        "Drude bond thermostat temperature must be positive\n");
  }
  if (s->param->drude_com_timescale <= 0) {
    return step_error(s,
        "Drude COM thermostat timescale must be positive\n");
  }
  if (s->param->drude_bond_timescale <= 0) {
    return step_error(s,
        "Drude bond thermostat timescale must be positive\n");
  }

  /*
   * determine the number of Drude bonds
   * this is equal to the number of Drude particles
   */
  num_drudeBond = 0;
  for (i = 0;  i < natoms;  i++) {
    if (atom[i].is_other && atom[i].is_hydrogen) num_drudeBond++;
  }

  num_nonPolarized = natoms - 2*num_drudeBond;
  if (num_nonPolarized < 0) {
    return step_error(s,
        "found more than the possible number of Drude particles");
  }

  /*
   * create the map that transforms between atoms and (com, dbond)
   */
  drudeBond = s->drudeBond
    = (DrudeBond *) calloc(num_drudeBond, sizeof(DrudeBond));
  if (NULL==drudeBond) {
    return step_error(s, "calloc() failed to allocate s->drudeBond");
  }
  nonPolarized = s->nonPolarized
    = (int32 *) calloc(num_nonPolarized, sizeof(int32));
  if (NULL==nonPolarized) {
    return step_error(s, "calloc() failed to allocate s->nonPolarized");
  }

  j = 0;
  for (i = 0;  i < natoms;  i++) {
    if (atom[i].is_other && atom[i].is_hydrogen) {
      drudeBond[j].drudeIndex = i;
      /*
       * search backwards,
       * assuming that the immediately preceding heavy atom
       * is the other end of this Drude bond
       */
      k = i;
      do {
        k--;
        if (k < 0 || (atom[k].is_other && atom[k].is_hydrogen)) {
          /* either ran out of particles or found another Drude particle */
          return step_error(s,
              "did not find heavy atom for Drude particle %d", i);
        }
      } while (atom[k].is_hydrogen);
      drudeBond[j].atomIndex = k;
      drudeBond[j].massRatio = atom[i].m / (atom[i].m + atom[k].m);
      j++;
    }
  }
  j = 0;
  k = 0;
  for (i = 0;  i < natoms;  i++) {
    if (i == drudeBond[j].drudeIndex) {
      j++;
    }
    else if (i != drudeBond[j].atomIndex) {
      nonPolarized[k] = i;
      k++;
    }
  }
  s->num_drudeBond = num_drudeBond;
  s->num_nonPolarized = num_nonPolarized;
  /* finished building the map */

  /*
   * allocate space for the transformed system state
   */
  s->num_dbond = num_dbond = num_drudeBond;
  s->num_com   = num_com   = num_drudeBond + num_nonPolarized;

  s->pos_com = (MD_Dvec *) calloc(num_com, sizeof(MD_Dvec));
  if (NULL==s->pos_com) {
    return step_error(s, "calloc() failed to allocate s->pos_com");
  }
  s->vel_com = (MD_Dvec *) calloc(num_com, sizeof(MD_Dvec));
  if (NULL==s->vel_com) {
    return step_error(s, "calloc() failed to allocate s->vel_com");
  }
  s->f_com = (MD_Dvec *) calloc(num_com, sizeof(MD_Dvec));
  if (NULL==s->f_com) {
    return step_error(s, "calloc() failed to allocate s->f_com");
  }
  s->mass_com = (double *) calloc(num_com, sizeof(double));
  if (NULL==s->mass_com) {
    return step_error(s, "calloc() failed to allocate s->mass_com");
  }
  s->scal_inv_mass_com = (double *) calloc(num_com, sizeof(double));
  if (NULL==s->scal_inv_mass_com) {
    return step_error(s, "calloc() failed to allocate s->scal_inv_mass_com");
  }

  s->pos_dbond = (MD_Dvec *) calloc(num_dbond, sizeof(MD_Dvec));
  if (NULL==s->pos_dbond) {
    return step_error(s, "calloc() failed to allocate s->pos_dbond");
  }
  s->vel_dbond = (MD_Dvec *) calloc(num_dbond, sizeof(MD_Dvec));
  if (NULL==s->vel_dbond) {
    return step_error(s, "calloc() failed to allocate s->vel_dbond");
  }
  s->f_dbond = (MD_Dvec *) calloc(num_dbond, sizeof(MD_Dvec));
  if (NULL==s->f_dbond) {
    return step_error(s, "calloc() failed to allocate s->f_dbond");
  }
  s->mass_dbond = (double *) calloc(num_dbond, sizeof(double));
  if (NULL==s->mass_dbond) {
    return step_error(s, "calloc() failed to allocate s->mass_dbond");
  }
  s->scal_inv_mass_dbond = (double *) calloc(num_dbond, sizeof(double));
  if (NULL==s->scal_inv_mass_dbond) {
    return step_error(s, "calloc() failed to allocate s->scal_inv_mass_dbond");
  }

  /*
   * initialize the mass arrays;
   * note that the com array stores nonpolarized atoms after polarized atoms
   */
  for (k = 0;  k < num_drudeBond;  k++) {
    i = drudeBond[k].atomIndex;
    j = drudeBond[k].drudeIndex;
    m = 0;  /* Chen: instead of 'drudeBond[k].massRatio' */
    s->mass_com[k] = atom[i].m;  /* Chen: instead of 'atom[i].m + atom[j].m' */
    s->scal_inv_mass_com[k] = MD_FORCE_CONST / s->mass_com[k];
    s->mass_dbond[k] = (1-m) * atom[j].m;
    s->scal_inv_mass_dbond[k] = MD_FORCE_CONST / s->mass_dbond[k];
  }
  for (k = 0;  k < num_nonPolarized;  k++) {
    i = nonPolarized[k];
    s->mass_com[k + num_drudeBond] = atom[i].m;
    s->scal_inv_mass_com[k + num_drudeBond] = MD_FORCE_CONST / atom[i].m;
  }

  /*
   * number of degrees of freedom
   */
  g_dbond = 3 * num_dbond;                   /* of Drude bonds */
  g_com = s->system->ndegfreedom - g_dbond;  /* of COMs */
  if (g_com <= 0) {
    return step_error(s, "no degrees of freedom remaining in Drude COMs\n");
  }

  /* 
   * 2K = g_com k_B T
   * here, k_B is modified so that units come out as internal energy units,
   * AMU A^2 / fs^2
   */
  s->twice_desired_ke_com = MD_BOLTZMAN * MD_ENERGY_CONST *
    g_com * s->param->drude_com_temperature;

  s->qmass_com = s->param->drude_com_timescale *
    s->param->drude_com_timescale * s->twice_desired_ke_com;

  s->dt_over_qmass_com = s->param->timestep / s->qmass_com;

  s->tempkonst_com = 2.0 / (g_com * MD_BOLTZMAN);

  if (g_dbond > 0) {
    /* same observation as above */
    s->twice_desired_ke_dbond = MD_BOLTZMAN * MD_ENERGY_CONST *
      g_dbond * s->param->drude_bond_temperature;

    s->qmass_dbond = s->param->drude_bond_timescale *
      s->param->drude_bond_timescale * s->twice_desired_ke_dbond;

    s->dt_over_qmass_dbond = s->param->timestep / s->qmass_dbond;

    s->tempkonst_dbond = 2.0 / (g_dbond * MD_BOLTZMAN);
  }
  else {
    s->twice_desired_ke_dbond = 0;  /* OK for zero Drude particles */
    s->qmass_dbond = 0;
    s->dt_over_qmass_dbond = 0;
    s->tempkonst_dbond = 0;
  }

  /* output */
  if (step_output(s,
        "initializing Drude thermalized oscillator integration, using\n")
      || step_output(s,
        "Zhongzhou Chen's enhancement to ignore Drude particle mass, with\n")
      || step_output(s,
        "dual Nose-Hoover thermostats with warm bath for\n")
      || step_output(s,
        "atomic COMs (centers of mass) and cold bath for Drude bonds:\n")
      || step_output(s,
        "  setting COM number degrees of freedom to %d\n", g_com)
      || step_output(s,
        "  setting COM thermostat temperature to %g K\n",
        s->param->drude_com_temperature)
      || step_output(s,
        "  setting COM thermostat timescale to %g fs\n",
        s->param->drude_com_timescale)
      || step_output(s,
        "  setting bond number degrees of freedom to %d\n", g_dbond)
      || step_output(s,
        "  setting bond thermostat temperature to %g K\n",
        s->param->drude_bond_temperature)
      || step_output(s,
        "  setting bond thermostat timescale to %g fs\n",
        s->param->drude_bond_timescale)) {
    return STEP_FAILURE;
  } 

  return STEP_SUCCESS;
}


void step_done_drude_chen(Step *s)
{
  /* free memory allocations */
  free(s->drudeBond);
  free(s->nonPolarized);
  free(s->pos_com);
  free(s->vel_com);
  free(s->f_com);
  free(s->mass_com);
  free(s->scal_inv_mass_com);
  free(s->pos_dbond);
  free(s->vel_dbond);
  free(s->f_dbond);
  free(s->mass_dbond);
  free(s->scal_inv_mass_dbond);
}


int step_compute_drude_chen(Step *s, int32 numsteps)
{
  const double dt = s->param->timestep;
  const double half_dt = 0.5 * dt;
  double pe;

  const int32 num_com   = s->num_com;    /* length of Drude COM arrays */
  const int32 num_dbond = s->num_dbond;  /* length of Drude bond arrays */
  MD_Dvec *pos_com   = s->pos_com;       /* position of Drude COMs */
  MD_Dvec *pos_dbond = s->pos_dbond;     /* position of Drude bonds */
  MD_Dvec *vel_com   = s->vel_com;       /* velocity of Drude COMs */
  MD_Dvec *vel_dbond = s->vel_dbond;     /* velocity of Drude bonds */
  MD_Dvec *f_com     = s->f_com;         /* force on Drude COMs */
  MD_Dvec *f_dbond   = s->f_dbond;       /* force on Drude bonds */
  const double *mass_com   = s->mass_com;
  const double *mass_dbond = s->mass_dbond;
  const double *scal_inv_mass_com   = s->scal_inv_mass_com;
  const double *scal_inv_mass_dbond = s->scal_inv_mass_dbond;

  const int32 doRigidWaters = s->doRigidWaters;
  const int32 resultsFreq = s->param->resultsFreq;
  int32 n, i, resultsCounter;
  double xi_com;     /* friction coefficient for hot degrees */
  double eta_com;    /* log s for hot degrees */
  double xi_dbond;   /* friction coefficient for cold degrees */
  double eta_dbond;  /* log s for cold degrees */
  const double dt_q_com   = s->dt_over_qmass_com;
  const double dt_q_dbond = s->dt_over_qmass_dbond;
  const double twoke_desired_com   = s->twice_desired_ke_com;
  const double twoke_desired_dbond = s->twice_desired_ke_dbond;
  const double qmass_com   = s->qmass_com;
  const double qmass_dbond = s->qmass_dbond;
  double xi_old, scal;
  double twoke_com   = 0;
  double twoke_dbond = 0;
#ifdef DEBUG_WATCH
  double extended_energy;
#endif

  if (step_output(s,
        "Running Drude oscillators using\n")
      || step_output(s,
        "Zhongzhou Chen's enhancement to ignore Drude particle mass with\n")
      || step_output(s,
        "dual Nose-Hoover thermostats for %d steps...\n", numsteps)) {
    return STEP_FAILURE;
  }

  if (doRigidWaters) {
    /* settle startup also computes initial force */
    if (step_settle_startup(s)) return STEP_FAILURE;
  }
  else {
    /* compute initial force */
    if (step_force(s)) return STEP_FAILURE;
  }
  pe = s->system->potential_energy;

  /* restore extended Langrangian variables from previous call */
  xi_com = s->xi_com;
  eta_com = s->eta_com;
  xi_dbond = s->xi_dbond;
  eta_dbond = s->eta_dbond;

  /* transform atom coordinates to (com, dbond) system */
  atomcoord_to_comdbond(s);

  /* also need twice ke for initial extended energy reduction */
  twoke_com = 0;
  for (i = 0;  i < num_com;  i++) {
    twoke_com += (vel_com[i].x * vel_com[i].x
        + vel_com[i].y * vel_com[i].y
        + vel_com[i].z * vel_com[i].z) * mass_com[i];
  }
  twoke_dbond = 0;
  for (i = 0;  i < num_dbond;  i++) {
    twoke_dbond += (vel_dbond[i].x * vel_dbond[i].x
        + vel_dbond[i].y * vel_dbond[i].y
        + vel_dbond[i].z * vel_dbond[i].z) * mass_dbond[i];
  }

  /* save reductions specific to Drude */
  s->system->drude_com_energy = MD_KCAL_MOL * 0.5 * twoke_com;
  s->system->drude_bond_energy = MD_KCAL_MOL * 0.5 * twoke_dbond;
  s->system->drude_extended_energy = pe + s->system->drude_com_energy
    + s->system->drude_bond_energy + MD_KCAL_MOL *
    (0.5 * qmass_com * xi_com * xi_com + twoke_desired_com * eta_com
     + 0.5 * qmass_dbond * xi_dbond * xi_dbond
     + twoke_desired_dbond * eta_dbond);
  s->system->drude_com_temperature =
    s->tempkonst_com * s->system->drude_com_energy;
  s->system->drude_bond_temperature =
    s->tempkonst_dbond * s->system->drude_bond_energy;

  /* results for step 0 */
  if (step_results(s, 0)) return STEP_FAILURE;

  /* propagate system forward n steps */
  resultsCounter = 0;
  for (n = 0;  n < numsteps;  n++) {

    /* half kick for COM */
    scal = 1.0 / (1.0 + half_dt * xi_com);
    for (i = 0;  i < num_com;  i++) {
      double konst = half_dt * scal_inv_mass_com[i];
      vel_com[i].x = scal * (vel_com[i].x + konst * f_com[i].x);
      vel_com[i].y = scal * (vel_com[i].y + konst * f_com[i].y);
      vel_com[i].z = scal * (vel_com[i].z + konst * f_com[i].z);
    }

    /* half kick for Drude bonds */
    scal = 1.0 / (1.0 + half_dt * xi_dbond);
    for (i = 0;  i < num_dbond;  i++) {
      double konst = half_dt * scal_inv_mass_dbond[i];
      vel_dbond[i].x = scal * (vel_dbond[i].x + konst * f_dbond[i].x);
      vel_dbond[i].y = scal * (vel_dbond[i].y + konst * f_dbond[i].y);
      vel_dbond[i].z = scal * (vel_dbond[i].z + konst * f_dbond[i].z);
    }

    /* atom positions still valid */
    if (doRigidWaters && step_settle_prep(s)) return STEP_FAILURE;

    /* drift for COM */
    for (i = 0;  i < num_com;  i++) {
      pos_com[i].x += dt * vel_com[i].x;
      pos_com[i].y += dt * vel_com[i].y;
      pos_com[i].z += dt * vel_com[i].z;
    }

    /* drift for Drude bonds */
    for (i = 0;  i < num_dbond;  i++) {
      pos_dbond[i].x += dt * vel_dbond[i].x;
      pos_dbond[i].y += dt * vel_dbond[i].y;
      pos_dbond[i].z += dt * vel_dbond[i].z;
    }

    /* transform (com, dbond) positions to atom coordinates */
    pos_comdbond_to_atomcoord(s);

    if (doRigidWaters) {
      /* also need velocities */
      vel_comdbond_to_atomcoord(s);
      if (step_settle1(s, dt)) return STEP_FAILURE;
    }

    /* compute force */
    if (step_force(s)) return STEP_FAILURE;
    pe = s->system->potential_energy;

    if (doRigidWaters) {
      /* transform everything from atom coordinates to (com, dbond) system */
      atomcoord_to_comdbond(s);
    }
    else {
      /* transform just the force coordinates */
      f_atomcoord_to_comdbond(s);
    }

    /* propagate the extended coordinates for COM */
    twoke_com = 0;
    for (i = 0;  i < num_com;  i++) {
      twoke_com += (vel_com[i].x * vel_com[i].x
          + vel_com[i].y * vel_com[i].y
          + vel_com[i].z * vel_com[i].z) * mass_com[i];
    }
    xi_old = xi_com;
    xi_com = xi_old + dt_q_com * (twoke_com - twoke_desired_com);
    eta_com += half_dt * (xi_com + xi_old);

    /* propagate the extended coordinates for Drude bonds */
    twoke_dbond = 0;
    for (i = 0;  i < num_dbond;  i++) {
      twoke_dbond += (vel_dbond[i].x * vel_dbond[i].x
          + vel_dbond[i].y * vel_dbond[i].y
          + vel_dbond[i].z * vel_dbond[i].z) * mass_dbond[i];
    }
    xi_old = xi_dbond;
    xi_dbond = xi_old + dt_q_dbond * (twoke_dbond - twoke_desired_dbond);
    eta_dbond += half_dt * (xi_dbond + xi_old);

    /* half kick for COM */
    scal = 1.0 - half_dt * xi_com;
    for (i = 0;  i < num_com;  i++) {
      double konst = half_dt * scal_inv_mass_com[i];
      vel_com[i].x = scal * vel_com[i].x + konst * f_com[i].x;
      vel_com[i].y = scal * vel_com[i].y + konst * f_com[i].y;
      vel_com[i].z = scal * vel_com[i].z + konst * f_com[i].z;
    }

    /* half kick for Drude bonds */
    scal = 1.0 - half_dt * xi_dbond;
    for (i = 0;  i < num_dbond;  i++) {
      double konst = half_dt * scal_inv_mass_dbond[i];
      vel_dbond[i].x = scal * vel_dbond[i].x + konst * f_dbond[i].x;
      vel_dbond[i].y = scal * vel_dbond[i].y + konst * f_dbond[i].y;
      vel_dbond[i].z = scal * vel_dbond[i].z + konst * f_dbond[i].z;
    }

    if (doRigidWaters) {
      vel_comdbond_to_atomcoord(s);
      if (step_settle2(s, dt)) return STEP_FAILURE;
      vel_atomcoord_to_comdbond(s);
    }

    twoke_com = 0;
    for (i = 0;  i < num_com;  i++) {
      twoke_com += (vel_com[i].x * vel_com[i].x
          + vel_com[i].y * vel_com[i].y
          + vel_com[i].z * vel_com[i].z) * mass_com[i];
    }

    twoke_dbond = 0;
    for (i = 0;  i < num_dbond;  i++) {
      twoke_dbond += (vel_dbond[i].x * vel_dbond[i].x
          + vel_dbond[i].y * vel_dbond[i].y
          + vel_dbond[i].z * vel_dbond[i].z) * mass_dbond[i];
    }

#ifdef DEBUG_WATCH
    /* make sure that extended energy is conserved */    
    extended_energy = pe + MD_KCAL_MOL * (0.5 * twoke_com
        + 0.5 * qmass_com * xi_com * xi_com + twoke_desired_com * eta_com
        + 0.5 * twoke_dbond + 0.5 * qmass_dbond * xi_dbond * xi_dbond
        + twoke_desired_dbond * eta_dbond);
    if (step_output(s, "extended energy = %.12g\n", extended_energy)) {
      return STEP_FAILURE;
    }
#endif

    /* submit results? */
    resultsCounter++;
    if (resultsFreq == resultsCounter) {
      resultsCounter = 0;

      /* save reductions specific to Drude */
      s->system->drude_com_energy = MD_KCAL_MOL * 0.5 * twoke_com;
      s->system->drude_bond_energy = MD_KCAL_MOL * 0.5 * twoke_dbond;
      s->system->drude_extended_energy = pe + s->system->drude_com_energy
        + s->system->drude_bond_energy + MD_KCAL_MOL *
        (0.5 * qmass_com * xi_com * xi_com + twoke_desired_com * eta_com
         + 0.5 * qmass_dbond * xi_dbond * xi_dbond
         + twoke_desired_dbond * eta_dbond);
      s->system->drude_com_temperature =
        s->tempkonst_com * s->system->drude_com_energy;
      s->system->drude_bond_temperature =
        s->tempkonst_dbond * s->system->drude_bond_energy;

      /* transform (com, dbond) velocities to atom coordinates */
      if ( ! doRigidWaters) vel_comdbond_to_atomcoord(s);

      if (step_results(s, resultsFreq)) return STEP_FAILURE;
    }
  }

  /* save extended Langrangian variables */
  s->xi_com = xi_com;
  s->eta_com = eta_com;
  s->xi_dbond = xi_dbond;
  s->eta_dbond = eta_dbond;

  /* save other intermediate quantities */
  s->twoke_com = twoke_com;
  s->twoke_dbond = twoke_dbond;

  /* save reductions specific to Drude */
  s->system->drude_com_energy = MD_KCAL_MOL * 0.5 * twoke_com;
  s->system->drude_bond_energy = MD_KCAL_MOL * 0.5 * twoke_dbond;
  s->system->drude_extended_energy = pe + s->system->drude_com_energy
    + s->system->drude_bond_energy + MD_KCAL_MOL *
    (0.5 * qmass_com * xi_com * xi_com + twoke_desired_com * eta_com
     + 0.5 * qmass_dbond * xi_dbond * xi_dbond
     + twoke_desired_dbond * eta_dbond);
  s->system->drude_com_temperature =
    s->tempkonst_com * s->system->drude_com_energy;
  s->system->drude_bond_temperature =
    s->tempkonst_dbond * s->system->drude_bond_energy;

  /* transform (com, dbond) velocities to atom coordinates */
  if ( ! doRigidWaters) vel_comdbond_to_atomcoord(s);

  return STEP_SUCCESS;
}


void atomcoord_to_comdbond(Step *s)
{
  const DrudeBond *drudeBond = s->drudeBond;
  const int32 num_drudeBond = s->num_drudeBond;
  const int32 *nonPolarized = s->nonPolarized;
  const int32 num_nonPolarized = s->num_nonPolarized;

  const MD_Dvec *f = s->system->force;
  const MD_Dvec *vel = s->system->vel;
  const MD_Dvec *pos = s->system->pos;

  MD_Dvec *pos_com   = s->pos_com;       /* position of Drude COMs */
  MD_Dvec *pos_dbond = s->pos_dbond;     /* position of Drude bonds */
  MD_Dvec *vel_com   = s->vel_com;       /* velocity of Drude COMs */
  MD_Dvec *vel_dbond = s->vel_dbond;     /* velocity of Drude bonds */
  MD_Dvec *f_com     = s->f_com;         /* force on Drude COMs */
  MD_Dvec *f_dbond   = s->f_dbond;       /* force on Drude bonds */

  double m;
  int32 i, j, k;

  for (k = 0;  k < num_drudeBond;  k++) {
    i = drudeBond[k].atomIndex;
    j = drudeBond[k].drudeIndex;
    m = 0;  /* Chen: instead of 'drudeBond[k].massRatio' */

    pos_dbond[k].x = pos[j].x - pos[i].x;
    pos_dbond[k].y = pos[j].y - pos[i].y;
    pos_dbond[k].z = pos[j].z - pos[i].z;
    vel_dbond[k].x = vel[j].x - vel[i].x;
    vel_dbond[k].y = vel[j].y - vel[i].y;
    vel_dbond[k].z = vel[j].z - vel[i].z;
    f_dbond[k].x = (1-m) * f[j].x - m * f[i].x;
    f_dbond[k].y = (1-m) * f[j].y - m * f[i].y;
    f_dbond[k].z = (1-m) * f[j].z - m * f[i].z;

    pos_com[k].x = pos[i].x + m * pos_dbond[k].x;
    pos_com[k].y = pos[i].y + m * pos_dbond[k].y;
    pos_com[k].z = pos[i].z + m * pos_dbond[k].z;
    vel_com[k].x = vel[i].x + m * vel_dbond[k].x;
    vel_com[k].y = vel[i].y + m * vel_dbond[k].y;
    vel_com[k].z = vel[i].z + m * vel_dbond[k].z;
    f_com[k].x = f[i].x + f[j].x;
    f_com[k].y = f[i].y + f[j].y;
    f_com[k].z = f[i].z + f[j].z;
  }

  for (k = 0;  k < num_nonPolarized;  k++) {
    i = nonPolarized[k];
    pos_com[k + num_drudeBond] = pos[i];
    vel_com[k + num_drudeBond] = vel[i];
    f_com[k + num_drudeBond] = f[i];
  }

} /* atomcoord_to_comdbond */


void f_atomcoord_to_comdbond(Step *s)
{
  const DrudeBond *drudeBond = s->drudeBond;
  const int32 num_drudeBond = s->num_drudeBond;
  const int32 *nonPolarized = s->nonPolarized;
  const int32 num_nonPolarized = s->num_nonPolarized;

  const MD_Dvec *f = s->system->force;

  MD_Dvec *f_com     = s->f_com;         /* force on Drude COMs */
  MD_Dvec *f_dbond   = s->f_dbond;       /* force on Drude bonds */

  double m;
  int32 i, j, k;

  for (k = 0;  k < num_drudeBond;  k++) {
    i = drudeBond[k].atomIndex;
    j = drudeBond[k].drudeIndex;
    m = 0;  /* Chen: instead of 'drudeBond[k].massRatio' */

    f_dbond[k].x = (1-m) * f[j].x - m * f[i].x;
    f_dbond[k].y = (1-m) * f[j].y - m * f[i].y;
    f_dbond[k].z = (1-m) * f[j].z - m * f[i].z;

    f_com[k].x = f[i].x + f[j].x;
    f_com[k].y = f[i].y + f[j].y;
    f_com[k].z = f[i].z + f[j].z;
  }

  for (k = 0;  k < num_nonPolarized;  k++) {
    i = nonPolarized[k];
    f_com[k + num_drudeBond] = f[i];
  }

} /* f_atomcoord_to_comdbond */


void vel_atomcoord_to_comdbond(Step *s)
{
  const DrudeBond *drudeBond = s->drudeBond;
  const int32 num_drudeBond = s->num_drudeBond;
  const int32 *nonPolarized = s->nonPolarized;
  const int32 num_nonPolarized = s->num_nonPolarized;

  const MD_Dvec *vel = s->system->vel;

  MD_Dvec *vel_com   = s->vel_com;       /* velocity of Drude COMs */
  MD_Dvec *vel_dbond = s->vel_dbond;     /* velocity of Drude bonds */

  double m;
  int32 i, j, k;

  for (k = 0;  k < num_drudeBond;  k++) {
    i = drudeBond[k].atomIndex;
    j = drudeBond[k].drudeIndex;
    m = 0;  /* Chen: instead of 'drudeBond[k].massRatio' */

    vel_dbond[k].x = vel[j].x - vel[i].x;
    vel_dbond[k].y = vel[j].y - vel[i].y;
    vel_dbond[k].z = vel[j].z - vel[i].z;

    vel_com[k].x = vel[i].x + m * vel_dbond[k].x;
    vel_com[k].y = vel[i].y + m * vel_dbond[k].y;
    vel_com[k].z = vel[i].z + m * vel_dbond[k].z;
  }

  for (k = 0;  k < num_nonPolarized;  k++) {
    i = nonPolarized[k];
    vel_com[k + num_drudeBond] = vel[i];
  }

} /* vel_atomcoord_to_comdbond */


void pos_comdbond_to_atomcoord(Step *s)
{
  const DrudeBond *drudeBond = s->drudeBond;
  const int32 num_drudeBond = s->num_drudeBond;
  const int32 *nonPolarized = s->nonPolarized;
  const int32 num_nonPolarized = s->num_nonPolarized;

  MD_Dvec *pos = s->system->pos;

  const MD_Dvec *pos_com   = s->pos_com;    /* position of Drude COMs */
  const MD_Dvec *pos_dbond = s->pos_dbond;  /* position of Drude bonds */

  double m;
  int32 i, j, k;

  for (k = 0;  k < num_drudeBond;  k++) {
    i = drudeBond[k].atomIndex;
    j = drudeBond[k].drudeIndex;
    m = 0;  /* Chen: instead of 'drudeBond[k].massRatio' */

    pos[i].x = pos_com[k].x - m * pos_dbond[k].x;
    pos[i].y = pos_com[k].y - m * pos_dbond[k].y;
    pos[i].z = pos_com[k].z - m * pos_dbond[k].z;
    pos[j].x = pos_com[k].x + (1-m) * pos_dbond[k].x;
    pos[j].y = pos_com[k].y + (1-m) * pos_dbond[k].y;
    pos[j].z = pos_com[k].z + (1-m) * pos_dbond[k].z;
  }

  for (k = 0;  k < num_nonPolarized;  k++) {
    i = nonPolarized[k];
    pos[i] = pos_com[k + num_drudeBond];
  }

} /* pos_comdbond_to_atomcoord */


void vel_comdbond_to_atomcoord(Step *s)
{
  const DrudeBond *drudeBond = s->drudeBond;
  const int32 num_drudeBond = s->num_drudeBond;
  const int32 *nonPolarized = s->nonPolarized;
  const int32 num_nonPolarized = s->num_nonPolarized;

  MD_Dvec *vel = s->system->vel;

  const MD_Dvec *vel_com   = s->vel_com;    /* velocity of Drude COMs */
  const MD_Dvec *vel_dbond = s->vel_dbond;  /* velocity of Drude bonds */

  double m;
  int32 i, j, k;

  for (k = 0;  k < num_drudeBond;  k++) {
    i = drudeBond[k].atomIndex;
    j = drudeBond[k].drudeIndex;
    m = 0;  /* Chen: instead of 'drudeBond[k].massRatio' */

    vel[i].x = vel_com[k].x - m * vel_dbond[k].x;
    vel[i].y = vel_com[k].y - m * vel_dbond[k].y;
    vel[i].z = vel_com[k].z - m * vel_dbond[k].z;
    vel[j].x = vel_com[k].x + (1-m) * vel_dbond[k].x;
    vel[j].y = vel_com[k].y + (1-m) * vel_dbond[k].y;
    vel[j].z = vel_com[k].z + (1-m) * vel_dbond[k].z;
  }

  for (k = 0;  k < num_nonPolarized;  k++) {
    i = nonPolarized[k];
    vel[i] = vel_com[k + num_drudeBond];
  }

} /* vel_comdbond_to_atomcoord */
