#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <errno.h>
#include "pmetest/pmetest.h"

/*
#define ORIGINAL_TEST_PROBLEM
*/

#define TEST3D

/* Coulomb constant */
#define C  332.0636

int parse_cmdline(int argc, char *argv[],
                    double *tolerance, double *cutoff, int *gridsize)
{
  long g;
  char *name = argv[0];
  char *s, *ends;
  char c;

  while (--argc > 0 && (*++argv)[0] == '-') {
    while ( (c = *++argv[0]) ) {
      switch (c) {
        case 't':
          if (*++argv[0]) {
            s = argv[0];
          }
          else {
            --argc;
            s = (++argv)[0];
          }
          if (s == NULL) {
            fprintf(stderr, "no value specified for tolerance\n");
            return -1;
          }
          *tolerance = strtod(s, &ends);
          if (errno) {
            fprintf(stderr, "out of range:  tolerance=%g\n", *tolerance);
            return -1;
          }
          else if (*ends != '\0') {
            fprintf(stderr, "parse error:  %s\n", s);
            return -1;
          }
          argv[0] = ends-1;
          break;
        case 'c':
          if (*++argv[0]) {
            s = argv[0];
          }
          else {
            --argc;
            s = (++argv)[0];
          }
          if (s == NULL) {
            fprintf(stderr, "no value specified for cutoff\n");
            return -1;
          }
          *cutoff = strtod(s, &ends);
          if (errno) {
            fprintf(stderr, "out of range:  cutoff=%g\n", *cutoff);
            return -1;
          }
          else if (*ends != '\0') {
            fprintf(stderr, "parse error:  %s\n", s);
            return -1;
          }
          argv[0] = ends-1;
          break;
        case 'g':
          /*
          s = (*++argv[0] ? argv[0] : (char *) --argc, (++argv)[0]);
          */
          if (*++argv[0]) {
            s = argv[0];
          }
          else {
            --argc;
            s = (++argv)[0];
          }
          if (s == NULL) {
            fprintf(stderr, "no value specified for gridsize\n");
            return -1;
          }
          g = strtol(s, &ends, 10);
          if (errno || g > INT_MAX || g < INT_MIN) {
            fprintf(stderr, "out of range:  gridsize=%ld\n", g);
            return -1;
          }
          else if (*ends != '\0') {
            fprintf(stderr, "parse error:  %s\n", s);
            return -1;
          }
          argv[0] = ends-1;
          *gridsize = (int) g;
          break;
        default:
          fprintf(stderr, "illegal option %c\n", c);
          fprintf(stderr, "Usage: %s -t TOLERANCE -c CUTOFF -g GRIDSIZE\n",
                  name);
          return -1;
      }
    }
  }
  return 0;
}


#define NELEMS(x)  (sizeof(x)/sizeof((x)[0]))

int main(int argc, char *argv[])
{
  Pmetest *pme;
  PmetestParams s_params;
  PmetestParams *params = &s_params;
  PmetestSystem s_system;
  PmetestSystem *system = &s_system;

#ifdef ORIGINAL_TEST_PROBLEM
  /* system data */
  MD_Dvec position[] = { {3,3,3}, {5,5,5} };
  MD_Dvec force[] = { {0,0,0}, {0,0,0} };
  MD_Dvec direct_force[] = { {0,0,0}, {0,0,0} };
  MD_Dvec recip_force[] = { {0,0,0}, {0,0,0} };
  double charge[] = { 1, -1 };

  /* parameters */
  MD_Dvec center = { 4, 4, 4 };
  double len = 8;
  double cutoff = 5;
  int32 natoms = 2;
#else
  /* system data */
#ifndef  TEST3D
  MD_Dvec position[] = { {-2,0,0}, {2,0,0} };
#else
  MD_Dvec position[] = { {-2,-2,-2}, {2,2,2} };
#endif
  MD_Dvec force[] = { {0,0,0}, {0,0,0} };
  MD_Dvec direct_force[] = { {0,0,0}, {0,0,0} };
  MD_Dvec recip_force[] = { {0,0,0}, {0,0,0} };
  double charge[] = { 1, -1 };

  /* parameters */
  MD_Dvec center = { 0, 0, 0 };
  double len = 20;
  double cutoff = 9;
  int32 natoms = 2;
#endif
  int32 gridsize = 0;
  double tolerance = 0.0;
  double c;
  double scaled_charge[NELEMS(charge)];

  /* other variables */
  int32 i;

  /* command line args */
  if (parse_cmdline(argc, argv, &tolerance, &cutoff, &gridsize)) {
    exit(1);
  }

  memset(params, 0, sizeof(PmetestParams));
  params->natoms = natoms;
  params->center = center;
  params->cellvec1.x = len;
  params->cellvec2.y = len;
  params->cellvec3.z = len;
  params->cutoff = cutoff;
  params->nxspacings = gridsize;
  params->nyspacings = gridsize;
  params->nzspacings = gridsize;
  params->tolerance = tolerance;
#ifdef ORIGINAL_TEST_PROBLEM
  params->nxspacings = 8;
  params->nyspacings = 8;
  params->nzspacings = 8;
  params->cutoff = 5.0;
#endif

  if ((pme = pmetest_create(params)) == NULL) {
    fprintf(stderr, "call to pme_create failed\n");
    exit(1);
  }

  printf("\n");
  printf("tolerance:  %g\n", params->tolerance);
  printf("cutoff:     %g\n", params->cutoff);
  printf("gridsize:   %d\n", params->nxspacings);
  printf("\n");

  /* compute scaled charge */
  c = sqrt(C);
  for (i = 0;  i < NELEMS(charge);  i++) {
    scaled_charge[i] = c * charge[i];
  }

  system->charge = scaled_charge;
  system->pos = position;
  system->f_elec = force;
  system->f_direct = direct_force;
  system->f_recip = recip_force;

  if (pmetest_compute(pme, system)) {
    fprintf(stderr, "call to pme_compute failed\n");
    exit(1);
  }

  printf("potential:  %10.6f\n", system->u_elec);
  printf("\n");
  printf("%4s  %10s  %10s  %10s\n",
          "atom", "force[i].x", "force[i].y", "force[i].z");
  for (i = 0;  i < natoms;  i++) {
    printf("%4d  %10.6f  %10.6f  %10.6f\n",
            i, force[i].x, force[i].y, force[i].z);
  }

#ifdef ORIGINAL_TEST_PROBLEM
  printf("\n");
  printf("tolerance %g\n", params->tolerance);
  printf("nxspacings %d\n", params->nxspacings);
  printf("nyspacings %d\n", params->nyspacings);
  printf("nzspacings %d\n", params->nzspacings);
  printf("interporder %d\n", params->interporder);
  printf("cellvec1 %g %g %g\n", params->cellvec1.x,
      params->cellvec1.y, params->cellvec1.z);
  printf("cellvec2 %g %g %g\n", params->cellvec2.x,
      params->cellvec2.y, params->cellvec2.z);
  printf("cellvec3 %g %g %g\n", params->cellvec3.x,
      params->cellvec3.y, params->cellvec3.z);
  printf("center %g %g %g\n", params->center.x,
      params->center.y, params->center.z);
  printf("cutoff %g\n", params->cutoff);
  printf("ewaldcof %g\n", params->ewaldcof);

  printf("\n");
  printf("RecipPotential = %g\n", system->u_recip);
  printf("RecipForce[0]= %g %g %g\n",
      recip_force[0].x, recip_force[0].y, recip_force[0].z);
  printf("RecipForce[1]= %g %g %g\n",
      recip_force[1].x, recip_force[1].y, recip_force[1].z);
  printf("DirectPotential = %g\n", system->u_direct);
  printf("DirectForce[0]= %g %g %g\n",
      direct_force[0].x, direct_force[0].y, direct_force[0].z);
  printf("DirectForce[1]= %g %g %g\n",
      direct_force[1].x, direct_force[1].y, direct_force[1].z);
#endif

  pmetest_destroy(pme);
  return 0;
}
