/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    mgrid.h
 * @brief   Multilevel summation (a.k.a. multiple grid).
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c Mgrid_tag class provides a fast method for computing
 * electrostatics.  This multilevel summation method produces smooth
 * potential energy and continuous forces for both nonperiodic and
 * periodic boundary conditions.  The approximation utilizes a repeated
 * smooth splitting of the interaction potential into a short-range
 * part evaluated directly and a slowly varying long-range part
 * which is interpolated to a lattice.  Repeated use of this idea 
 * gives rise to a hierarchical method for which, under reasonable
 * assumptions, the computational cost is linear in the number of
 * atoms.
 *
 * The original paper detailing this method for nonperiodic systems is
 * @li R. D. Skeel, I. Tezcan, D. J. Hardy, "Multiple Grid Methods for
 *   Classical Molecular Dynamics," J. Comp. Chem., 23, 673-684, 2002.
 *   http://bionum.cs.uiuc.edu/SkTH02.pdf
 *
 * Further analysis, improvements, and the extension of the method to
 * periodic boundaries are presented in my dissertation, which should
 * be available shortly.
 *
 * Caveats:
 * @li For now, the method is restricted to cubes that are either nonperiodic
 *   or fully periodic.  Extensions to the implementation are underway that
 *   will permit orthogonal nonperiodic systems and ortho-rhombic periodic
 *   systems.
 * 
 * See the notes for @c MgridParam_tag for recommended parameter selection.
 */


#ifndef MGRID_H
#define MGRID_H

#include "mdapi/mdtypes.h"
#include "mgrid/lattice.h"

#ifdef __cplusplus
extern "C" {
#endif

  /****************************************************************************
   * internal definitions
   ****************************************************************************/

  /**@brief (For internal use only.)
   */
  typedef struct MgridCell_tag {
    /* cells for geometric hashing, implement as cursor linked list */
    int32 head;          /* index of first atom in this cell */
    int32 cnt;           /* count number of atoms in this cell */
    int32 nnbrs;         /* length of neighbor cell list */
    int32 nbr[14];       /* up to half-shell of neighbors, including self */
    MD_Dvec offset[14];  /* offset for neighbors that are periodic images */
  } MgridCell;


  /****************************************************************************
   * user interface
   ****************************************************************************/

  /* error return value */
  enum {
    MGRID_FAIL = -1     /**< Return value from failed function call. */
  };


  /* boundary */
  /**@brief Boundary conditions.
   *
   * Set @c MgridParam_tag::boundary to indicate boundary conditions.
   */
  enum MgridBoundary_tag {
    MGRID_BOUNDARY_BEGIN = 0,
    MGRID_PERIODIC,     /**< Periodic boundary conditions. */
    MGRID_NONPERIODIC,  /**< Nonperiodic boundary conditions. */
    MGRID_BOUNDARY_END
  };


  /* choice of grid approximation */
  /**@brief Approximation.
   *
   * Set @c MgridParam_tag::approx to indicate choice of the
   * approximation scheme.
   */
  enum MgridApprox_tag {
    MGRID_APPROX_BEGIN = 0,
    MGRID_CUBIC,
    /**< Cubic numerical Hermite interpolation, @f$\mathrm{C}^1 @f$ continuous. */
    MGRID_BSPLINE,
    /**< B-spline approximation, @f$\mathrm{C}^2 @f$ continuous. */
    MGRID_QUINTIC1,
    /**< Quintic polynomial interpolation, @f$\mathrm{C}^1 @f$ continuous. */
    MGRID_QUINTIC2,
    /**< Quintic polynomial interpolation, @f$\mathrm{C}^2 @f$ continuous. */
    MGRID_HEPTIC1,
    /**< Heptic (7th degree) polynomial interpolation,
     * @f$\mathrm{C}^1 @f$ continuous. */
    MGRID_HEPTIC3,
    /**< Heptic (7th degree) polynomial interpolation,
     * @f$\mathrm{C}^3 @f$ continuous. */
    MGRID_NONIC1,
    /**< Nonic (9th degree) polynomial interpolation,
     * @f$\mathrm{C}^1 @f$ continuous. */
    MGRID_NONIC4,
    /**< Nonic (9th degree) polynomial interpolation,
     * @f$\mathrm{C}^4 @f$ continuous. */
    MGRID_HERMITE,
    /**< Hermite interpolation, @f$\mathrm{C}^1 @f$ continuous. */
    MGRID_APPROX_END
  };


  /* choice of splitting */
  /**@brief Splitting.
   *
   * Set @c MgridParam_tag::split to indicate choice of splitting.
   * This selects the function that will be approximated.
   */
  enum MgridSplit_tag {
    MGRID_SPLIT_BEGIN = 0,
    MGRID_TAYLOR1,
    /**< 1st order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^1 @f$ continuous. */
    MGRID_TAYLOR2,
    /**< 2nd order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^2 @f$ continuous. */
    MGRID_TAYLOR3,
    /**< 3rd order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^3 @f$ continuous. */
    MGRID_TAYLOR4,
    /**< 4th order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^4 @f$ continuous. */
    MGRID_TAYLOR5,
    /**< 5th order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^5 @f$ continuous. */
    MGRID_TAYLOR6,
    /**< 6th order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^6 @f$ continuous. */
    MGRID_TAYLOR7,
    /**< 7th order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^7 @f$ continuous. */
    MGRID_TAYLOR8,
    /**< 8th order Taylor polynomial in @f$ r^2 @f$,
     * @f$\mathrm{C}^8 @f$ continuous. */
    MGRID_ERRMIN3,
    /**< Degree 3 polynomial in @f$ r^2 @f$, @f$\mathrm{C}^2 @f$ continuous,
     * that minimizes the maximum error bound for cubic numerical Hermite
     * interpolation. */
    MGRID_EXSELF1,
    /**< Smoothed part is piecewise-defined degree 3 to degree 1 polynomial
     * in @f$ r^2 @f$, @f$\mathrm{C}^2 @f$ continuous,
     * computes exact self-forces within @f$ \frac{1}{2} a @f$. */
    MGRID_EXSELF2,
    /**< Smoothed part is piecewise-defined degree 3 to degree 1 polynomial
     * in @f$ r^2 @f$, @f$\mathrm{C}^2 @f$ continuous,
     * computes exact self-forces within @f$ \frac{1}{3} a @f$. */
    MGRID_EXSELF3,
    /**< Smoothed part is piecewise-defined degree 3 to degree 1 polynomial
     * in @f$ r^2 @f$, @f$\mathrm{C}^2 @f$ continuous,
     * computes exact self-forces within @f$ \frac{1}{4} a @f$. */
    MGRID_EXSELF7,
    /**< Smoothed part is piecewise-defined degree 3 polynomial in @f$ r @f$
     * to degree 1 polynomial
     * in @f$ r^2 @f$, @f$\mathrm{C}^2 @f$ continuous,
     * computes exact self-forces within @f$ \frac{1}{2} a @f$. */
    MGRID_ODDPR1,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^1 @f$ continuous. */
    MGRID_ODDPR2,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^2 @f$ continuous. */
    MGRID_ODDPR3,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^3 @f$ continuous. */
    MGRID_ODDPR4,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^4 @f$ continuous. */
    MGRID_ODDPR5,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^5 @f$ continuous. */
    MGRID_ODDPR6,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^6 @f$ continuous. */
    MGRID_ODDPR7,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^7 @f$ continuous. */
    MGRID_ODDPR8,
    /**< Smoothed part contains some odd powers of @f$ r @f$,
     * @f$\mathrm{C}^8 @f$ continuous. */
    MGRID_LOWDEG1,
    MGRID_LOWDEG2,
    MGRID_LOWDEG3,
    MGRID_LOWDEG4,
    MGRID_LOWDEG5,
    MGRID_LOWDEG6,
    MGRID_LOWDEG7,
    MGRID_LOWDEG8,
    MGRID_LOWALT5_1,
    MGRID_LOWALT6_1,
    MGRID_LOWALT7_1,
    MGRID_LOWALT8_1,
    MGRID_LOWALT8_2,
    MGRID_EXSFPC2,
    /**< Compute exact self force (with cubic numerical Hermite, h=2, a>=12),
     * C2 joining polynomial in r to constant function, switching at 7/8. */
    MGRID_EXSFPQ2,
    /**< Compute exact self force (with cubic numerical Hermite, h=2, a>=12),
     * C2 joining polynomial in r to quadratic function, switching at 7/8. */
    MGRID_EXSFRC2,
    /**< Compute exact self force (with cubic numerical Hermite, h=2, a>=12),
     * C2 joining 1/r term plus polynomial in r to constant function, 7/8. */
    MGRID_EXSFRQ2,
    /**< Compute exact self force (with cubic numerical Hermite, h=2, a>=12),
     * C2 joining 1/r term plus polynomial in r to quadratic function, 7/8. */
    MGRID_EXACT3,
    MGRID_EXACT2,
    MGRID_EXACT1,
    MGRID_EXACT0,
    MGRID_SPLIT_END
  };


  /**@brief Helper class to configure @c Mgrid_tag class.
   *
   * User sets the internal fields of this class in order to
   * configure the parameters needed by @c Mgrid_tag class.
   * The @c MgridParam_tag class is intended to be setup once
   * and used to initialize the @c Mgrid_tag class.
   *
   * The recommended parameters depend on the accuracy desired:
   * @li for low accuracy choose:
   *   @f$ 3 \leq @f$ @c spacing @f$ \leq 4 @f$,
   *   @c cutoff @f$ \approx 8 @f$,
   *   @c approx @f$ = @f$  @c MGRID_CUBIC,
   *   @c split @f$ = @f$ @c MGRID_TAYLOR2.
   * @li for medium accuracy choose:
   *   @c spacing @f$ \approx 3 @f$,
   *   @f$ 8 \leq @f$ @c cutoff @f$ \leq 12 @f$,
   *   @c approx @f$ = @f$  @c MGRID_QUINTIC1,
   *   @c split @f$ = @f$ @c MGRID_TAYLOR3.
   * @li for high accuracy choose:
   *   @f$ 2 \leq @f$ @c spacing @f$ \leq 3 @f$,
   *   @c cutoff @f$ \approx 12 @f$,
   *   @c approx @f$ = @f$  @c MGRID_HEPTIC1,
   *   @c split @f$ = @f$ @c MGRID_TAYLOR4
   *   (or perhaps @c MGRID_NONIC1 and @c MGRID_TAYLOR5).
   */
  typedef struct MgridParam_tag {
    MD_Dvec center;  /**< Center of cubic domain. */
    double length;   /**< Length of cubic domain. */
    double cutoff;   /**< Cutoff distance, typically 8 to 12 Angstroms. */
    double spacing;  /**< Lattice spacing, typically about 3 Angstroms. */
    int32 nspacings; /**< Number of spacings along each dimension
                       to cover domain. */
    int32 nlevels;   /**< Number of levels in the grid hierarchy. */
    int32 boundary;  /**< Select either nonperiodic or periodic boundaries. */
    int32 natoms;    /**< Number of atoms in the system. */
    int32 approx;    /**< Select the approximation scheme to use. */
    int32 split;     /**< Select the splitting to use. */
  } MgridParam;


  /* compute forces using system data */
  /**@brief Helper class to provide and return results from @c Mgrid_tag class.
   *
   * User sets the internal fields of this class in order to provide
   * data to @c Mgrid_tag class and receive its results.  The user
   * retains ownership of the arrays.  The arrays are of length
   * @c MgridParam_tag::natoms.
   *
   * Input:@n
   * The @c pos and @c charge arrays must be provided.  The atomic positions
   * in @c pos must map into the cubic domain defined by @c MgridParam_tag.
   * For periodic simulations, this means that they might have to be
   * "wrapped" back into the box.
   * Depending on the desired units, the atomic charges supplied to
   * the @c charge array will need to be weighted by the square root of
   * the conversion constant.
   *
   * Optional input:@n
   * The @c excl_list and @c scaled14_list are jagged arrays as defined
   * by the force library.  These are supplied if the user wishes the
   * mgrid library to compute electrostatic exclusions, along with
   * @c scaling14 multiplicative constant if applicable.  All things
   * considered, there is no discernible numerical advantage to having
   * electrostatic exclusions subtracted out sooner rather later, so it
   * is probably best to just have the force library deal with it.
   *
   * Output:@n
   * The electrostatic potential energy is returned in @c u_elec, the
   * sum of the short-range part @c u_short and the long-range part
   * @c u_long.  The array @c f_elec designated to receive the electrostatic
   * forces must be provided.  The arrays @c f_short and @c f_long provide,
   * respectively, the short-range and long-range parts of the force for
   * each atom, intended for diagnostic purposes.
   */
  typedef struct MgridSystem_tag {

  /* output, user supplies arrays */
    double u_elec;
    /**< Electrostatic potential energy. */
    double u_short;
    /**< Contribution to the potential from the short-range part. */
    double u_long;
    /**< Contribution to the potential from the long-range part. */
    MD_Dvec *f_elec;
    /**< Atomic forces due to the electrostatic potential energy. */
    MD_Dvec *f_short;
    /**< Contribution to the forces from the short-range part,
     * this array is optional. */
    MD_Dvec *f_long;
    /**< Contribution to the forces from the long-range part,
     * this array is optional. */

  /* input */
    MD_Dvec *pos;
    /**< Atomic positions, must be within cubic domain. */
    double *charge;
    /**< Atomic charges, likely to be weighted by the square root of
     * the unit conversion constant. */

  /* input, have mgrid process exclusions */
    int32 **excl_list;
    /**< List of nonbonded exclusions as returned by the @c Force_tag class
     * method @c force_excl_list().  Only provide this if you wish
     * @c Mgrid_tag to deal with nonbonded electrostatic exclusions. */
    int32 **scaled14_list;
    /**< List of scaled 1-4 interactions as returned by the @c Force_tag class
     * method @c force_scaled14_list().  Only provide this if you wish
     * @c Mgrid_tag to deal with scaled 1-4 electrostatic interactions. */
    double scaling14;
    /**< Multiplicative constant for scaled 1-4 electrostatic interactions,
     * set to @c ForceParam_tag::scaling14.  Only provide this if you wish
     * @c Mgrid_tag to deal with scaled 1-4 electrostatic interactions. */
  } MgridSystem;


  /* contents opaque to user */
  /**@brief Multilevel summation (a.k.a. multiple grid) solver.
   *
   * Members should be treated as private.
   */
  typedef struct Mgrid_tag {
    MgridParam param;     /* copy user parameters */


    /** short range force contribution ***/

    int (*short_force)(struct Mgrid_tag *, MgridSystem *);
      /* short range force evaluation "virtual" method */

    void (*short_done)(struct Mgrid_tag *);
      /* short range destructor "virtual" method */

    MD_Dvec lo;           /* lowest corner of grid cell lattice */
    double inv_cellsize;  /* inv_cellsize=(1/cellsize) */
    double inv_cutoff;    /* inv_cutoff=(1/cutoff) */

    int32 *next;          /* next "pointer" for cursor linked list */
      /* length is natoms, points to next atom within this grid cell */

    MgridCell *cell;      /* lattice of grid cells */
    int32 ncells;         /* total number of grid cells */
    int32 ndimcells;      /* number of grid cells in each dimension */
      /* (ncells == ndimcells * ndimcells * ndimcells) */

    int32 is_split_even_powers;
      /* splitting has even powers of r/a, for choosing evaluation routine */

    int (*cell_interactions)(struct Mgrid_tag *, MgridSystem *);
      /* evaluation routine for short range part */


    /*** long range force contribution ***/

    int (*long_force)(struct Mgrid_tag *, MgridSystem *);
      /* long range force evaluation "virtual" method */

    void (*long_done)(struct Mgrid_tag *);
      /* long range destructor "virtual" method */

    MD_Dvec origin;       /* lowest corner of domain, (0,0,0) of lattice */
    double inv_spacing;   /* 1/h, where h = lattice spacing */
    double u_self;        /* self potential */
    double g_zero;        /* smoothing g(r) evaluated at r=0 */

    MgridLattice *qgrid;  /* array of lattice of charge, length nlevels */

    MgridLattice *egrid;  /* array of lattice of potential, length nlevels */

    /* for interpolation schemes using only function values */
    double *scaling;      /* array of scaling for direct sum, length nlevels */

    MgridLattice gdsum;   /* direct sum weights for lattice cutoff parts */
    int32 gdsum_radius;   /* radius of lattice point "sphere" */

    MgridLattice glast;   /* last level direct sum weights */
    int32 glast_radius;   /* radius of lattice point "sphere" */

    /* for interpolation schemes using function values and derivatives */
    MgridLattice *gdsum_list;  /* direct sum weights, for each level */
    int32 *gdsum_radius_list;  /* radius of "sphere", for each level */

    /* for interpolation schemes using function values and derivatives */
    MgridLattice *opres_list;  /* restriction operator, indexed by level */
    MgridLattice *oppro_list;  /* prolongation operator, indexed by level */

    /* for interpolation schemes using function values and derivatives */
    MgridLattice *is_zero;     /* flag telling if matrix is all zeros */

  } Mgrid;


  /* help user choose suitable params */

  /**@brief Chooses suitable @c MgridParam_tag parameters.
   *
   * @param[in,out] p  Partially initialized @c MgridParam_tag object.
   *
   * Given some number of parameters, this function tries to determine
   * the rest of the parameters.  The following parameters must be set:
   * @li @c MgridParam_tag::center
   * @li @c MgridParam_tag::cutoff
   * @li @c MgridParam_tag::boundary
   * @li @c MgridParam_tag::natoms
   * @li @c MgridParam_tag::approx
   * @li @c MgridParam_tag::split
   *
   * If periodic boundaries are employed, then @c MgridParam_tag::length
   * must also be set.
   *
   * The method of filling in the remaining parameters differs depending
   * on the selected boundary conditions.
   *
   * For periodic boundaries,
   * the number of spacings @c MgridParam_tag::nspacings is chosen to
   * be a power of 2 such that @c MgridParam_tag::spacing is no smaller
   * than that provided.  If @c MgridParam_tag::nspacings is set but not
   * @c MgridParam_tag::spacing, then @c MgridParam_tag::nspacings is
   * changed to be a power of 2 no larger than the amount given.
   * MgridParam_tag::nlevels is set to its maximum amount.
   *
   * For nonperiodic boundaries,
   * any of @c MgridParam_tag::length, @c MgridParam_tag::spacing,
   * @c MgridParam_tag::nspacings can be determined if two are provided.
   * MgridParam_tag::nlevels is set to its maximum amount.
   *
   * @return 0 for success or @c MGRID_FAIL on failure.
   */
  int mgrid_param_config(MgridParam *p);


  /* convert between types and strings */

  /**@brief Convert string to boundary option.
   *
   * Assist in configuring @c MgridParam_tag by converting the strings
   * "periodic" and "nonperiodic" into their @c MgridBoundary_tag values.
   *
   * @return Nonzero boundary value or 0 if string is not recognized.
   */
  int mgrid_string_to_boundary(const char *);

  /**@brief Convert boundary option to string.
   *
   * Provide diagnostic information by converting @c MgridParam_tag::boundary
   * to a meaningful string form.
   *
   * @return Corresponding string form or "unknown" if value is out of range.
   */
  const char *mgrid_boundary_to_string(int);

  /**@brief Convert string to approximation option.
   *
   * Assist in configuring @c MgridParam_tag by converting strings
   * into their @c MgridApprox_tag values.  Leave off the "MGRID_"
   * prefix; string matching is not case sensitive.
   *
   * @return Nonzero approximation value or 0 if string is not recognized.
   */
  int mgrid_string_to_approx(const char *);

  /**@brief Convert approximation option to string.
   *
   * Provide diagnostic information by converting @c MgridParam_tag::approx
   * to a meaningful string form.
   *
   * @return Corresponding string form or "unknown" if value is out of range.
   */
  const char *mgrid_approx_to_string(int);

  /**@brief Convert string to splitting option.
   *
   * Assist in configuring @c MgridParam_tag by converting strings
   * into their @c MgridSplit_tag values.  Leave off the "MGRID_"
   * prefix; string matching is not case sensitive.
   *
   * @return Nonzero splitting value or 0 if string is not recognized.
   */
  int mgrid_string_to_split(const char *);

  /**@brief Convert splitting option to string.
   *
   * Provide diagnostic information by converting @c MgridParam_tag::split
   * to a meaningful string form.
   *
   * @return Corresponding string form or "unknown" if value is out of range.
   */
  const char *mgrid_split_to_string(int);


  /* methods for Mgrid */

  /**@brief Constructor.
   *
   * Initializes a @c Mgrid_tag object.
   * Must call this routine first.
   *
   * @return 0 for success or @c MGRID_FAIL on failure.
   */
  int mgrid_init(Mgrid *);

  /**@brief Setup the @c Mgrid_tag object.
   *
   * @param[in] s  The charges are needed for the system.
   * @param[in] p  The completely filled in @c MgridParam_tag object.
   *
   * This routine performs memory allocation and computes tables of
   * constants needed for efficient force evaluation.  It is intended
   * that this routine is called once before a simulation begins.
   *
   * The only thing needed from @c s are the charges, which should not
   * change between calls to the force evaluation function.
   *
   * The @c MgridParam_tag object @c p must be completely filled in,
   * presumably from a successful call to @c mgrid_param_config().
   *
   * @return 0 on success or @c MGRID_FAIL on failure.
   */
  int mgrid_setup(Mgrid *, const MgridSystem *s, const MgridParam *p);

  /**@brief Compute electrostatic forces and potential energy.
   *
   * @param[in,out] s  Provides input data and receives results.
   *
   * This routine performs the multilevel summation computation.
   * The @c MgridSystem_tag object @c s must provide the atomic positions
   * (wrapped into the cubic domain) and the (possibly weighted) charges.
   * The updated force array(s) and potential energy values are returned.
   * It is intended that this routine may be called successively during
   * a simulation with updated positions provided.
   *
   * @return 0 for success or @c MGRID_FAIL on failure.
   */
  int mgrid_force(Mgrid *, MgridSystem *s);

  /**@brief Destructor.
   *
   * Frees memory allocations made internally during @c mgrid_setup().
   * Call when finished using @c Mgrid_tag object.
   */
  void mgrid_done(Mgrid *);


  /* validate atom positions within bounding cell */

  /**@brief Validate atomic positions.
   *
   * @param[in] s  Provides atomic positions.
   *
   * Given that @c Mgrid_tag has been setup by @c mgrid_setup(),
   * this routine verifies that all positions are within the
   * cubic domain.
   *
   * @return The number of atoms MgridParam_tag::natoms is returned
   * on success.  Otherwise, the index of the first atom found outside
   * the domain is returned.
   */
  int mgrid_system_validate(const Mgrid *, const MgridSystem *s);


  /* compute exact smoothed part (for nonperiodic boundaries) */

  /**@brief Compute exact smoothed part.
   *
   * @param[in,out] s  Provides input data and receives results.
   *
   * For nonperiodic boundaries, this routine computes the all pairs
   * solution for the smoothed (long-range) part.  This is a quadratic
   * computation, provided solely for the purpose of analysis.
   */
  int mgrid_exact_smooth(Mgrid *, MgridSystem *s);


  /* internal methods for Mgrid */
  int mgrid_short_setup(Mgrid *);
  int mgrid_ncubic_setup(Mgrid *, const MgridSystem *);
  int mgrid_pcubic_setup(Mgrid *, const MgridSystem *);
  int mgrid_nbspline_setup(Mgrid *, const MgridSystem *);
  int mgrid_pbspline_setup(Mgrid *, const MgridSystem *);
  int mgrid_nquintic1_setup(Mgrid *, const MgridSystem *);
  int mgrid_pquintic1_setup(Mgrid *, const MgridSystem *);
  int mgrid_nquintic2_setup(Mgrid *, const MgridSystem *);
  int mgrid_pquintic2_setup(Mgrid *, const MgridSystem *);
  int mgrid_nheptic1_setup(Mgrid *, const MgridSystem *);
  int mgrid_pheptic1_setup(Mgrid *, const MgridSystem *);
  int mgrid_nheptic3_setup(Mgrid *, const MgridSystem *);
  int mgrid_pheptic3_setup(Mgrid *, const MgridSystem *);
  int mgrid_nnonic1_setup(Mgrid *, const MgridSystem *);
  int mgrid_pnonic1_setup(Mgrid *, const MgridSystem *);
  int mgrid_nnonic4_setup(Mgrid *, const MgridSystem *);
  int mgrid_pnonic4_setup(Mgrid *, const MgridSystem *);
  int mgrid_nhermite_setup(Mgrid *, const MgridSystem *);
  int mgrid_phermite_setup(Mgrid *, const MgridSystem *);
  int mgrid_setup_longrange(Mgrid *, const MgridSystem *);

#ifdef __cplusplus
}
#endif

#endif /* MGRID_H */
