/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * spcheck.c - simparam module implementation of simparam_check()
 *   got too long for simparam.c file
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "mdsim/const.h"
#include "mdsim/simparam.h"
#include "mdsim/system.h"
#include "mdsim/error.h"
#include "debug/debug.h"


/* prototypes for internal routines */
static int output(SimParam *p);
static int restart(SimParam *p);
static int dcd(SimParam *p);
static int results(SimParam *p, System *sys, MD_Engine *e);
static int potential(SimParam *p);
static int unsupported(SimParam *);


/*
 * call to validate simparams and setup additional
 * filenames and related data
 *
 * to be called after simparam_set() calls
 */
int simparam_check(SimParam *p, System *sys, MD_Engine *e)
{
  /* must provide at least one force field "parameters" (X-Plor) file */
  if (adt_getLengthList(&(p->parameters)) == 0) {
    return error("must specify \"parameters\" at least once");
  }

  /* must provide one "structure" (X-Plor PSF) file */
  if (p->structure == NULL) {
    return error("must specify \"structure\"");
  }

  /*
   * must provide "coordinates" (coordinate PDB) for initial positions
   *
   * may also provide "bincoordinates" (binary) to be used instead
   * of "coordinates" for initial positions - although then both will
   * be read
   */
  if (p->coordinates == NULL) {
    return error("must specify \"coordinates\"");
  }

  /* can't provide both "binvelocities" and "namdbinvelocities" */
  if (p->binvelocities && p->namdbinvelocities) {
    return error("must not specifiy both \"binvelocities\" and "
       "\"namdbinvelocities\"");
  }

  /* must provide one "outputname" for output of trajectories */
  if (p->outputname == NULL) {
    return error("must specify \"outputname\"");
  }

  /* must provide "restartname" if "restartfreq" is defined */
  if (p->restartfreq != 0 && p->restartname == NULL) {
    return error("must specify \"restartname\" with \"restartfreq\"");
  }

  /* must provide "restartfreq" if "restartname" is defined */
  if (p->restartname && p->restartfreq == 0) {
    return error("must specify \"restartfreq\" with \"restartname\"");
  }

  /* require "restartfreq" to be positive */
  if (p->restartfreq < 0) {
    return error("must have \"restartfreq\" > 0");
  }

  /* must provide "dcdfile" if "dcdfreq" is defined */
  if (p->dcdfreq != 0 && p->dcdfile == NULL) {
    return error("must specify \"dcdfile\" with \"dcdfreq\"");
  }

  /* must provide "dcdfreq" if "dcdfile" is defined */
  if (p->dcdfile && p->dcdfreq == 0) {
    return error("must specify \"dcdfreq\" with \"dcdfile\"");
  }

  /* require "dcdfreq" to be positive */
  if (p->dcdfreq < 0) {
    return error("must have \"dcdfreq\" > 0");
  }

  /* must provide "veldcdfile" if "veldcdfreq" is defined */
  if (p->veldcdfreq != 0 && p->veldcdfile == NULL) {
    return error("must specify \"veldcdfile\" with \"veldcdfreq\"");
  }

  /* must provide "veldcdfreq" if "veldcdfile" is defined */
  if (p->veldcdfile && p->veldcdfreq == 0) {
    return error("must specify \"veldcdfreq\" with \"veldcdfile\"");
  }

  /* require "veldcdfreq" to be positive */
  if (p->veldcdfreq < 0) {
    return error("must have \"veldcdfreq\" > 0");
  }

  /* require "outputenergies" to be positive */
  if (p->outputenergies < 0) {
    return error("must have \"outputenergies\" > 0");
  }

  /* require "numsteps" to be nonnegative */
  if (p->numsteps < 0) {
    return error("must have \"numsteps\" >= 0");
  }

  /* require "potentialName" if "potentialSelect" defined */
  if (adt_getLengthList(&(p->potential_select)) > 0
      && p->potential_name == NULL) {
    return error(
        "must define \"PotentialName\" if \"PotentialSelect\" is defined");
  }


  /*** set default values ***/

  if (p->binaryoutput == NULL
      && (p->binaryoutput = strdup("on")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->binaryrestart == NULL
      && (p->binaryrestart = strdup("on")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->binaryresults == NULL
      && (p->binaryresults = strdup("off")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->restartsave == NULL
      && (p->restartsave = strdup("off")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->dcdunitcell == NULL
      && (p->dcdunitcell = strdup("off")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->outputenergies == 0) {
    p->outputenergies = 1;
  }

  if (p->fixedatoms == NULL
      && (p->fixedatoms = strdup("off")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->fixedatomsfile == NULL
      && (p->fixedatomsfile = strdup(p->coordinates)) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->fixedatomscol == NULL
      && (p->fixedatomscol = strdup("O")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->constraints == NULL
      && (p->constraints = strdup("off")) == NULL) {
    return error(MSG_NOMEM);
  }

  if (p->consexp == 0) {
    p->consexp = 2;
  }

  /*** check more requirements ***/

  /* require "binaryoutput" to be either "on"/"yes" or "off"/"no" */
  if (strcasecmp(p->binaryoutput, "on") != 0
      && strcasecmp(p->binaryoutput, "yes") != 0
      && strcasecmp(p->binaryoutput, "off") != 0
      && strcmp(p->binaryoutput, "no") != 0) {
    return error("must have \"binaryoutput\" "
	"be \"on\"/\"yes\" or \"off\"/\"no\"");
  }

  /* require "binaryrestart" to be either "on"/"yes" or "off"/"no" */
  if (strcasecmp(p->binaryrestart, "on") != 0
      && strcasecmp(p->binaryrestart, "yes") != 0
      && strcasecmp(p->binaryrestart, "off") != 0
      && strcmp(p->binaryrestart, "no") != 0) {
    return error("must have \"binaryrestart\" "
	"be \"on\"/\"yes\" or \"off\"/\"no\"");
  }

  /* require "binaryresults" to be either "on"/"yes" or "off"/"no" */
  if (strcasecmp(p->binaryresults, "on") != 0
      && strcasecmp(p->binaryresults, "yes") != 0
      && strcasecmp(p->binaryresults, "off") != 0
      && strcmp(p->binaryresults, "no") != 0) {
    return error("must have \"binaryresults\" "
	"be \"on\"/\"yes\" or \"off\"/\"no\"");
  }

  /* require "restartsave" to be either "on"/"yes" or "off"/"no" */
  if (strcasecmp(p->restartsave, "on") != 0
      && strcasecmp(p->restartsave, "yes") != 0
      && strcasecmp(p->restartsave, "off") != 0
      && strcmp(p->restartsave, "no") != 0) {
    return error("must have \"restartsave\" "
	"be \"on\"/\"yes\" or \"off\"/\"no\"");
  }

  /* require "dcdunitcell" to be either "on"/"yes" or "off"/"no" */
  if (strcasecmp(p->dcdunitcell, "on") != 0
      && strcasecmp(p->dcdunitcell, "yes") != 0
      && strcasecmp(p->dcdunitcell, "off") != 0
      && strcmp(p->dcdunitcell, "no") != 0) {
    return error("must have \"dcdunitcell\" "
	"be \"on\"/\"yes\" or \"off\"/\"no\"");
  }

  /* require "fixedatoms" to be either "on"/"yes" or "off"/"no" */
  if (strcasecmp(p->fixedatoms, "on") != 0
      && strcasecmp(p->fixedatoms, "yes") != 0
      && strcasecmp(p->fixedatoms, "off") != 0
      && strcmp(p->fixedatoms, "no") != 0) {
    return error("must have \"fixedatoms\" "
	"be \"on\"/\"yes\" or \"off\"/\"no\"");
  }

  /* require "fixedatomscol" to be either "X", "Y", "Z", "O", or "B" */
  if (strcmp(p->fixedatomscol, "X") != 0
      && strcmp(p->fixedatomscol, "Y") != 0
      && strcmp(p->fixedatomscol, "Z") != 0
      && strcmp(p->fixedatomscol, "O") != 0
      && strcmp(p->fixedatomscol, "B") != 0) {
    return error("must have \"fixedatomscol\" "
	"be either \"X\", \"Y\", \"Z\", \"O\", or \"B\"");
  }

  /* require "constraints" to be either "on"/"yes" or "off"/"no" */
  if (strcasecmp(p->constraints, "on") != 0
      && strcasecmp(p->constraints, "yes") != 0
      && strcasecmp(p->constraints, "off") != 0
      && strcmp(p->constraints, "no") != 0) {
    return error("must have \"constraints\" "
	"be \"on\"/\"yes\" or \"off\"/\"no\"");
  }

  /*** set helper variable for constraints ***/
  p->is_constraints = (strcasecmp(p->constraints, "on") == 0
      || strcasecmp(p->constraints, "yes") == 0);

  /* require "consexp" to be positive, even integer */
  if (p->consexp <= 0 || (p->consexp & 0x01) != 0) {
    return error("must have \"consexp\" be positive, even integer");
  }

  /* require "conskcol" to be either "X", "Y", "Z", "O", or "B"
   * if constraints are active */
  if (p->is_constraints
      && (p->conskcol == NULL
        || (strcmp(p->conskcol, "X") != 0
          && strcmp(p->conskcol, "Y") != 0
          && strcmp(p->conskcol, "Z") != 0
          && strcmp(p->conskcol, "O") != 0
          && strcmp(p->conskcol, "B") != 0)
        )) {
    return error("must have \"conskcol\" "
	"be either \"X\", \"Y\", \"Z\", \"O\", or \"B\"");
  }

  if (p->is_constraints && p->consref == NULL) {
    return error("must set \"consref\" to filename "
        "when constraints are active");
  }

  if (p->is_constraints && p->conskfile == NULL) {
    return error("must set \"conskfile\" to filename "
        "when constraints are active");
  }

  /*** set additional helper variables ***/

  p->is_binaryoutput = (strcasecmp(p->binaryoutput, "on") == 0
      || strcasecmp(p->binaryoutput, "yes") == 0);
  p->is_binaryrestart = (strcasecmp(p->binaryrestart, "on") == 0
      || strcasecmp(p->binaryrestart, "yes") == 0);
  p->is_binaryresults = (strcasecmp(p->binaryresults, "on") == 0
      || strcasecmp(p->binaryresults, "yes") == 0);
  p->is_restartsave = (strcasecmp(p->restartsave, "on") == 0
      || strcasecmp(p->restartsave, "yes") == 0);
  p->is_dcdunitcell = (strcasecmp(p->dcdunitcell, "on") == 0
      || strcasecmp(p->dcdunitcell, "yes") == 0);
  p->is_fixedatoms = (strcasecmp(p->fixedatoms, "on") == 0
      || strcasecmp(p->fixedatoms, "yes") == 0);

  /* setup output filenames */
  if (output(p)) return error(MSG, "output() failed");

  /* setup restart filenames */
  if (p->restartname && restart(p)) return error(MSG, "restart() failed");

  /* setup dcd filenames */
  if ((p->dcdfile || p->veldcdfile) && dcd(p)) {
    return error(MSG, "dcd() failed");
  }

  /* setup results output */
  if (results(p, sys, e)) return error(MSG, "results() failed");

  /* setup potential output */
  if (p->potential_name && potential(p)) {
    return error(MSG, "potential() failed");
  }

  /*** check for unsupported features ***/
  if (unsupported(p)) return error(MSG, "unsupported() failed");

  return 0;
}


/*
 * setup output filenames
 *
 * return 0 for success, nonzero for failure
 */
int output(SimParam *p)
{
  int len;

  len = strlen(p->outputname) + 1;
  p->output_pos = (char *) malloc(len + strlen(FN_POS_EXT));
  p->output_vel = (char *) malloc(len + strlen(FN_VEL_EXT));
  len += strlen(FN_BAK_EXT);
  p->output_posbak = (char *) malloc(len + strlen(FN_POS_EXT));
  p->output_velbak = (char *) malloc(len + strlen(FN_VEL_EXT));
  if (p->output_pos == NULL || p->output_vel == NULL
      || p->output_posbak == NULL || p->output_velbak == NULL) {
    return error(MSG_NOMEM);
  }
  sprintf(p->output_pos, "%s%s", p->outputname, FN_POS_EXT);
  sprintf(p->output_vel, "%s%s", p->outputname, FN_VEL_EXT);
  sprintf(p->output_posbak, "%s%s", p->output_pos, FN_BAK_EXT);
  sprintf(p->output_velbak, "%s%s", p->output_vel, FN_BAK_EXT);
  return 0;
}


/*
 * setup restart filename buffers
 *
 * return 0 for success, nonzero for failure
 */
int restart(SimParam *p)
{
  char buf[32];  /* convert step number to string, is enough space */
  int slenlo;    /* string length of lowest step number */
  int slenhi;    /* string length of highest step number */
  int slenmax;   /* max string length for step numbers */
  int len;

  ASSERT(p->restartname != NULL);

  /* convert first and last time steps to strings, find max length */
  sprintf(buf, "%d", (int) p->firsttimestep);
  slenlo = strlen(buf);
  sprintf(buf, "%d", (int) (p->firsttimestep + p->numsteps));
  slenhi = strlen(buf);
  slenmax = (slenhi > slenlo ? slenhi : slenlo);
  p->restart_stepnum_width = slenmax;

  /* allocate enough space for each filename */
  len = strlen(p->restartname) + slenmax + 2;
  p->restart_pos = (char *) calloc(len + strlen(FN_POS_EXT), 1);
  p->restart_vel = (char *) calloc(len + strlen(FN_VEL_EXT), 1);
  len += strlen(FN_BAK_EXT);
  p->restart_posbak = (char *) calloc(len + strlen(FN_POS_EXT), 1);
  p->restart_velbak = (char *) calloc(len + strlen(FN_VEL_EXT), 1);
  if (p->restart_pos == NULL || p->restart_vel == NULL
      || p->restart_posbak == NULL || p->restart_velbak == NULL) {
    return error(MSG_NOMEM);
  }
  return 0;
}


int dcd(SimParam *p)
{
  int len;

  ASSERT(p->dcdfile != NULL || p->veldcdfile != NULL);

  /* create backup file for position DCD file */
  if (p->dcdfile != NULL) {
    len = strlen(p->dcdfile) + 1;
    p->dcdfile_bak = (char *) malloc(len + strlen(FN_BAK_EXT));
    if (p->dcdfile_bak == NULL) return error(MSG_NOMEM);
    sprintf(p->dcdfile_bak, "%s%s", p->dcdfile, FN_BAK_EXT);
  }

  /* create backup file for velocity DCD file */
  if (p->veldcdfile != NULL) {
    len = strlen(p->veldcdfile) + 1;
    p->veldcdfile_bak = (char *) malloc(len + strlen(FN_BAK_EXT));
    if (p->veldcdfile_bak == NULL) return error(MSG_NOMEM);
    sprintf(p->veldcdfile_bak, "%s%s", p->veldcdfile, FN_BAK_EXT);
  }
  return 0;
}


/*
 * setup results output
 *
 * includes header line of labels and byte offsets into
 * engine "result" data
 *
 * return 0 for success, nonzero for failure
 */
int results(SimParam *p, System *sys, MD_Engine *e)
{
  char buf[32];  /* convert step number to string, is enough space */
  int slenlo;    /* string length of lowest step number */
  int slenhi;    /* string length of highest step number */
  int slenmax;   /* max string length for step numbers */
  const int slenmin = 6;  /* length of "# step" on header line */
#ifdef DEBUG_SUPPORT
  int32 nres = 0;  /* count number of "results_info" list elements */
#endif
  int32 nlab = 0;  /* count number of "results_label" list elements */
  MD_Member m;     /* for type and array length of each "result" member */
  const char *cs;  /* to process each string in results array */
  int32 cslen;     /* length of cs string */
  ResultsInfo res; /* keep track of byte offset, type, and len for member */
  int i, j, slen;
  int results_len;
  char *s;

  /* setup default list results */
  if (adt_getLengthList(&(p->results)) == 0) {

    /*** for now, just output "energy" by default ***/
    if ((s = strdup("energy")) == NULL
        || adt_appendList(&(p->results), &s)) {
      return error(MSG_NOMEM);
    }
  }

  /*
   * how often do results get printed?
   * default to "outputenergies" if unspecified
   */
  if (p->resultsfreq == 0) {
    p->resultsfreq = p->outputenergies;
  }
  else if (p->resultsfreq < 0) {
    return error("must have \"resultsfreq\" > 0");
  }

  /* how often do headers get printed? */
  if (p->resultsheaderfreq == 0) {
    p->resultsheaderfreq = RESULTS_HEADER_FREQ;
  }
  else if (p->resultsheaderfreq < 0) {
    return error("must have \"resultsheaderfreq\" > 0");
  }

  /* create backup file name if results are saved to log file */
  if (p->resultsname) {
    p->results_bak = (char *) malloc(strlen(p->resultsname)
        + strlen(FN_BAK_EXT) + 1);
    if (p->results_bak == NULL) return error(MSG_NOMEM);
    sprintf(p->results_bak, "%s%s", p->resultsname, FN_BAK_EXT);
  }

  /* compute offsets, expanding arrays and vectors as scalars */
  results_len = adt_getLengthList(&(p->results));
  for (i = 0;  i < results_len;  i++) {

    /* obtain each string in results array */
    cs = *((const char **) adt_indexList(&(p->results), i));
    ASSERT(cs != NULL);
    cslen = strlen(cs);

    /* lookup cs as member of engine "result", compute its byte offset */
    res.offset =
      ((char *) MD_type_member(e, sys->result_attr.type, NULL, cs, &m))
      - ((char *) NULL);
    res.type = m.type;
    res.len = m.len;

    /* make sure cs is the name of a member */
    if (MD_errnum(e)) {
      return error("\"%s\" is not valid for \"results\"", cs);
    }

    /* make sure engine "result" member has primary type */
    if (m.type != MD_CHAR && m.type != MD_INT32
        && m.type != MD_FLOAT && m.type != MD_DOUBLE
        && m.type != MD_FVEC && m.type != MD_DVEC
        && m.type != MD_NAME && m.type != MD_STRING
        && m.type != MD_MESSAGE) {
      char msg[100];
      sprintf(msg, "member \"%.60s\" does not have primary type", cs);
      return error(MSG_ENGINE, msg);
    }

    /* see if it is already a scalar (non-array or string) */
    if (m.len == 1 || m.type == MD_CHAR) {

      /* store offset */
      if (adt_appendList(&(p->results_info), &res)) {
        return error(MSG_NOMEM);
      }
      ASSERT( ++nres );

      /* expand 3D-vector "scalar" into separate _x, _y, _z labels */
      if (m.type == MD_DVEC || m.type == MD_FVEC) {
        char *s[3];

        /* store separate x,y,z labels */
        if ((s[0] = (char *) malloc(cslen + 3)) == NULL
            || (s[1] = (char *) malloc(cslen + 3)) == NULL
            || (s[2] = (char *) malloc(cslen + 3)) == NULL
            || adt_appendList(&(p->results_label), &(s[0]))
            || adt_appendList(&(p->results_label), &(s[1]))
            || adt_appendList(&(p->results_label), &(s[2]))) {
          return error(MSG_NOMEM);
        }
        nlab += 3;
        sprintf(s[0], "%s_x", cs);
        sprintf(s[1], "%s_y", cs);
        sprintf(s[2], "%s_z", cs);
      }

      /* otherwise, just copy label as is */
      else {
        if ((s = strdup(cs)) == NULL
            || adt_appendList(&(p->results_label), &s)) {
          return error(MSG_NOMEM);
        }
        ++nlab;
      }
      continue;
    }

    /* otherwise store ResultsInfo for each individual array element */
    res.len = 1;

    /* loop over array, expand labels as _1, _2, _3 (1-based indexing) */
    for (j = 0;  j < m.len;  j++) {
      slen = cslen;

      /* find string length of index */
      sprintf(buf, "%d", j+1);
      slen += strlen(buf) + 1;

      /* store each offset */
      if (adt_appendList(&(p->results_info), &res)) {
        return error(MSG_NOMEM);
      }
      ASSERT( ++nres );

      /* increment offset for next ResultsInfo element */
      res.offset += MD_SIZEOF(m.type);

      /*
       * expand 3D-vector "scalar" into separate _x, _y, _z labels
       * (e.g. for mth array element labels are _m_x, _m_y, _m_z)
       */
      if (m.type == MD_DVEC || m.type == MD_FVEC) {
        char *s[3];

        /* store separate x,y,z labels */
        if ((s[0] = (char *) malloc(cslen + 3)) == NULL
            || (s[1] = (char *) malloc(cslen + 3)) == NULL
            || (s[2] = (char *) malloc(cslen + 3)) == NULL
            || adt_appendList(&(p->results_label), &(s[0]))
            || adt_appendList(&(p->results_label), &(s[1]))
            || adt_appendList(&(p->results_label), &(s[2]))) {
          return error(MSG_NOMEM);
        }
        nlab += 3;
        sprintf(s[0], "%s_%s_x", cs, buf);
        sprintf(s[1], "%s_%s_y", cs, buf);
        sprintf(s[2], "%s_%s_z", cs, buf);
      }

      /* otherwise, just append index to label */
      else {
        if ((s = (char *) malloc(slen + 1)) == NULL
            || adt_appendList(&(p->results_label), &s)) {
          return error(MSG_NOMEM);
        }
        ++nlab;
        sprintf(s, "%s_%s", cs, buf);
      }
    }

  } /* end loop over "results" array */

  /* set default width if too small */
  if (p->resultswidth <= 1) {
    p->resultswidth = RESULTS_WIDTH;
  }

  /* convert first and last time steps to strings, find max length */
  sprintf(buf, "%d", (int) p->firsttimestep);
  slenlo = strlen(buf);
  sprintf(buf, "%d", (int) (p->firsttimestep + p->numsteps));
  slenhi = strlen(buf);
  slenmax = (slenhi > slenlo ? slenhi : slenlo);
  if (slenmax < slenmin) slenmax = slenmin;
  p->results_stepnum_width = slenmax;

  /* header contains entries from "results_label" */
  p->results_header =
    (char *) malloc(nlab * p->resultswidth + slenmax + 1);
  if (p->results_header == NULL) return error(MSG_NOMEM);

  /* create header */
  s = p->results_header;
  s += sprintf(p->results_header, "#%*s", slenmax - 1, "step");
  for (i = 0;  i < nlab;  i++) {
    cs = *((const char **) adt_indexList(&(p->results_label), i));
    ASSERT(cs != NULL);
    s += sprintf(s, "%*.*s", p->resultswidth, p->resultswidth - 1, cs);
    if (strlen(cs) >= p->resultswidth) {
      warn("results label \"%s\" will be truncated to \"%.*s\"",
          cs, p->resultswidth - 1, cs);
    }
  }
  ASSERT(adt_getLengthList(&(p->results_label)) == nlab);
  ASSERT(adt_getLengthList(&(p->results_info)) == nres);

#if 0
  /*** FOR DEBUGGING ***/
  printf("space allocated:  %d\n", nlab * p->resultswidth + slenmax + 1);
  printf("length of header: %d\n", strlen(p->results_header));
  printf("%s\n", p->results_header);
  printf("nlab = %d   p->results_label.len = %d\n",
      (int) nlab, (int) (p->results_label.len));
  printf("nres = %d   p->results_info.len = %d\n",
      (int) nres, (int) (p->results_info.len));
  printf("offset:");
  for (i = 0;  i < nres;  i++) {
    printf("  %d", (int) *((int32*) ADT_array_elem(&(p->results_info),i)));
  }
  printf("\n");
  printf("size of engine result = %d\n", MD_SIZEOF(sys->result_attr.type));
  exit(0);  /*** FOR DEBUGGING ***/
#endif
  return 0;
}


int potential(SimParam *p)
{
  const char *cs;
  char *s;
  int i, potential_len;

  /* error if no potentials requested */
  if (adt_getLengthList(&(p->potential_select)) == 0) {
    BUG("no potentials were selected");
  }

  /* require "potentialAlias" to be same length as "potentialSelect" */
  if (adt_getLengthList(&(p->potential_alias))
      != adt_getLengthList(&(p->potential_select))) {
    return error("defined \"PotentialAlias\" too many times");
  }

  /*
   * how often do potentials get printed?
   * default to "outputenergies" if unspecified
   */
  if (p->potential_freq == 0) {
    p->potential_freq = p->outputenergies;
  }
  else if (p->potential_freq < 0) {
    return error("must have \"PotentialFreq\" > 0");
  }

  /* how often do headers get printed? */
  if (p->potential_headerfreq == 0) {
    p->potential_headerfreq = POTENTIAL_HEADER_FREQ;
  }
  else if (p->potential_headerfreq < 0) {
    return error("must have \"PotentialHeaderfreq\" > 0");
  }

  /* create backup file name */
  ASSERT(p->potential_name != NULL);
  p->potential_bak = (char *) malloc(strlen(p->potential_name)
      + strlen(FN_BAK_EXT) + 1);
  if (p->potential_bak == NULL) return error(MSG_NOMEM);
  sprintf(p->potential_bak, "%s%s", p->potential_name, FN_BAK_EXT);

  /* set default width if too small */
  if (p->potential_width <= 1) {
    p->potential_width = POTENTIAL_WIDTH;
  }

  /* header contains entries from "potential_alias" */
  /* (reuse "results_stepnum_width") */
  potential_len = adt_getLengthList(&(p->potential_alias));
  ASSERT(adt_getLengthList(&(p->potential_select)) == potential_len);
  p->potential_header = (char *) malloc(potential_len * p->potential_width
      + p->results_stepnum_width + 1);
  if (p->potential_header == NULL) return error(MSG_NOMEM);

  /* create header */
  s = p->potential_header;
  s += sprintf(p->potential_header, "#%*s",
      p->results_stepnum_width - 1, "step");
  for (i = 0;  i < potential_len;  i++) {
    cs = *((const char **) adt_indexList(&(p->potential_alias), i));
    ASSERT(cs != NULL);
    s += sprintf(s, "%*.*s", p->potential_width, p->potential_width - 1, cs);
    if (strlen(cs) >= p->potential_width) {
      warn("potential label \"%s\" will be truncated to \"%.*s\"",
          cs, p->potential_width - 1, cs);
    }
  }
  return 0;
}


/*
 * check on features that are not yet supported
 *
 * return 0 for success, nonzero for failure
 */
int unsupported(SimParam *p)
{
  if (p->is_binaryresults) {
    return error("binary results file logging is not yet supported");
  }
  return 0;
}
