/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * simparam.c - simparam module implementation,
 *   purpose is to parse and store front end simulation parameters,
 *   also see spcheck.c for implementation of simparam_check()
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* #define DEBUG_WATCH */
/* #define DEBUG_SUPPORT */
#include "mdsim/const.h"
#include "mdsim/simparam.h"
#include "mdsim/error.h"
#include "debug/debug.h"


/* create new restart filenames using step number */
int simparam_restart(SimParam *p, int32 stepnum)
{
  if (p->is_restartsave) {
    if (sprintf(p->restart_pos, "%s.%d%s", p->restartname,
          stepnum, FN_POS_EXT) <= 0
        || sprintf(p->restart_vel, "%s.%d%s", p->restartname,
          stepnum, FN_VEL_EXT) <= 0) {
      return FAIL;
    }
  }
  else if (sprintf(p->restart_pos, "%s%s", p->restartname, FN_POS_EXT) <= 0
      || sprintf(p->restart_vel, "%s%s", p->restartname, FN_VEL_EXT) <= 0) {
    return FAIL;
  }
  return 0;
}


/* create new backup restart filenames from current restart filenames */
int simparam_restartbak(SimParam *p)
{
  ASSERT(p->restart_pos[0] != '\0');
  ASSERT(p->restart_vel[0] != '\0');
  if (sprintf(p->restart_posbak, "%s%s", p->restart_pos, FN_BAK_EXT) <= 0
      || sprintf(p->restart_velbak, "%s%s", p->restart_vel, FN_BAK_EXT) <= 0) {
    return FAIL;
  }
  return 0;
}


/******************************************************************************
 *
 * Source code below deals with SimParam members *generically* through
 * the Member[] array.  For details, see the notes that follow.
 *
 ******************************************************************************/

/*
 * Notes:
 *
 * Generically parse value strings and store into appropriate SimParam
 * members.  The functionality of the simparam module routines is driven
 * by the MemberInfo array (see below) by identifying the keyword (stored
 * using member table hashing), parsing the value string based on the
 * indicated type, and storing to the address offset of the corresponding
 * SimParam member.
 *
 * Extending this to recognize additional simulation parameters is simple,
 * with only two code changes required:
 *
 * 1.  Add the new external data member to SimParam type definition
 *     (in simparam.h header).
 *
 * 2.  Put a corresponding entry into Member[] array below.  To make
 *     life easier, make the SimParam member name match the keyword
 *     string name.  Include the offset and correct type.  Note that
 *     the order of Member[] array entries *does not matter*.
 *
 * There are six types supported:  string (arbitrary length), array of
 * string, int32, double, filename (either absolute path or preceded by
 * cwd - current working directory), and array of filename.  The array type
 * is used only for "parameters" (to read multiple force field parameter
 * files).  If other array types are needed (say, for string or int32), the
 * type constants will have to be modified with another entry to the store()
 * switch below, and possibly some additions to the constructor and
 * destructor.
 *
 * Caveats:  "cwd" must be initialized before any filename types
 * (also explained in simparam.h).
 */

/* types */
enum {
  NONE = 0,  /* indicates type value not assigned */
  STRING,    /* string */
  STRARRAY,  /* array of strings */
  FNAME,     /* filename is a string (either absolute or preceded by cwd) */
  FNARRAY,   /* array of filenames (either absolute or preceded by cwd) */
  INT32,     /* 4-byte integer from mdapi/mdtypes.h */
  DOUBLE     /* 8-byte floating point */
};

/* need an instantiation to compute member offsets */
static SimParam SP;

/* macro to provide number of elements in static array */
#define NELEMS(x)  (sizeof(x) / sizeof(x[0]))

/* macro to compute offsets using static version */
#define OFFSET(x)  ((const char *) &(SP.x) - (const char *) &SP)

/* type definition for table elements below */
typedef struct MemberInfo_tag {
  const char *keywd;
  int32 offset;
  int32 type;
} MemberInfo;

/*
 * table of SimParam member types and offsets
 * note: order of entries does not matter,
 *   "cwd" and "engine" not present,
 *   must be set using simparam_setup()
 */
static const MemberInfo Member[] = {
  { "parameters", OFFSET(parameters), FNARRAY },
  { "structure", OFFSET(structure), FNAME },
  { "coordinates", OFFSET(coordinates), FNAME },
  { "bincoordinates", OFFSET(bincoordinates), FNAME },
  { "velocities", OFFSET(velocities), FNAME },
  { "binvelocities", OFFSET(binvelocities), FNAME },
  { "namdbinvelocities", OFFSET(namdbinvelocities), FNAME },
  { "fixedatoms", OFFSET(fixedatoms), STRING },
  { "fixedatomsfile", OFFSET(fixedatomsfile), FNAME },
  { "fixedatomscol", OFFSET(fixedatomscol), STRING },
  { "constraints", OFFSET(constraints), STRING },
  { "consexp", OFFSET(consexp), INT32 },
  { "consref", OFFSET(consref), FNAME },
  { "conskfile", OFFSET(conskfile), FNAME },
  { "conskcol", OFFSET(conskcol), STRING },
  { "outputname", OFFSET(outputname), FNAME },
  { "binaryoutput", OFFSET(binaryoutput), STRING },
  { "binaryrestart", OFFSET(binaryrestart), STRING },
  { "binaryresults", OFFSET(binaryresults), STRING },
  { "restartname", OFFSET(restartname), FNAME },
  { "restartsave", OFFSET(restartsave), STRING },
  { "restartfreq", OFFSET(restartfreq), INT32 },
  { "dcdfile", OFFSET(dcdfile), FNAME },
  { "dcdfreq", OFFSET(dcdfreq), INT32 },
  { "veldcdfile", OFFSET(veldcdfile), FNAME },
  { "veldcdfreq", OFFSET(veldcdfreq), INT32 },
  { "dcdunitcell", OFFSET(dcdunitcell), STRING },
  { "outputenergies", OFFSET(outputenergies), INT32 },
  { "firsttimestep", OFFSET(firsttimestep), INT32 },
  { "numsteps", OFFSET(numsteps), INT32 },
  { "results", OFFSET(results), STRARRAY },
  { "resultsname", OFFSET(resultsname), FNAME },
  { "resultsfreq", OFFSET(resultsfreq), INT32 },
  { "resultswidth", OFFSET(resultswidth), INT32 },
  { "resultsheaderfreq", OFFSET(resultsheaderfreq), INT32 },
  { "potentialname", OFFSET(potential_name), STRING },
  { "potentialselect", OFFSET(potential_select), STRARRAY },
  { "potentialalias", OFFSET(potential_alias), STRARRAY },
  { "potentialfreq", OFFSET(potential_freq), INT32 },
  { "potentialwidth", OFFSET(potential_width), INT32 },
  { "potentialheaderfreq", OFFSET(potential_headerfreq), INT32 },
};

/* prototype internal routines */
static int store(SimParam *p, int32 offset, int32 type, const char *s);

/*
 * prepare for simparam_set() calls
 *
 * sets "cwd" (current working directory) and
 * "engine" (name of engine) simparams
 */
int simparam_setup(SimParam *p, const char *cwd, const char *engine)
{
  ASSERT(cwd != NULL);
  ASSERT(engine != NULL);

  /* make sure these are not already called */
  if (p->cwd) BUG("cwd already set");
  else if (p->engine) BUG("engine already set");

  if ((p->cwd = strdup(cwd)) == NULL
      || (p->engine = strdup(engine)) == NULL) {
    return error(MSG_NOMEM);
  }
  return 0;
}

/* constructor for SimParam */
int simparam_init(SimParam *p)
{
  int32 k;

  /* clear simparam data space */
  memset(p, 0, sizeof(SimParam));
  /* init table of members */
  if (adt_initializeTable(&(p->member), 0)) {
    return error(MSG_NOMEM);
  }
  for (k = 0;  k < NELEMS(Member);  k++) {
    if (adt_insertTable(&(p->member), Member[k].keywd, k) != k) {
      return error(MSG_NOMEM);
    }
    if (Member[k].type == STRARRAY || Member[k].type == FNARRAY) {
      /* init array */
      adt_List *pa = (adt_List *)((char *) p + Member[k].offset);
      if (adt_initializeList(pa, sizeof(char *), 0, NULL)) {
        return error(MSG_NOMEM);
      }
    }
  }
  /* init non-generic members */
  if (adt_initializeList(&(p->results_info), sizeof(ResultsInfo), 0, NULL)
      || adt_initializeList(&(p->results_label), sizeof(char *), 0, NULL)) {
    return error(MSG_NOMEM);
  }
  return 0;
}

/* destructor for SimParam */
void simparam_done(SimParam *p)
{
  int32 j, k, len;

  /* free non-generic members */
  free(p->output_pos);
  free(p->output_vel);
  free(p->output_posbak);
  free(p->output_velbak);
  free(p->restart_pos);
  free(p->restart_vel);
  free(p->restart_posbak);
  free(p->restart_velbak);
  free(p->results_header);
  free(p->results_bak);
  adt_cleanupList(&(p->results_info));
  len = (int32) adt_getLengthList(&(p->results_label));
  for (k = 0;  k < len;  k++) {
    free(*((char **) adt_indexList(&(p->results_label), k)));
  }
  adt_cleanupList(&(p->results_label));

  /* free generic members */
  for (k = 0;  k < NELEMS(Member);  k++) {
    if (Member[k].type == STRARRAY || Member[k].type == FNARRAY) {
      /* free array */
      adt_List *pa = (adt_List *)((char *) p + Member[k].offset);
      len = (int32) adt_getLengthList(pa);
      for (j = 0;  j < len;  j++) {
        free(*((char **) adt_indexList(pa, j)));
      }
      adt_cleanupList(pa);
    }
    else if (Member[k].type == STRING || Member[k].type == FNAME) {
      /* free string */
      char **ps = (char **)((char *) p + Member[k].offset);
      free(*ps);
    }
  }

  /* free table of members */
  adt_cleanupTable(&(p->member));

  /* clear memory */
  memset(p, 0, sizeof(SimParam));
}

#ifdef DEBUG_SUPPORT
/* print SimParam members to standard output */
int simparam_debug(SimParam *p)
{
  int32 k, j, len, n;
  double d;
  const char *s;
  adt_List *pa;

  printf("===== listing of simparam contents =====\n");
  for (k = 0;  k < NELEMS(Member);  k++) {
    printf("simparam: ");
    switch (Member[k].type) {
      case INT32:
        n = *((int32 *)((char *) p + Member[k].offset));
        printf("%s = %d\n", Member[k].keywd, n);
        break;
      case DOUBLE:
        d = *((double *)((char *) p + Member[k].offset));
        printf("%s = %g\n", Member[k].keywd, d);
        break;
      case STRING:
      case FNAME:
        s = *(const char **)((char *) p + Member[k].offset);
        if (s != NULL) {
          printf("%s = %s\n", Member[k].keywd, s);
        }
        else {
          printf("%s = (nil)\n", Member[k].keywd);
        }
        break;
      case STRARRAY:
      case FNARRAY:
        pa = (adt_List *)((char *) p + Member[k].offset);
        len = (int32) adt_getLengthList(pa);
        if (len > 0) {
          /*
           * don't access 0th element unless there is something in array
           * otherwise, the array length will grow!
           *
           * --> no longer the case with revised ADT library
           */
          s = *((const char **) adt_indexList(pa, 0));
          printf("%s = %s\n", Member[k].keywd, s);
          for (j = 1;  j < len;  j++) {
            s = *((const char **) adt_indexList(pa, j));
            printf("simparam: %s = %s\n", Member[k].keywd, s);
          }
        }
        else {
          printf("%s = (empty)\n", Member[k].keywd);
        }
        break;
      default:
        BUG("unexpected member type");
    }
  }
  return 0;
}
#endif

/* set SimParam member */
int simparam_set(SimParam *p, const char *keywd, const char *val)
{
  int32 k;

  /* lookup keyword in table of members */
  if ((k = adt_lookupTable(&(p->member), keywd)) == ADT_ERROR) {
    return NOTFOUND;
  }
  else {
    return store(p, Member[k].offset, Member[k].type, val);
  }
}

/*
 * parse config file string
 * store value into SimParam member
 *
 * returns 0 for success
 * on error, takes assignment as far as possible, returning:
 *   REINIT - simparam reinitialized
 *   BADVAL - illegal value assigned
 *   REINIT | BADVAL - both
 *   error(MSG_NOMEM) - (FAIL) can't allocate memory
 */
int store(SimParam *p, int32 offset, int32 type, const char *s)
{
  void *v = (void *)((char *) p + offset);
  const char *cwd = (p->cwd ? p->cwd : "");
  int32 cwdlen = strlen(cwd);
  int32 slen = strlen(s);
  int32 noslash = (cwdlen > 0 && cwd[cwdlen - 1] != '/');
  char **ps;
  char *pc;
  adt_List *pa;
  int32 *pi;
  double *pd;
  int retval = 0;
  int k;
  char c;

  switch (type) {
    case STRING:
      ps = (char **) v;
      if (*ps != NULL) {
        free(*ps);
        retval |= REINIT;
      }
      if ((*ps = strdup(s)) == NULL) return error(MSG_NOMEM);
      break;
    case FNAME:
      ASSERT(cwdlen > 0);
      ASSERT(noslash == 0 || noslash == 1);
      ps = (char **) v;
      if (*ps != NULL) {
        free(*ps);
        retval |= REINIT;
      }
      if (s[0] == '/') {
        if ((*ps = strdup(s)) == NULL) return error(MSG_NOMEM);
      }
      else {
        if ((*ps = (char *) malloc(cwdlen + noslash + slen + 1)) == NULL) {
          return error(MSG_NOMEM);
        }
        strcpy(*ps, cwd);
        if (noslash) strcat(*ps, "/");
        strcat(*ps, s);
      }
      break;
    case STRARRAY:
      pa = (adt_List *) v;
      if ((pc = strdup(s)) == NULL || adt_appendList(pa, &pc)) {
        return error(MSG_NOMEM);
      }
      break;
    case FNARRAY:
      ASSERT(cwdlen > 0);
      ASSERT(noslash == 0 || noslash == 1);
      pa = (adt_List *) v;
      if (s[0] == '/') {
        if ((pc = strdup(s)) == NULL) return error(MSG_NOMEM);
      }
      else {
        if ((pc = (char *) malloc(cwdlen + noslash + slen + 1)) == NULL) {
          return error(MSG_NOMEM);
        }
        strcpy(pc, cwd);
        if (noslash) strcat(pc, "/");
        strcat(pc, s);
      }
      if (adt_appendList(pa, &pc)) {
        return error(MSG_NOMEM);
      }
      break;
    case INT32:
      pi = (int32 *) v;
      if (*pi != 0) retval |= REINIT;
      switch (sscanf(s, "%d%c", &k, &c)) {
        case 0:
          return (retval | BADVAL);
        case 2:
          retval |= BADVAL;
      }
      *pi = k;
      break;
    case DOUBLE:
      pd = (double *) v;
      if (*pd != 0.0) retval |=  REINIT;
      switch (sscanf(s, "%lf%c", pd, &c)) {
        case 0:
          return (retval | BADVAL);
        case 2:
          retval |= BADVAL;
      }
      break;
    default:
      BUG("invalid type for parsing SimParam member");
  }
  return retval;
}


char *simparam_potential_nextlabel(SimParam *p, const char *sel)
{
  char *alias;
  char name[8];
  int n;

  switch (simparam_potential_type(p, sel)) {
    case BOND:
      n = ++(p->bondcnt);
      strcpy(name, "bond");
      break;
    case ANGLE:
      n = ++(p->anglecnt);
      strcpy(name, "angle");
      break;
    case DIHED:
      n = ++(p->dihedcnt);
      strcpy(name, "dihed");
      break;
    case IMPR:
      n = ++(p->imprcnt);
      strcpy(name, "impr");
      break;
    case ELEC:
      n = ++(p->eleccnt);
      strcpy(name, "elec");
      break;
    case VDW:
      n = ++(p->vdwcnt);
      strcpy(name, "vdw");
      break;
    case BOUND:
      n = ++(p->boundcnt);
      strcpy(name, "bound");
      break;
    case EPOT:
      n = ++(p->epotcnt);
      strcpy(name, "epot");
      break;
    default:
      error("(%s,%d): illegal value for PotentialSelect \"%s\"",
          __FILE__, __LINE__, sel);
      return NULL;
  }

  alias = (char *) malloc(20);  /* string of 19 chars should be long enough */
  if (alias == NULL) {
    error(MSG_NOMEM);
    return NULL;
  }
  snprintf(alias, 20, "%s%d", name, n);
  return alias;
}


int simparam_potential_type(SimParam *p, const char *sel)
{
  char name[8];
  int a, b, c, d;
  int type = INVALID;
  char ch;

  if (sscanf(sel, "%7s", name) == 1) {
    if (strcasecmp(name, "bond") == 0 
        && sscanf(sel, "%*s %d %d%c", &a, &b, &ch) == 2) {
      type = BOND;
    }
    else if (strcasecmp(name, "angle") == 0 
        && sscanf(sel, "%*s %d %d %d%c", &a, &b, &c, &ch) == 3) {
      type = ANGLE;
    }
    else if (strcasecmp(name, "dihed") == 0 
        && sscanf(sel, "%*s %d %d %d %d%c", &a, &b, &c, &d, &ch) == 4) {
      type = DIHED;
    }
    else if (strcasecmp(name, "impr") == 0 
        && sscanf(sel, "%*s %d %d %d %d%c", &a, &b, &c, &d, &ch) == 4) {
      type = IMPR;
    }
    else if (strcasecmp(name, "elec") == 0 
        && sscanf(sel, "%*s %d%c", &a, &ch) == 1) {
      type = ELEC;
    }
    else if (strcasecmp(name, "vdw") == 0 
        && sscanf(sel, "%*s %d%c", &a, &ch) == 1) {
      type = VDW;
    }
    else if (strcasecmp(name, "bound") == 0 
        && sscanf(sel, "%*s %d%c", &a, &ch) == 1) {
      type = BOUND;
    }
    else if (strcasecmp(name, "epot") == 0 
        && sscanf(sel, "%*s %d%c", &a, &ch) == 1) {
      type = EPOT;
    }
  }
  return type;
}
