/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * sim.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <errno.h>
/* #define DEBUG_WATCH */
/* #define DEBUG_SUPPORT */
#include "mdsim/const.h"
#include "mdsim/sim.h"
/* #include "mdsim/defeng.h" */
/* #include "simen/simen.h" */
#include "deven/deven.h"
#include "mdsim/error.h"
#include "debug/debug.h"


int sim_init(Sim *s)
{
  /* clear memory for config file info */
  s->info = NULL;
  s->info_len = 0;

  /* initialize config file reader */
  if (mdio_initializeConfig(&(s->cfg))) {
    return error(MSG, "mdio_initializeConfig() failed");
  }
  mdio_setErrorHandlerFile((mdio_File *) &(s->cfg), mdio_errhandler);

  /* initialize keyword array */
  if (adt_initializeList(&(s->keyword), sizeof(char *), 0, NULL)) {
    return error(MSG_NOMEM);
  }

  /* initialize simparam object */
  if (simparam_init(&(s->simparam))) {
    return error(MSG, "simparam_init() failed");
  }

  /* initialize engparam object */
  if (engparam_init(&(s->engparam))) {
    return error(MSG, "engparam_init() failed");
  }

  /* initialize system object */
  if (system_init(&(s->system))) {
    return error(MSG, "system_init() failed");
  }

  /* initialize callback object */
  if (callback_init(&(s->callback))) {
    return error(MSG, "callback_init() failed");
  }

  /*** engine interface is initialized later ***/

  return 0;
}


int sim_input_configfile(Sim *s, int cfg_argc, char *cfg_argv[])
{
  SimParam *p = &(s->simparam);
  mdio_Config *cfg = &(s->cfg);
  int32 k;
  const char *cfgname = NULL;
  const char *cwd = NULL;
  const char *engine = NULL;
  char *defcwd = NULL;  /* default cwd */

  /* verify that there is at least one config file in list */
  if (cfg_argc < 1) {
    BUG("need at least one config file in list");
  }

  /* take default cwd from first name */
  cfgname = cfg_argv[0];

  /* read config files */
  for (k = 0;  k < cfg_argc;  k++) {
    if (mdio_readConfig(cfg, cfg_argv[k])) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) cfg));
    }
  }

  /* obtain raw config data */
  s->info = mdio_getDataConfig(cfg, &(s->info_len));

  /*
   * search config data for two keywords:
   *   cwd - current working directory for finding files
   *   engine - name of engine to use
   *
   * signal that this simparam has been processed by
   * perverting its mdio_ConfigData entry: negate the line number
   */
  for (k = 0;  k < s->info_len;  k++) {
    /* check for cwd */
    if (strcasecmp(s->info[k].keyword, "cwd") == 0) {
      if (cwd) {  /* warning if reinitializing */
        warn("(%s,%d): reinitialized simparam \"%s\"",
            s->info[k].filename, s->info[k].linenum, s->info[k].keyword);
      }
      cwd = s->info[k].value;  /* point to it */
      s->info[k].linenum = -s->info[k].linenum;  /* signal it processed */
    }
    /* otherwise check for engine */
    else if (strcasecmp(s->info[k].keyword, "engine") == 0) {
      if (engine) {  /* warning if reinitializing */
        warn("(%s,%d): reinitialized simparam \"%s\"",
            s->info[k].filename, s->info[k].linenum, s->info[k].keyword);
      }
      engine = s->info[k].value;  /* point to it */
      s->info[k].linenum = -s->info[k].linenum;  /* signal it processed */
    }
  }

  /* if cwd not yet defined, then set defcwd to path of config file */
  if (cwd == NULL) {
    char *end = strrchr(cfgname, '/');
    if (end == NULL) {
      if ((defcwd = strdup(".")) == NULL) {
        return error(MSG_NOMEM);
      }
    }
    else {
      if ((defcwd = (char *) malloc(end - cfgname + 1)) == NULL) {
        return error(MSG_NOMEM);
      }
      strncpy(defcwd, cfgname, end - cfgname);
      /* have to nil-terminate after strncpy() */
      defcwd[end - cfgname] = '\0';
    }
  }

  /* setup special simparams */
  if (simparam_setup(p, (cwd ? cwd : defcwd), (engine ? engine : "default"))) {
    free(defcwd);
    return error("simparam_setup() failed");
  }
  free(defcwd);

  return 0;
}


int sim_start_engine(Sim *s)
{
  MD_Engine *e = &(s->engine);
  SimParam *p = &(s->simparam);
  EngParam *ep = &(s->engparam);
  System *sys = &(s->system);

  /* init engine */
#if 0
  /*** for now, use default engine no matter what ***/
  if (MD_init(e, p->engine, 0, default_engine_init, default_engine_done)
      || MD_wait(e)) {
    return error(MSG_MDAPI, "from MD_init()", MD_errmsg(e));
  }
#endif
#if 0
  /** for now, use simen no matter what ***/
  if (MD_init(e, p->engine, 0, simen_init, simen_done) || MD_wait(e)) {
    return error(MSG_MDAPI, "from MD_init()", MD_errmsg(e));
  }
#endif
  /** for now, use deven no matter what ***/
  if (MD_init(e, p->engine, 0, deven_init, deven_done) || MD_wait(e)) {
    return error(MSG_MDAPI, "from MD_init()", MD_errmsg(e));
  }

  /* setup engparam for this engine */
  if (engparam_setup(ep, e)) {
    return error(MSG, "engparam_setup() failed");
  }

  /* setup system for this engine */
  if (system_setup(sys, e)) {
    return error(MSG, "system_setup() failed");
  }
  return 0;
}


int sim_parse_configfile(Sim *s)
{
  SimParam *sp = &(s->simparam);
  EngParam *ep = &(s->engparam);
  System *sys = &(s->system);
  MD_Engine *e = &(s->engine);
  mdio_ConfigData *c = s->info;
  adt_List *keyword = &(s->keyword);
  char *str;
  char *pch;
  int32 k, val;

  for (k = 0;  k < s->info_len;  k++) {

    /* duplicate config file keyword into this keyword buffer */
    if ((str = strdup(c[k].keyword)) == NULL) {
      return error(MSG_NOMEM);
    }

    /* convert buffered keyword to lower case */
    for (pch = str;  *pch != '\0';  pch++) {
      *pch = tolower(*pch);
    }

    /* store another keyword into array (this increases array length) */
    if (adt_appendList(keyword, &str)) {
      return error(MSG_NOMEM);
    }

    if (c[k].linenum < 0) {
      continue;  /* already processed */
    }
    else if ((val = simparam_set(sp, str, c[k].value)) == 0) {

      /*** must deal with potential select/alias context ***/
      /* if alias was not set for next to last select, set default label */
      if (adt_getLengthList(&(sp->potential_select))
          > adt_getLengthList(&(sp->potential_alias)) + 1) {
        /* make life easier by defining variables */
        adt_List *select = &(sp->potential_select);
        adt_List *alias = &(sp->potential_alias);
        int select_len = adt_getLengthList(select);
        char *selstr = *((char **) adt_indexList(select, select_len - 2));
        char *alistr = simparam_potential_nextlabel(sp, selstr);

        /* check memory allocation for alistr */
        if (alistr == NULL) {
          return FAIL;
        }
        /* append default label for selected potential */
        if (adt_appendList(alias, &alistr)) {
          free(alistr);
          return error(MSG_NOMEM);
        }
        ASSERT(adt_getLengthList(alias) == select_len - 1);
      }

      continue;  /* successfully set simparam */
    }
    else if (val == FAIL) {
      return error(MSG, "simparam_set() failed");
    }
    else if (val == NOTFOUND &&
        (val = engparam_set(ep, str, c[k].value)) == 0) {
      continue;  /* successfully set engparam */
    }
    else if (val == FAIL) {
      return error(MSG, "engparam_set() failed");
    }
    else {       /* some warning(s) occurred */
      if (val & ~(NOTFOUND | NOTPARAM | REINIT | BADVAL | TRUNC)) {
        BUG("unexpected return value from simparam_set() or engparam_set()");
      }
      /* multiple warnings might be set (can't use else-if) */
      if (val & NOTFOUND) {
        warn("(%s,%d): keyword \"%s\" is not recognized",
            c[k].filename, c[k].linenum, c[k].keyword);
      }
      if (val & NOTPARAM) {
        warn("(%s,%d): engine data array \"%s\" is not simparam",
            c[k].filename, c[k].linenum, c[k].keyword);
      }
      if (val & REINIT) {
        warn("(%s,%d): reinitialized simparam \"%s\"",
            c[k].filename, c[k].linenum, c[k].keyword);
      }
      if (val & BADVAL) {
        warn("(%s,%d): illegal value \"%s\" for simparam \"%s\"",
            c[k].filename, c[k].linenum, c[k].value, c[k].keyword);
      }
      if (val & TRUNC) {
        warn("(%s,%d): string \"%s\" too long for simparam \"%s\"",
            c[k].filename, c[k].linenum, c[k].value, c[k].keyword);
      }
    }

  } /* end config keyword processing loop */

  /*** make sure each potential select has an alias ***/
  if (adt_getLengthList(&(sp->potential_select))
      > adt_getLengthList(&(sp->potential_alias))) {
    /* make life easier by defining variables */
    adt_List *select = &(sp->potential_select);
    adt_List *alias = &(sp->potential_alias);
    int select_len = adt_getLengthList(select);
    char *selstr = *((char **) adt_indexList(select, select_len - 1));
    char *alistr = simparam_potential_nextlabel(sp, selstr);

    /* check memory allocation for alistr */
    if (alistr == NULL) {
      return FAIL;
    }
    /* append default label for selected potential */
    if (adt_appendList(alias, &alistr)) {
      free(alistr);
      return error(MSG_NOMEM);
    }
    ASSERT(adt_getLengthList(alias) == select_len);
  }

  if (simparam_check(sp, sys, e)) return error(MSG, "simparam_check() failed");
  if (engparam_check(ep)) return error(MSG, "engparam_check() failed");

#ifdef DEBUG_SUPPORT
  if (simparam_debug(sp)) return error(MSG, "simparam_debug() failed");
  if (engparam_debug(ep)) return error(MSG, "engparam_debug() failed");
#endif

  return 0;
}


int sim_input_sysdata(Sim *s)
{
  System *sys = &(s->system);
  SimParam *p = &(s->simparam);

  /* read system data files and setup engine data arrays */
  if (system_input(sys, p)) {
    return error(MSG, "system_input() failed");
  }
  if (p->potential_name && system_setup_potential(sys, p)) {
    return error(MSG, "system_setup_potential() failed");
  }

  return 0;
}


int sim_setup_callbacks(Sim *s)
{
  Callback *c = &(s->callback);
  MD_Engine *e = &(s->engine);
  SimParam *p = &(s->simparam);
  System *sys = &(s->system);

  /* setup callbacks */
  if (callback_setup(c, e, p, sys)) {
    return error(MSG, "callback_setup() failed");
  }
  return 0;
}


int sim_run(Sim *sim)
{
  System *s = &(sim->system);
  SimParam *p = &(sim->simparam);
  Callback *c = &(sim->callback);
  MD_Engine *e = &(sim->engine);
  int32 stepnum, laststep, minstep;
  int32 next_restart, next_results;
  int32 next_posdcd, next_veldcd;
  int32 next_potential;
  int32 need_pos, need_vel, need_result;
  int32 need_potential;
  const int32 dcd_result = c->with_result;
  MD_Dvec *pos = NULL, *vel = NULL;
  void *result = NULL;

  /* for potentials */
  int32 is_u_bond = FALSE;
  int32 is_u_angle = FALSE;
  int32 is_u_dihed = FALSE;
  int32 is_u_impr = FALSE;
  int32 is_u_elec = FALSE;
  int32 is_u_vdw = FALSE;
  int32 is_u_bound = FALSE;
  int32 is_pot_elec = FALSE;

  double *u_bond = NULL;
  double *u_angle = NULL;
  double *u_dihed = NULL;
  double *u_impr = NULL;
  double *u_elec = NULL;
  double *u_vdw = NULL;
  double *u_bound = NULL;
  double *pot_elec = NULL;


  /* set first step number (if not 0) */
  if (p->firsttimestep != 0) {
    MD_firststep(e, p->firsttimestep);
  }

  ASSERT(p->numsteps >= 0);
  stepnum = p->firsttimestep;
  laststep = p->firsttimestep + p->numsteps;
  if (laststep < p->firsttimestep || laststep == MD_INT32_MAX) {
    return error("\"firsttimestep\" = %d is too large for %d \"numsteps\"",
        p->firsttimestep, p->numsteps);
  }

  if (c->is_restart || p->restartname == NULL) {
    /* either callback is supported or no restarts desired */
    next_restart = laststep + 1;
  }
  else {
    /* setup restart callback manually */
    if (callback_setup_restart(c)) {
      return error(MSG, "callback_setup_restart() failed");
    }
    next_restart = stepnum;
  }

  if (c->is_posdcd || p->dcdfile == NULL) {
    /* either callback is supported or no position DCD desired */
    next_posdcd = laststep + 1;
  }
  else {
    /* setup posdcd callback manually */
    if (callback_setup_posdcd(c)) {
      return error(MSG, "callback_setup_posdcd() failed");
    }
    next_posdcd = stepnum;
  }

  if (c->is_veldcd || p->veldcdfile == NULL) {
    /* either callback is supported or no velocity DCD desired */
    next_veldcd = laststep + 1;
  }
  else {
    /* setup veldcd callback manually */
    if (callback_setup_veldcd(c)) {
      return error(MSG, "callback_setup_veldcd() failed");
    }
    next_veldcd = stepnum;
  }

  if (c->is_results) {
    /* callback is supported */
    next_results = laststep + 1;
  }
  else {
    /* setup results callback manually */
    if (callback_setup_results(c)) {
      return error(MSG, "callback_setup_results() failed");
    }
    next_results = stepnum;
  }

  if (c->is_potential || p->potential_name == NULL) {
    /* either callback is supported or no potentials desired */
    next_potential = laststep + 1;
  }
  else {
    adt_List *pndex = &(s->potential_index);
    int pndex_len = adt_getLengthList(pndex);
    int i;

    /* setup potentials callback manually */
    if (callback_setup_potential(c)) {
      return error(MSG, "callback_setup_potential() failed");
    }
    next_potential = stepnum;

    /* establish which arrays are needed */
    for (i = 0;  i < pndex_len;  i++) {
      PotentialIndex n = *((PotentialIndex *) adt_indexList(pndex, i));
      switch (n.type) {
        case BOND:   is_u_bond = TRUE;   break;
        case ANGLE:  is_u_angle = TRUE;  break;
        case DIHED:  is_u_dihed = TRUE;  break;
        case IMPR:   is_u_impr = TRUE;   break;
        case ELEC:   is_u_elec = TRUE;   break;
        case VDW:    is_u_vdw = TRUE;    break;
        case BOUND:  is_u_bound = TRUE;  break;
        case EPOT:   is_pot_elec = TRUE; break;
      }
    }

    /* allocate array space if direct access is not supported */
    if ((s->u_bond_attr.access & MD_DIRECT) == 0) {
      if ((is_u_bond &&
            (u_bond = s->u_bond = calloc(s->bond_len, sizeof(double)))
            == NULL)
          || (is_u_angle &&
            (u_angle = s->u_angle = calloc(s->angle_len, sizeof(double)))
            == NULL)
          || (is_u_dihed &&
            (u_dihed = s->u_dihed = calloc(s->dihed_len, sizeof(double)))
            == NULL)
          || (is_u_impr &&
            (u_impr = s->u_impr = calloc(s->impr_len, sizeof(double)))
            == NULL)
          || (is_u_elec &&
            (u_elec = s->u_elec = calloc(s->atom_len, sizeof(double)))
            == NULL)
          || (is_u_vdw &&
            (u_vdw = s->u_vdw = calloc(s->atom_len, sizeof(double)))
            == NULL)
          || (is_u_bound &&
            (u_bound = s->u_bound = calloc(s->atom_len, sizeof(double)))
            == NULL)
          || (is_pot_elec &&
            (pot_elec = s->pot_elec = calloc(s->atom_len, sizeof(double)))
            == NULL)
          ) {
        return error(MSG_NOMEM);
      }
    }
  }

  do {
    /* find minimum step number by which to advance system */
    minstep = laststep;
    if (minstep > next_restart)    minstep = next_restart;
    if (minstep > next_posdcd)     minstep = next_posdcd;
    if (minstep > next_veldcd)     minstep = next_veldcd;
    if (minstep > next_results)    minstep = next_results;
    if (minstep > next_potential)  minstep = next_potential;

    /* run simulation for minimum number of steps */
    if (MD_run(e, minstep - stepnum, MD_CBFIRST)) {
      return error(MSG_MDAPI, "from MD_run()", MD_errmsg(e));
    }
    ASSERT(MD_stepnum(e) == minstep);

    /* we have advanced system to step number "minstep" */
    stepnum = minstep;

    /* find out what data is needed from engine */
    need_pos = next_restart == stepnum || next_posdcd == stepnum;
    need_vel = next_restart == stepnum || next_veldcd == stepnum;
    need_result = next_results == stepnum || (dcd_result
        && (next_posdcd == stepnum || next_veldcd == stepnum));
    need_potential = next_potential == stepnum;

    /*
     * get data from engine if needed
     */

    if (need_pos) {
      /* get position data from engine */
      if (s->pos_attr.access & MD_DIRECT) {
        /* use direct access */
        if ((pos = MD_direct(e, s->pos_id)) == NULL || MD_wait(e)) {
          return error(MSG_MDAPI, "from MD_direct()", MD_errmsg(e));
        }
      }
      else {
        /* use read access with alternate buffer space */
        ASSERT(s->posbuf != NULL);
        pos = s->posbuf;
        if (MD_read(e, s->pos_id, pos, s->atom_len) || MD_wait(e)) {
          return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
        }
      }
    }

    if (need_vel) {
      /* get velocity data from engine */
      if (s->vel_attr.access & MD_DIRECT) {
        /* use direct access */
        if ((vel = MD_direct(e, s->vel_id)) == NULL || MD_wait(e)) {
          return error(MSG_MDAPI, "from MD_direct()", MD_errmsg(e));
        }
      }
      else {
        /* use read access with alternate buffer space */
        ASSERT(s->velbuf != NULL);
        vel = s->velbuf;
        if (MD_read(e, s->vel_id, vel, s->atom_len) || MD_wait(e)) {
          return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
        }
      }
    }

    if (need_result) {
      /* get result data from engine */
      if (s->result_attr.access & MD_DIRECT) {
        /* use direct access */
        if ((result = MD_direct(e, s->result_id)) == NULL || MD_wait(e)) {
          return error(MSG_MDAPI, "from MD_direct()", MD_errmsg(e));
        }
      }
      else {
        /* use read access with alternate buffer space */
        ASSERT(s->resultbuf != NULL);
        result = s->resultbuf;
        if (MD_read(e, s->result_id, result, 1) || MD_wait(e)) {
          return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
        }
      }
    }

    if (need_potential) {
      /* get potential data from engine */
      if (s->u_bond_attr.access & MD_DIRECT) {
        /* use direct access (if u_bond has DIRECT, they all have it) */
        if ((is_u_bond &&
              ((u_bond = MD_direct(e, s->u_bond_id)) == NULL || MD_wait(e)))
            || (is_u_angle &&
              ((u_angle = MD_direct(e, s->u_angle_id)) == NULL || MD_wait(e)))
            || (is_u_dihed &&
              ((u_dihed = MD_direct(e, s->u_dihed_id)) == NULL || MD_wait(e)))
            || (is_u_impr &&
              ((u_impr = MD_direct(e, s->u_impr_id)) == NULL || MD_wait(e)))
            || (is_u_elec &&
              ((u_elec = MD_direct(e, s->u_elec_id)) == NULL || MD_wait(e)))
            || (is_u_vdw &&
              ((u_vdw = MD_direct(e, s->u_vdw_id)) == NULL || MD_wait(e)))
            || (is_u_bound &&
              ((u_bound = MD_direct(e, s->u_bound_id)) == NULL || MD_wait(e)))
            || (is_pot_elec &&
              ((pot_elec = MD_direct(e, s->pot_elec_id)) == NULL || MD_wait(e)))
            ) {
          return error(MSG_MDAPI, "from MD_direct()", MD_errmsg(e));
        }
      }
      else {
        /* use read access with alternate buffer space */
        if ((is_u_bond &&
              (MD_read(e, s->u_bond_id, u_bond, s->bond_len) || MD_wait(e)))
            || (is_u_angle &&
              (MD_read(e, s->u_angle_id, u_angle, s->angle_len) || MD_wait(e)))
            || (is_u_dihed &&
              (MD_read(e, s->u_dihed_id, u_dihed, s->dihed_len) || MD_wait(e)))
            || (is_u_impr &&
              (MD_read(e, s->u_impr_id, u_impr, s->impr_len) || MD_wait(e)))
            || (is_u_elec &&
              (MD_read(e, s->u_elec_id, u_elec, s->atom_len) || MD_wait(e)))
            || (is_u_vdw &&
              (MD_read(e, s->u_vdw_id, u_vdw, s->atom_len) || MD_wait(e)))
            || (is_u_bound &&
              (MD_read(e, s->u_bound_id, u_bound, s->atom_len) || MD_wait(e)))
            || (is_pot_elec &&
              (MD_read(e, s->pot_elec_id, pot_elec, s->atom_len) || MD_wait(e)))
            ) {
          return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
        }
      }
    }

    /*
     * manually invoke callback routines
     * (if engine data arrays do not support callback access)
     */

    if (next_restart == stepnum) {
      /* process restart callback manually */
      ASSERT(pos != NULL);
      c->restart[0].data = pos;
      c->restart[0].attrib = s->pos_attr;
      ASSERT(vel != NULL);
      c->restart[1].data = vel;
      c->restart[1].attrib = s->vel_attr;
      if (callback_restart(c, c->restart, 2, stepnum)) {
        return error(MSG, "callback_restart() failed");
      }
      /* advance to next stepnum for restart callback */
      next_restart += p->restartfreq;
    }

    if (next_posdcd == stepnum) {
      int32 len = 1;
      /* process posdcd callback manually */
      ASSERT(pos != NULL);
      c->cbposdcd[0].data = pos;
      c->cbposdcd[0].attrib = s->pos_attr;
      if (dcd_result) {
        len = 2;
        ASSERT(result != NULL);
        c->cbposdcd[1].data = result;
        c->cbposdcd[1].attrib = s->result_attr;
      }
      if (callback_posdcd(c, c->cbposdcd, len, stepnum)) {
        return error(MSG, "callback_posdcd() failed");
      }
      /* advance to next stepnum for posdcd callback */
      next_posdcd += p->dcdfreq;
    }

    if (next_veldcd == stepnum) {
      /* process veldcd callback manually */
      int32 len = 1;
      /* process veldcd callback manually */
      ASSERT(vel != NULL);
      c->cbveldcd[0].data = vel;
      c->cbveldcd[0].attrib = s->vel_attr;
      if (dcd_result) {
        len = 2;
        ASSERT(result != NULL);
        c->cbveldcd[1].data = result;
        c->cbveldcd[1].attrib = s->result_attr;
      }
      if (callback_veldcd(c, c->cbveldcd, len, stepnum)) {
        return error(MSG, "callback_veldcd() failed");
      }
      /* advance to next stepnum for posdcd callback */
      next_veldcd += p->veldcdfreq;
    }

    if (next_results == stepnum) {
      /* process results callback manually */
      ASSERT(result != NULL);
      c->results[0].data = result;
      c->results[0].attrib = s->result_attr;
      if (callback_results(c, c->results, 1, stepnum)) {
        return error(MSG, "callback_results() failed");
      }
      /* advance to next stepnum for results callback */
      next_results += p->resultsfreq;
    }

    if (next_potential == stepnum) {
      adt_List *pndex = &(s->potential_index);
      int pndex_len = adt_getLengthList(pndex);
      int i, j;

      for (i = 0, j = 0;  i < pndex_len;  i++, j++) {
        PotentialIndex n = *((PotentialIndex *) adt_indexList(pndex, i));
        switch (n.type) {
          case BOND:
            c->potential[j].data
              = ((char *) u_bond) + n.index * sizeof(double);
            c->potential[j].attrib = s->u_bond_attr;
            break;
          case ANGLE:
            c->potential[j].data
              = ((char *) u_angle) + n.index * sizeof(double);
            c->potential[j].attrib = s->u_angle_attr;
            break;
          case DIHED:
            c->potential[j].data
              = ((char *) u_dihed) + n.index * sizeof(double);
            c->potential[j].attrib = s->u_dihed_attr;
            break;
          case IMPR:
            c->potential[j].data
              = ((char *) u_impr) + n.index * sizeof(double);
            c->potential[j].attrib = s->u_impr_attr;
            break;
          case ELEC:
            c->potential[j].data
              = ((char *) u_elec) + n.index * sizeof(double);
            c->potential[j].attrib = s->u_elec_attr;
            break;
          case VDW:
            c->potential[j].data
              = ((char *) u_vdw) + n.index * sizeof(double);
            c->potential[j].attrib = s->u_vdw_attr;
            break;
          case BOUND:
            c->potential[j].data
              = ((char *) u_bound) + n.index * sizeof(double);
            c->potential[j].attrib = s->u_bound_attr;
            break;
          case EPOT:
            c->potential[j].data
              = ((char *) pot_elec) + n.index * sizeof(double);
            c->potential[j].attrib = s->pot_elec_attr;
            break;
          default:
            j--;    /* can't retrieve anything for invalid request */
                    /* "redo" this list entry */
        }
      }

      /* execute potentials callback */
      if (callback_potential(c, c->potential, c->potential_len, stepnum)) {
        return error(MSG, "callback_potential() failed");
      }

#ifdef DEBUG_WATCH
      /* to validate computation */
      if (is_u_bond) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->bond_len;  i++) {
          accum += u_bond[i].potential;
        }
        printf("bond potential:   %.6f\n", accum);
      }
      if (is_u_angle) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->angle_len;  i++) {
          accum += u_angle[i].potential;
        }
        printf("angle potential:  %.6f\n", accum);
      }
      if (is_u_dihed) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->dihed_len;  i++) {
          accum += u_dihed[i].potential;
        }
        printf("dihed potential:  %.6f\n", accum);
      }
      if (is_u_impr) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->impr_len;  i++) {
          accum += u_impr[i].potential;
        }
        printf("impr potential:   %.6f\n", accum);
      }
      if (is_u_elec) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->atom_len;  i++) {
          accum += u_elec[i];
        }
        printf("elec potential:   %.6f\n", accum);
      }
      if (is_u_vdw) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->atom_len;  i++) {
          accum += u_vdw[i];
        }
        printf("vdw potential:    %.6f\n", accum);
      }
      if (is_u_bound) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->atom_len;  i++) {
          accum += u_bound[i];
        }
        printf("bound potential:  %.6f\n", accum);
      }
      if (is_pot_elec) {
        double accum = 0.0;
        int32 i;
        for (i = 0;  i < s->atom_len;  i++) {
          /* scaling by 1/2 charge and summing should give elec pot energy */
          accum += pot_elec[i] * 0.5 * s->atom[i].q;
        }
        printf("elec potential:   %.6f\n", accum);
      }
#endif

      /* advance to next stepnum for potentials callback */
      next_potential += p->resultsfreq;
    }

  } while (stepnum < laststep);

  return 0;
}


int sim_output_sysdata(Sim *sim)
{
  MD_Engine *e = &(sim->engine);
  SimParam *p = &(sim->simparam);
  System *s = &(sim->system);
  int32 k;
  MD_Dvec *pos = NULL;
  MD_Dvec *vel = NULL;

  /* get trajectory data from engine */
  if (s->pos_attr.access & MD_DIRECT) {
    /* use direct access */
    if ((pos = (MD_Dvec *) MD_direct(e, s->pos_id)) == NULL || MD_wait(e)
        || (vel = (MD_Dvec *) MD_direct(e, s->vel_id)) == NULL || MD_wait(e)) {
      return error(MSG_MDAPI, "from MD_direct()", MD_errmsg(e));
    }
    /* convert coordinate pdb velocities in alternate buffer space */
    if ( ! p->is_binaryoutput) {
      ASSERT(s->velbuf != NULL);
      /* copy converted velocities from A/fs to A/ps into buffer */
      for (k = 0;  k < s->atom_len;  k++) {
        s->velbuf[k].x = MD_ANGSTROM_PS * vel[k].x;
        s->velbuf[k].y = MD_ANGSTROM_PS * vel[k].y;
        s->velbuf[k].z = MD_ANGSTROM_PS * vel[k].z;
      }
      /* change "vel" to point to this buffer */
      vel = s->velbuf;
    }
  }
  else {
    /* use read access with alternate buffer space */
    ASSERT(s->posbuf != NULL);
    ASSERT(s->velbuf != NULL);
    pos = s->posbuf;
    vel = s->velbuf;
    if (MD_read(e, s->pos_id, pos, s->atom_len) || MD_wait(e)
        || MD_read(e, s->vel_id, vel, s->atom_len) || MD_wait(e)) {
      return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
    }
    /* convert coordinate pdb velocities */
    if ( ! p->is_binaryoutput) {
      /* convert velocities from A/fs to A/ps within buffer */
      for (k = 0;  k < s->atom_len;  k++) {
        vel[k].x *= MD_ANGSTROM_PS;
        vel[k].y *= MD_ANGSTROM_PS;
        vel[k].z *= MD_ANGSTROM_PS;
      }
    }
  }

  /* do not overwrite existing files */
  if (rename(p->output_pos, p->output_posbak) && errno != ENOENT) {
    return warn("unable to rename \"%s\" to \"%s\"",
        p->output_pos, p->output_posbak);
  }
  if (rename(p->output_vel, p->output_velbak) && errno != ENOENT) {
    return warn("unable to rename \"%s\" to \"%s\"",
        p->output_vel, p->output_velbak);
  }

  if (p->is_binaryoutput) {
    /* output position and velocity to binary coordinates file */
    mdio_Bincoord bin;
    if (mdio_initializeBincoord(&(bin))) {
      return error(MSG, "mdio_initializeBincoord() failed");
    }
    mdio_setErrorHandlerFile((mdio_File *) &bin, mdio_errhandler);
    if (mdio_setBincoord(&bin, pos, s->atom_len)
        || mdio_writeBincoord(&bin, p->output_pos)
        || mdio_setBincoord(&bin, vel, s->atom_len)
        || mdio_writeBincoord(&bin, p->output_vel)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) &bin));
    }
    mdio_cleanupBincoord(&bin);
  }
  else {
    /* output position and velocity to PDB file */
    mdio_Pdbcoord pdb;
    ASSERT(s->pdbatompos != NULL);
    ASSERT(s->pdbatomvel != NULL);
    if (mdio_initializePdbcoord(&(pdb))) {
      return error(MSG, "mdio_initializePdbcoord() failed");
    }
    mdio_setErrorHandlerFile((mdio_File *) &pdb, mdio_errhandler);
    if (mdio_setPdbcoord(&pdb, pos, s->atom_len)
        || mdio_setAtomPdbcoord(&pdb, s->pdbatompos, s->atom_len)
        || mdio_writePdbcoord(&pdb, p->output_pos)
        || mdio_setPdbcoord(&pdb, vel, s->atom_len)
        /* note: reset Pdbatom array only if memory is not aliased */
        || (s->pdbatomvel != s->pdbatompos ?
          mdio_setAtomPdbcoord(&pdb, s->pdbatomvel, s->atom_len) : 0)
        || mdio_writePdbcoord(&pdb, p->output_vel)) {
      return error(MSG_MDIO, mdio_getErrorMessageFile((mdio_File *) &pdb));
    }
    mdio_cleanupPdbcoord(&pdb);
  }
  return 0;
}


void sim_done(Sim *s)
{
  char **str;
  int32 len, k;

  /* stop engine */
  MD_done(&(s->engine));

  /* cleanup callback object */
  callback_done(&(s->callback));

  /* cleaup system object */
  system_done(&(s->system));

  /* cleanup engparam object */
  engparam_done(&(s->engparam));

  /* cleanup simparam object */
  simparam_done(&(s->simparam));

  /* cleanup config file reader object */
  mdio_cleanupConfig(&(s->cfg));

  /* cleanup buffered keyword list */
  str = (char **) adt_getDataList(&(s->keyword));
  len = (int32) adt_getLengthList(&(s->keyword));
  for (k = 0;  k < len;  k++) {
    free(str[k]);
  }
  adt_cleanupList(&(s->keyword));
}
