/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * engparam.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* #define DEBUG_SUPPORT */
#include "mdsim/const.h"
#include "mdsim/engparam.h"
#include "mdsim/error.h"
#include "debug/debug.h"


/*
 * EngParam constructor
 * clear memory, initialize array and table structures
 */
int engparam_init(EngParam *p)
{
  /* clear memory */
  memset(p, 0, sizeof(EngParam));

  /* init array of engine parameter values */
  if (adt_initializeList(&(p->member), sizeof(EngParamMember), 0, NULL)) {
    return error(MSG_NOMEM);
  }

  /* init table of engine parameter names */
  if (adt_initializeTable(&(p->name), 0)) {
    return error(MSG_NOMEM);
  }
  return 0;
}


/*
 * check for existence of engine "param" data array
 * if it exists, allocate space for it and read default values
 * engine needs to have already been initialized
 * call engparam_setup() before calling engparam_set()
 */
int engparam_setup(EngParam *p, MD_Engine *e)
{
  const int32 rw = MD_READ | MD_WRITE;

  /* store engine interface pointer */
  p->eng = e;

  /* check engine for (optional) "param" datum */
  if ((p->param_id = MD_idnum(e, "param")) != MD_FAIL) {
    /* have it, check consistency */
    p->param_attr = MD_attrib(e, p->param_id);
    if (p->param_attr.type != MD_type(e, "Param")
        || (p->param_attr.access & rw) != rw
        || (p->param_attr.access & (MD_RESIZE | MD_ERESIZE)) != 0
        || p->param_attr.len != 1
        || p->param_attr.max != 1) {
      return error(MSG_ENGINE, "incorrect attributes for \"param\"");
    }
    /* allocate space for param data */
    if ((p->param_obj = malloc(MD_SIZEOF(p->param_attr.type))) == NULL) {
      return error(MSG_NOMEM);
    }
    /* read param to obtain default parameters */
    if (MD_read(e, p->param_id, p->param_obj, 1) || MD_wait(e)) {
      return error(MSG_MDAPI, "from MD_read()", MD_errmsg(e));
    }
  }
  else if (MD_reset(e)) {
    return error(MSG_MDAPI, "from MD_reset()", MD_errmsg(e));
  }
  else {
    /* reset "param_id" */
    p->param_id = 0;
  }
  return 0;
}


/*
 * EngParam destructor
 * cleanup memory allocations, table and array structures
 */
void engparam_done(EngParam *p)
{
  free(p->buf);
  free(p->param_obj);
  adt_cleanupTable(&(p->name));
  adt_cleanupList(&(p->member));
  memset(p, 0, sizeof(EngParam));
}


/*
 * write updated "param" data array back to engine
 * call engparam_check() after calls to engparam_set()
 */
int engparam_check(EngParam *p)
{
  ASSERT(p != NULL);

  /* write param back to engine */
  if (p->param_obj) {
    if (MD_write(p->eng, p->param_id, p->param_obj, 1)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(p->eng));
    }
  }
  return 0;
}


#ifdef DEBUG_SUPPORT
/*
 * print values of updated engine parameters
 */
int engparam_debug(EngParam *p)
{
  EngParamMember *pm = NULL;
  adt_List *pa = &(p->member);
  int32 len = pa->len;
  int32 k;

  printf("===== listing of engparam contents =====\n");
  for (k = 0;  k < len;  k++) {
    printf("engparam: ");
    pm = (EngParamMember *) adt_indexList(pa, k);
    ASSERT(pm != NULL);
    if (pm->isparam) {
      printf("(param member) ");
    }
    else {
      printf("(engine datum) ");
    }
    printf("%s = ", pm->name);
    if (pm->isstr) {
      printf("%s\n", pm->val.s);
    }
    else {
      switch (pm->type) {
        case MD_CHAR:
          printf("%c\n", pm->val.c);
          break;
        case MD_INT32:
          printf("%d\n", pm->val.n);
          break;
        case MD_FLOAT:
          printf("%g\n", (double) (pm->val.f));
          break;
        case MD_DOUBLE:
          printf("%g\n", pm->val.d);
          break;
        case MD_FVEC:
          printf("%g %g %g\n", (double) (pm->val.fv.x),
              (double) (pm->val.fv.y), (double) (pm->val.fv.z));
          break;
        case MD_DVEC:
          printf("%g %g %g\n", pm->val.dv.x, pm->val.dv.y, pm->val.dv.z);
          break;
        default:
          BUG("unexpected EngParamMember type");
      }
    }
  }

  return 0;
}
#endif


/*
 * identify keywd as (1) Param member, or (2) engine data array;
 * attempt to parse val string, convert to correct type, and store
 *
 * return 0 for success
 *
 * types possible are MD_CHAR, MD_INT32, MD_FLOAT, MD_DOUBLE, MD_FVEC,
 * MD_DVEC, string (array of MD_CHAR of length > 1 or of soon deprecated
 * types MD_NAME, MD_STRING, MD_MESSAGE) - all primary types
 *
 * only MD_CHAR array is permitted length > 1, all others must have fixed
 * length = 1
 *
 * provide extra flexibility for engines by permitting strings to be
 * initialized in multiple ways - could be any of:
 *  * param member of type MD_CHAR of (fixed) length > 1
 *  * param member of (soon deprecated) type MD_NAME, MD_STRING, MD_MESSAGE
 *  * engine data array of type MD_CHAR of fixed length > 1
 *  * engine data array of type MD_CHAR of variable length, resized to fit
 *  * engine data array of (soon deprecated) type MD_NAME, MD_STRING, or
 *    MD_MESSAGE
 *
 * extensive error checking:
 *  * out of memory - return error(MSG_NOMEM)
 *  * unexpected MDAPI error - return error(MSG_MDAPI, routine, errmsg)
 *    most likely due to incorrect MDAPI implementation
 *  * engine bug - return error(MSG_ENGINE, message)
 *    where message string specifies nature of engine bug,
 *    most likely engine does not meet minimal compatibility requirements
 *  * error values returned:
 *    + NOTFOUND - unrecognized keywd
 *    + NOTPARAM - recognized engine data array is not a simulation parameter
 *    + REINIT - attempt to reinitialize simulation parameter
 *    + BADVAL - syntax error when parsing val
 *    + TRUNC - val string would be truncated to fit into allowed space
 *    + (REINIT | BADVAL) - both of these occurred
 *    + (REINIT | TRUNC) - both of these occurred
 */
int engparam_set(EngParam *p, const char *keywd, const char *val)
{
  const int32 rw = MD_READ | MD_WRITE;
  int32 slen = 0;
  MD_Engine *e = NULL;
  adt_Table *nametable = NULL;
  adt_List *memberlist = NULL;
  int32 index;
  int32 listlen = -1;
  int32 idnum = MD_FAIL;
  MD_Attrib attr = { MD_FAIL, MD_FAIL, MD_FAIL, MD_FAIL };
  MD_Member mem = { 0, 0, NULL };
  void *mem_addr = NULL;
  void *v = NULL;
  EngParamMember epm;
  int retval = 0;
  int match = 0;
  int isbuffered = 0;
  char msg[72] = { 0 };
  char extra;

  /*
   * summary:
   *
   * 1. lookup keywd in name table to make sure we don't reinitialize
   * 2. see if keywd is a "param" member, check consistency
   * 3. otherwise keywd should be engine data array, check consistency
   * 4. parse val
   * 5. store "param" member into param_obj, otherwise MD_write() into
   *    engine data array
   * 6. append epm to EngParamMember array
   * 7. add keywd into hash table
   *
   * note: MD_write() for "param" is done later by engparam_check()
   */

  ASSERT(p != NULL);
  ASSERT(p->eng != NULL);
  ASSERT(keywd != NULL);
  ASSERT(val != NULL);
  e = p->eng;
  nametable = &(p->name);
  memberlist = &(p->member);
  listlen = (int32) adt_getLengthList(memberlist);
  slen = strlen(val);
  memset(&epm, 0, sizeof(EngParamMember));

  /* see if keywd is member of engine-defined type Param */
  if (p->param_obj && (mem_addr = MD_type_member(e, p->param_attr.type,
          p->param_obj, keywd, &mem)) != NULL) {
    /* treat len=1 as singleton char, otherwise treat as string */
    if (mem.type == MD_CHAR && mem.len > 1) {
      if (slen >= mem.len) {
        slen = mem.len - 1;
        retval |= TRUNC;
      }
      epm.isstr = TRUE;
    }
    /* otherwise must have len=1 */
    else if (mem.len != 1) {
      sprintf(msg, "\"Param\" member \"%.38s\" has wrong length", keywd);
      return error(MSG_ENGINE, msg);
    }
    /* check special predefined string types */
    else if (mem.type == MD_NAME
        || mem.type == MD_STRING
        || mem.type == MD_MESSAGE) {
      if (slen >= MD_SIZEOF(mem.type)) {
        slen = MD_SIZEOF(mem.type) - 1;
        retval |= TRUNC;
      }
      epm.isstr = TRUE;
    }
    epm.name = keywd;
    epm.type = mem.type;
    epm.len = mem.len;
    epm.isparam = TRUE;
  }
  else if (p->param_obj) {
    /* 
     * an error occurred from MD_type_member()
     * MD_ERR_NAME is recoverable - next try for engine data array
     */
    switch (MD_errnum(e)) {
      case MD_ERR_NAME:
        break;
      case MD_ERR_MEMALLOC:
        return error(MSG_NOMEM);
      case MD_ERR_NEWTYPE:
        return error(MSG_ENGINE, "\"Param\" type has duplicate member names");
      default:
        return error(MSG_MDAPI, "from MD_type_member()", MD_errmsg(e));
    }
    if (MD_reset(e)) {
      return error(MSG_MDAPI, "from MD_reset()", MD_errmsg(e));
    }
  }

  /* see if keywd is instead for engine data array */
  if (mem_addr == NULL) {
    /* get engine data array identification number */
    idnum = MD_idnum(e, keywd);
    if (idnum == MD_FAIL) {
      /*
       * an error occurred from MD_idnum()
       * MD_ERR_NAME is recoverable - return NOTFOUND
       */
      switch (MD_errnum(e)) {
        case MD_ERR_NAME:
          if (MD_reset(e)) {
            return error(MSG_MDAPI, "from MD_reset()", MD_errmsg(e));
          }
          return NOTFOUND;
        default:
          return error(MSG_MDAPI, "from MD_idnum()", MD_errmsg(e));
      }
    }
    /* get engine data array attributes */
    attr = MD_attrib(e, idnum);
    if (attr.type == MD_FAIL) {
      return error(MSG_MDAPI, "from MD_attrib()", MD_errmsg(e));
    }
    /* must have read & write access to be simulation parameter */
    else if ((attr.access & rw) != rw) {
      return NOTPARAM;
    }
    /* check for resize-able char array, treat as string */
    else if (attr.type == MD_CHAR && (attr.access & MD_SETLEN) == MD_SETLEN) {
      if ((attr.access & MD_SETMAX) != MD_SETMAX && slen >= attr.max) {
        if (attr.max == 0) return TRUNC;
        slen = attr.max - 1;
        retval |= TRUNC;
        /*
         * need buffer padded to string length plus nil terminator
         */
        if (p->buflen < attr.max) {
          v = realloc(p->buf, attr.max);
          if (v == NULL) return error(MSG_NOMEM);
          p->buf = (char *) v;
          p->buflen = attr.max;
        }
        isbuffered = TRUE;
      }
      if (MD_setlen(e, idnum, slen + 1)) {
        switch (MD_errnum(e)) {
          case MD_ERR_MEMALLOC:
            return error(MSG_NOMEM);
          default:
            return error(MSG_MDAPI, "from MD_setlen()", MD_errmsg(e));
        }
      }
      attr.len = slen + 1;
      epm.isstr = TRUE;
    }
    /* otherwise should not be able to resize */
    else if ((attr.access & (MD_RESIZE | MD_ERESIZE)) != 0) {
      return NOTPARAM;
    }
    /* and also should have len=1 and max=1 */
    else if (attr.len != 1 || attr.max != 1) {
      return NOTPARAM;
    }
    /* check for MD_NAME, MD_STRING, MD_MESSAGE */
    else if (attr.type == MD_NAME
        || attr.type == MD_STRING
        || attr.type == MD_MESSAGE) {
      if (slen >= MD_SIZEOF(attr.type)) {
        slen = MD_SIZEOF(attr.type) - 1;
        retval |= TRUNC;
      }
      /*
       * need buffer padded to at least as long as MD_SIZEOF()
       * because MD_write() has to move that many bytes
       */
      if (p->buflen < MD_SIZEOF(attr.type)) {
        v = realloc(p->buf, MD_SIZEOF(attr.type));
        if (v == NULL) return error(MSG_NOMEM);
        p->buf = (char *) v;
        p->buflen = MD_SIZEOF(attr.type);
      }
      isbuffered = TRUE;
      epm.isstr = TRUE;
    }
    epm.name = keywd;
    epm.type = attr.type;
    epm.len = attr.len;
  }

  ASSERT(epm.len == 1 || (epm.isstr && epm.type == MD_CHAR));
  /* parse from val string */
  switch (epm.type) {
    case MD_CHAR:
      if (!epm.isstr) {
        switch (sscanf(val, "%c%c", &(epm.val.c), &extra)) {
          case 1: break;
          case 0: return (retval | BADVAL);
          case 2: retval |= BADVAL; break;
          default: BUG("unexpected return value from sscanf()");
        }
        break;
      }
      /* fall through to set string pointer */
    case MD_NAME:
    case MD_STRING:
    case MD_MESSAGE:
      epm.val.s = val;
      break;
    case MD_INT32:
      switch (sscanf(val, "%d%c", &(epm.val.n), &extra)) {
        case 1: break;
        case 0: return (retval | BADVAL);
        case 2: retval |= BADVAL; break;
        default: BUG("unexpected return value from sscanf()");
      }
      break;
    case MD_FLOAT:
      switch (sscanf(val, "%f%c", &(epm.val.f), &extra)) {
        case 1: break;
        case 0: return (retval | BADVAL);
        case 2: retval |= BADVAL; break;
        default: BUG("unexpected return value from sscanf()");
      }
      break;
    case MD_DOUBLE:
      switch (sscanf(val, "%lf%c", &(epm.val.d), &extra)) {
        case 1: break;
        case 0: return (retval | BADVAL);
        case 2: retval |= BADVAL; break;
        default: BUG("unexpected return value from sscanf()");
      }
      break;
    case MD_FVEC:
      match = sscanf(val, "%f , %f , %f%c", &(epm.val.fv.x),
          &(epm.val.fv.y), &(epm.val.fv.z), &extra);
      if (match < 3) {
        match = sscanf(val, "%f %f %f%c", &(epm.val.fv.x),
            &(epm.val.fv.y), &(epm.val.fv.z), &extra);
      }
      switch (match) {
        case 3: break;
        case 0:
        case 1:
        case 2: return (retval | BADVAL);
        case 4: retval |= BADVAL; break;
        default: BUG("unexpected return value from sscanf()");
      }
      break;
    case MD_DVEC:
      match = sscanf(val, "%lf , %lf , %lf%c", &(epm.val.dv.x),
          &(epm.val.dv.y), &(epm.val.dv.z), &extra);
      if (match < 3) {
        match = sscanf(val, "%lf %lf %lf%c", &(epm.val.dv.x),
            &(epm.val.dv.y), &(epm.val.dv.z), &extra);
      }
      switch (match) {
        case 3: break;
        case 0:
        case 1:
        case 2: return (retval | BADVAL);
        case 4: retval |= BADVAL; break;
        default: BUG("unexpected return value from sscanf()");
      }
      break;
    default:
      /* unknown type that we can't deal with */
      if (!epm.isparam) return NOTPARAM;
      sprintf(msg, "\"Param\" member \"%.38s\" has unknown type", keywd);
      return error(MSG_ENGINE, msg);
  }

  /* store parsed value */
  if (mem_addr) {
    /* store into param_obj space, delay MD_write() until later */
    if (epm.isstr) {
      memcpy(mem_addr, epm.val.s, slen);
      ((char *) mem_addr)[slen] = '\0';  /* must nil-terminate */
    }
    else {
      memcpy(mem_addr, &(epm.val), MD_SIZEOF(epm.type));
    }
  }
  else {
    /*
     * MD_write() into engine data array, expected to succeed
     * because space requirements and access already checked
     */
    if (epm.isstr) {
      const char *s = epm.val.s;
      if (isbuffered) {
        ASSERT(p->buflen > slen);
        strncpy(p->buf, s, slen);
        p->buf[slen] = '\0';  /* must nil-terminate */
        s = p->buf;
      }
      if (MD_write(e, idnum, s, epm.len)) {
        return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
      }
    }
    else if (MD_write(e, idnum, &(epm.val), epm.len)) {
      return error(MSG_MDAPI, "from MD_write()", MD_errmsg(e));
    }
  }

  /* make sure keywd isn't already in table */
  if ((index = adt_lookupTable(nametable, keywd)) == ADT_ERROR) {
    /* append epm to EngParamMember array */
    if (adt_appendList(memberlist, &epm)) {
      return error(MSG_NOMEM);
    }
    /* hash keywd to name table with EngParamMember array index */
    if (adt_insertTable(nametable, keywd, listlen) != listlen) {
      return error(MSG_NOMEM);
    }
  }
  else {
    /* overwrite EngParamMember list entry */
    if (adt_updateList(memberlist, index, &epm)) {
      BUG("unexpected error from adt_indexList()");
    }
    /* update return value */
    retval |= REINIT;
  }

  return retval;
}
