/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    pdbcoord.h
 * @brief   PDB coordinate reader and writer.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c mdio_Pdbcoord class reads coordinates from PDB files, where
 * atomic coordinates are parsed from @c ATOM and @c HETATM records.  
 * Note that other records are completely ignored.
 * This class can also be used to write a PDB coordinate file, saved as
 * @c ATOM records, with preliminary remarks indicating the filename,
 * the creator, the date, and the time.
 *
 * The complete PDB file specification is available at
 * http://www.rcsb.org/pdb/file_formats/pdb/pdbguide2.2/guide2.2_frame.html .
 *
 * The @c mdio_Pdbcoord class is derived from @c mdio_File so that error
 * handling is performed by typecasting (@c mdio_Pdbcoord @c *) to
 * (@c mdio_File @c *).
 */

#ifndef MDIO_PDBCOORD_H
#define MDIO_PDBCOORD_H

#include "adt/list.h"
#include "mdio/file.h"
#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief PDB coordinate reader and writer class.
   *
   * Members should be treated as private.
   */
  typedef struct mdio_Pdbcoord_t {
    mdio_File file;
    adt_List dvec;
    adt_List atom;
  } mdio_Pdbcoord;


  /**@brief @c ATOM and @c HETATM record information.
   *
   * Auxiliary structure for information from PDB @c ATOM or @c HETATM
   * records.  The field names and notes below are from the PDB format
   * description guide:
   * http://www.rcsb.org/pdb/docs/format/pdbguide2.2/guide2.2_frame.html .
   * The extra space in the @c char arrays are for nil-termination and
   * 32-bit padding.  The column numbering given is FORTRAN-based,
   * ranging from 1 to 80.
   */
  typedef struct mdio_Pdbatom_t {
    /*
     *                        definition                         columns
     */
    char record[8];  /**< Record name: "ATOM  " or "HETATM"   (col:  1 -  6) */
    char serial[8];  /**< Atom serial number.                 (col:  7 - 11) */
    char name[8];    /**< Atom name.                          (col: 13 - 16) */
    char altLoc[4];  /**< Alternate location identifier.      (col:      17) */
    char resName[4]; /**< Residue name.                       (col: 18 - 20) */
    char chainID[4]; /**< Chain identifier.                   (col:      22) */
    char resSeq[8];  /**< Residue sequence number.            (col: 23 - 26) */
    char iCode[4];   /**< Code for insertion of residues.     (col:      27) */
    float occupancy; /**< Occupancy.                          (col: 55 - 60) */
    float tempFactor;/**< Temperature factor.                 (col: 61 - 66) */
    char segID[8];   /**< Segment identifier, left-justified. (col: 73 - 76) */
    char element[4]; /**< Element symbol, right-justified.    (col: 77 - 78) */
    char charge[4];  /**< Charge on the atom.                 (col: 79 - 80) */
  } mdio_Pdbatom;


/* constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated PDB coordinate file reader and writer
   * object.
   *
   * @return Pointer to new object or @c NULL on failure.
   */
  mdio_Pdbcoord *mdio_createPdbcoord(void);


  /**@brief Destructor.
   *
   * Frees dynamically allocated PDB coordinate file reader and writer
   * object after freeing allocated arrays of @c MD_Dvec and @c mdio_Pdbatom.
   *
   * Note that @c MD_Dvec and @c mdio_Pdbatom arrays obtained from reading
   * a file do not persist beyond the destruction of the @c mdio_Pdbcoord
   * object.
   */
  void mdio_destroyPdbcoord(mdio_Pdbcoord *);


/* reading file and obtaining data */

  /**@brief Read PDB coordinate file.
   *
   * @param[in] name  A nil-terminated string naming the PDB coordinate file.
   * @param[in] n_expect  The number of coordinates expected in the file.
   *
   * If the number of coordinates is unknown, then @c n_expect should be
   * set to 0.  Otherwise, this value is used to validate the expected
   * number of coordinates.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.  It is considered an
   * error for @c n_expect to not match the number of coordinates
   * contained in the file.
   */
  int mdio_readPdbcoord(mdio_Pdbcoord *, const char *name, int n_expect);


  /**@brief Obtain array of coordinates.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * If this array resulted from @c mdio_readPdbcoord(), then it will
   * persist only until destruction of the @c mdio_Pdbcoord object.
   *
   * @return The @c MD_Dvec coordinate array as a pointer.
   */
  MD_Dvec *mdio_getPdbcoord(mdio_Pdbcoord *, int *nelems);


  /**@brief Obtain array of @c ATOM record information.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * If this array resulted from @c mdio_readPdbcoord(), then it will
   * persist only until destruction of the @c mdio_Pdbcoord object.
   *
   * @return The @c mdio_Pdbatom record information array as a pointer.
   */
  mdio_Pdbatom *mdio_getAtomPdbcoord(mdio_Pdbcoord *, int *nelems);


/* providing data and writing files */

  /**@brief Set an array of coordinates.
   *
   * @param[in] dvec  Array of coordinates.
   * @param[in] nelems  Number of elements in array.
   *
   * The @c dvec array will still belong to the user, which means that
   * no attempt will be made to free it, and the burden of any dynamic
   * memory management still belongs to the user.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_setPdbcoord(mdio_Pdbcoord *, MD_Dvec *dvec, int nelems);


  /**@brief Set an array of @c ATOM record information.
   *
   * @param[in] atom  Array of @c ATOM record information.
   * @param[in] nelems  Number of elements in array.
   *
   * The @c atom array will still belong to the user, which means that
   * no attempt will be made to free it, and the burden of any dynamic
   * memory management still belongs to the user.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_setAtomPdbcoord(mdio_Pdbcoord *, mdio_Pdbatom *atom, int nelems);


  /**@brief Write PDB coordinate file.
   *
   * @param[in] name  A nil-terminated string naming the PDB coordinate file.
   *
   * The coordinates and corresponding @c ATOM record information will
   * be whatever is known by the @c mdio_Pdbcoord object, either from
   * previous calls to @c mdio_setPdbcoord() and @c mdio_setAtomPdbcoord()
   * or a previous read using @c mdio_readPdbcoord().
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_writePdbcoord(mdio_Pdbcoord *, const char *name);


/* constructor and destructor for pre-allocated mdio_Pdbcoord object */

  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated @c mdio_Pdbcoord object.
   * See @c mdio_createPdbcoord() for a description of expected arguments.
   */
  int mdio_initializePdbcoord(mdio_Pdbcoord *);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated @c mdio_Pdbcoord object
   * (i.e. one constructed using @c mdio_initializePdbcoord() ).
   */
  void mdio_cleanupPdbcoord(mdio_Pdbcoord *);

#ifdef __cplusplus
}
#endif

#endif /* MDIO_PDBCOORD_H */
