/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    param.h
 * @brief   Force field parameter file reader.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c mdio_Param class reads one or more X-Plor force field parameter
 * files, storing the results into arrays of MDAPI force field parameter
 * data types defined in @c mdtypes.h.  These arrays are continued if
 * subsequent parameter files are read.
 *
 * The @c mdio_Param class is derived from @c mdio_File so that error
 * handling is performed by simply typecasting (@c mdio_Param @c *)
 * to a (@c mdio_File @c *).
 */

#ifndef MDIO_PARAM_H
#define MDIO_PARAM_H

#include "adt/list.h"
#include "adt/table.h"
#include "mdio/file.h"
#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /*
   * Set the "filetype" field to identify the type of parameter files.
   * Must permit param files of only a single type, because the
   * interpretation of PSF depends on the type of param files read.
   *
   * Specifically, X-Plor dihedral multiplicities are determined both by
   * how many terms are read from param file and the number of times that
   * a dihedral interaction is repeated in the PSF.  For example, a set of
   * dihedral parameters of multiplicity 3 may be read in from the param
   * file, but if a dihedral interaction using that set of parameters is
   * listed only once in the PSF, then that dihedral is treated as having
   * multiplicity 1, using just the first term read from the param file.
   *
   * For the first parameter file reading, X-Plor format is assumed.
   * If the first non-white char is '*', then MDIO_PARAM_CHARMM is
   * returned and an attempt is made to read as CHARMM parameter file.
   */
  enum { MDIO_PARAM_XPLOR = 1, MDIO_PARAM_CHARMM = 2 };


  /**@brief Force field parameter file reader class.
   *
   * Members should be treated as private.
   */
  typedef struct mdio_Param_t {
    mdio_File file;
    adt_List atomprm;
    adt_List bondprm;
    adt_List angleprm;
    adt_List dihedprm;
    adt_List imprprm;
    adt_List nbfixprm;
    adt_Table tokentable;
    char *token;
    char *buffer;
    int bufferlen;
    int tokenflags;
    char tokenfmt[32];
    char *startbuf;
    int filetype;
  } mdio_Param;


/* constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated force field parameter file reader object.
   *
   * @return Pointer to new object or @c NULL on failure.
   */
  mdio_Param *mdio_createParam(void);


  /**@brief Destructor.
   *
   * Frees dynamically allocated force field parameter file reader object
   * after freeing the arrays of collected force field parameter data types.
   *
   * Note that the force field parameter arrays obtained do not persist
   * beyond destruction of the @c mdio_Param object.
   */
  void mdio_destroyParam(mdio_Param *);


/* reading parameter files and obtaining parsed data */

  /**@brief Read force field parameter file.
   *
   * @param[in] name  A nil-terminated string naming the force field
   *   parameter file.
   *
   * Opens the file, reads and parses each line of the file, and closes
   * the file.  Incrementally builds arrays storing the force field
   * parameter data types.  These arrays are continued if subsequent files
   * are read.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_readParam(mdio_Param *, const char *name);


  /**@brief Obtain array of atom parameters.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The atom parameter array as a pointer.
   */
  MD_AtomPrm *mdio_getAtomParam(mdio_Param *, int *nelems);


  /**@brief Obtain array of bond parameters.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The bond parameter array as a pointer.
   */
  MD_BondPrm *mdio_getBondParam(mdio_Param *, int *nelems);


  /**@brief Obtain array of angle parameters.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The angle parameter array as a pointer.
   */
  MD_AnglePrm *mdio_getAngleParam(mdio_Param *, int *nelems);


  /**@brief Obtain array of dihedral parameters.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The dihedral parameter array as a pointer.
   */
  MD_TorsPrm *mdio_getDihedParam(mdio_Param *, int *nelems);


  /**@brief Obtain array of improper parameters.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The improper parameter array as a pointer.
   */
  MD_TorsPrm *mdio_getImprParam(mdio_Param *, int *nelems);


  /**@brief Obtain array of NBfix parameters.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * @return The NBfix parameter array as a pointer.
   *
   * Note that the @c prm[] fields that index the @c MD_AtomPrm array
   * are left undetermined until the call is made to @c mdio_indexParamTopo().
   */
  MD_NbfixPrm *mdio_getNbfixParam(mdio_Param *, int *nelems);


/* constructor and destructor for preallocated mdio_Param object */

  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated @c mdio_Param object.
   * See @c mdio_createParam() for a description of expected arguments.
   */
  int mdio_initializeParam(mdio_Param *);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated @c mdio_Param object
   * (i.e. one constructed using @c mdio_initializeParam() ).
   */
  void mdio_cleanupParam(mdio_Param *);

#ifdef __cplusplus
}
#endif

#endif /* MDIO_PARAM_H */
