/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 *
 * param.c
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "mdio/param.h"
#include "debug/debug.h"


#define INIT_BUFLEN  (100)


/*
 * prototypes for static functions
 */
static int xplor_init(mdio_Param *prm);
static void xplor_done(mdio_Param *prm);
static int xplor_lookup(mdio_Param *prm, const char *keyword);
static void xplor_reset(mdio_Param *prm);
static int xplor_error(mdio_Param *prm);
static const char *xplor_keyword(int tokid);
static int xplor_readline(mdio_Param *prm);
static const char *xplor_token(mdio_Param *prm);
static int xplor_parse_name(mdio_Param *prm, MD_Name name);
static int xplor_parse_int(mdio_Param *prm, int *n);
static int xplor_parse_real(mdio_Param *prm, double *x);
static int xplor_parse_bond(mdio_Param *prm);
static int xplor_parse_angle(mdio_Param *prm);
static int xplor_parse_dihedral(mdio_Param *prm);
static int xplor_parse_improper(mdio_Param *prm);
static int xplor_parse_nbfix(mdio_Param *prm);
static int xplor_parse_nonbonded(mdio_Param *prm);
static int xplor_parse_hbonded(mdio_Param *prm);
static int xplor_read(mdio_Param *prm);

static int charmm_read(mdio_Param *prm);
static int charmm_parse_bond(mdio_Param *p);
static int charmm_parse_angle(mdio_Param *p);
static int charmm_parse_dihedral(mdio_Param *p);
static int charmm_parse_improper(mdio_Param *p);
static int charmm_parse_nonbonded(mdio_Param *p);
static int charmm_parse_nbfix(mdio_Param *p);


mdio_Param *mdio_createParam(void)
{
  mdio_Param *p;
  p = (mdio_Param *) malloc(sizeof(mdio_Param));
  if (p == NULL) {
    ERRMSG("out of memory");
    return NULL;
  }
  if (mdio_initializeParam(p)) {
    free(p);
    return NULL;
  }
  return p;
}


int mdio_initializeParam(mdio_Param *p)
{
  ASSERT(p != NULL);
  memset(p, 0, sizeof(mdio_Param));
  if (mdio_initializeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->atomprm), sizeof(MD_AtomPrm), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"atomprm\" list");
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->bondprm), sizeof(MD_BondPrm), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"bondprm\" list");
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->angleprm), sizeof(MD_AnglePrm), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"angleprm\" list");
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->dihedprm), sizeof(MD_TorsPrm), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"dihedprm\" list");
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->imprprm), sizeof(MD_TorsPrm), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"torsprm\" list");
    return MDIO_ERROR;
  }
  if (adt_initializeList(&(p->nbfixprm), sizeof(MD_NbfixPrm), 0, NULL)) {
    mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_NOMEM,
        "cannot initialize \"nbfixprm\" list");
    return MDIO_ERROR;
  }
  if (xplor_init(p)) {
    ERRMSG("cannot initialize xplor params parser");
    xplor_done(p);
    return MDIO_ERROR;
  }
  return 0;
}


void mdio_destroyParam(mdio_Param *p)
{
  ASSERT(p != NULL);
  mdio_cleanupParam(p);
  free(p);
}


void mdio_cleanupParam(mdio_Param *p)
{
  ASSERT(p != NULL);
  xplor_done(p);
  adt_cleanupList(&(p->atomprm));
  adt_cleanupList(&(p->bondprm));
  adt_cleanupList(&(p->angleprm));
  adt_cleanupList(&(p->dihedprm));
  adt_cleanupList(&(p->imprprm));
  adt_cleanupList(&(p->nbfixprm));
  mdio_cleanupFile(&(p->file));
}


int mdio_readParam(mdio_Param *p, const char *name)
{
  int retval;

  ASSERT(p != NULL);
  ASSERT(name != NULL);

  /* open file */
  if (mdio_openFile(&(p->file), name, MDIO_FILE_TEXT | MDIO_FILE_READ)) {
    return MDIO_ERROR;
  }

  /* reset parser */
  xplor_reset(p);

  /* read file - no forgiveness for syntax errors! */
  if (p->filetype == MDIO_PARAM_XPLOR
      && (retval = xplor_read(p)) != 0) {
    if (retval == MDIO_PARAM_CHARMM) {
      /* set an additional error message */
      mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_READ,
          "expecting to read another X-Plor parameter file");
    }
    mdio_closeFile(&(p->file));
    return MDIO_ERROR;
  }
  else if (p->filetype == MDIO_PARAM_CHARMM
      && charmm_read(p) != 0) {
    mdio_closeFile(&(p->file));
    return MDIO_ERROR;
  }
  else { /* don't yet know what type of parameter file */
    /* first attempt reading as X-Plor parameter file */
    printf("# attempting to read parameter file as X-Plor format\n");
    retval = xplor_read(p);
    if (retval == MDIO_PARAM_CHARMM) {
      printf("# attempting to read parameter file as CHARMM format\n");
      /* attempt reading as CHARMM parameter file */
      retval = charmm_read(p);
    }
    if (retval != 0) {
      printf("# failed to read parameter file\n");
      mdio_closeFile(&(p->file));
      return MDIO_ERROR;
    }
  }

  /* close file */
  if (mdio_closeFile(&(p->file))) {
    return MDIO_ERROR;
  }
  return 0;
}


MD_AtomPrm *mdio_getAtomParam(mdio_Param *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthList(&(p->atomprm));
  return (MD_AtomPrm *) adt_getDataList(&(p->atomprm));
}


MD_BondPrm *mdio_getBondParam(mdio_Param *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthList(&(p->bondprm));
  return (MD_BondPrm *) adt_getDataList(&(p->bondprm));
}


MD_AnglePrm *mdio_getAngleParam(mdio_Param *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthList(&(p->angleprm));
  return (MD_AnglePrm *) adt_getDataList(&(p->angleprm));
}


MD_TorsPrm *mdio_getDihedParam(mdio_Param *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthList(&(p->dihedprm));
  return (MD_TorsPrm *) adt_getDataList(&(p->dihedprm));
}


MD_TorsPrm *mdio_getImprParam(mdio_Param *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthList(&(p->imprprm));
  return (MD_TorsPrm *) adt_getDataList(&(p->imprprm));
}


MD_NbfixPrm *mdio_getNbfixParam(mdio_Param *p, int *nelems)
{
  ASSERT(p != NULL);
  ASSERT(nelems != NULL);
  *nelems = adt_getLengthList(&(p->nbfixprm));
  return (MD_NbfixPrm *) adt_getDataList(&(p->nbfixprm));
}


/*****************************************************************************
 *
 * Tokenize and parse X-Plor parameter files
 *
 *****************************************************************************/

#define  TOKENFMT    "[a-zA-Z0-9.*%#+-]"

/*
 * token bit flags used for parsing
 * state stored in prm->tokenflags field
 * ERROR flag is set if there is some syntax or formatting error
 * RECOVER flag is set after ERROR to attempt continued reading
 */

enum {
  /* tokens */
  ERROR     = 0x00000001,  /* internal state codes */
  RECOVER   = 0x00000002,
  ENDOFILE  = 0x00000004,
  COMMENT   = 0x00000008,

  EQUALS    = 0x00000010,  /* the "=" sign */

  BOND      = 0x00000020,  /* primary keywords */
  ANGLE     = 0x00000040,
  DIHEDRAL  = 0x00000080,
  IMPROPER  = 0x00000100,
  NBFIX     = 0x00000200,
  NONBONDED = 0x00000400,
  HBONDED   = 0x00000800,
  HBONDS    = 0x00001000,
  NBONDS    = 0x00002000,
  SET       = 0x00004000,
  LEARN     = 0x00008000,
  REDUCE    = 0x00010000,
  RESET     = 0x00020000,
  VERBOSE   = 0x00040000,

  UB        = 0x00080000,  /* option for "ANGLE" */

  MULTIPLE  = 0x00100000,  /* option for "DIHEDRAL" and "IMPROPER" */

  ALL       = 0x00200000,  /* options for "RESET" */
  TYPE      = 0x00400000,
  ATOM      = 0x00800000,

  END       = 0x01000000,  /* use to close blocks: */
                           /* "HBONDS" "NBONDS" "SET" "LEARN" "REDUCE" */

  PUSHBACK  = 0x40000000,  /* internal state code */
                           /* to push token back on stream */

  UNKNOWN   = (int)0x80000000,  /* the "unknown" token id */

  /* bitfield masks */
  PRIMARY_TOKEN = BOND | ANGLE | DIHEDRAL | IMPROPER | NBFIX | NONBONDED
    | HBONDED | HBONDS | NBONDS | SET | LEARN | REDUCE | RESET | VERBOSE,
  RESET_OPTIONS = ALL | TYPE | ATOM,
  OTHER_OPTIONS = UB | MULTIPLE
};


/*
 * functions
 */

/* constructor */
int xplor_init(mdio_Param *prm)
{
  adt_Table *t;

  ASSERT(prm != NULL);
  t = &(prm->tokentable);

  /* buffer and token arrays starts with INIT_BUFLEN elements */
  prm->buffer = (char *) malloc(INIT_BUFLEN);
  prm->token = (char *) malloc(INIT_BUFLEN);
  if (prm->buffer == NULL || prm->token == NULL) {
    ERRMSG("out of memory");
    mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  prm->bufferlen = INIT_BUFLEN;

  /* init token for start of file */
  prm->token[0] = '\0';

  /* create token format specifier */
  sprintf(prm->tokenfmt, "%%%d%s", prm->bufferlen - 1, TOKENFMT);

  /*
   * init table of keyword tokens 
   * for fast searching need to associate strings with token codes
   */
  if (adt_initializeTable(t, 0)) {
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_NOMEM,
        "cannot initialize token table");
    return MDIO_ERROR;
  }
  if (adt_insertTable(t, "", ENDOFILE) != ENDOFILE
      || adt_insertTable(t, "=", EQUALS) != EQUALS
      || adt_insertTable(t, "BOND", BOND) != BOND
      || adt_insertTable(t, "ANGLE", ANGLE) != ANGLE
      || adt_insertTable(t, "ANGL", ANGLE) != ANGLE
      || adt_insertTable(t, "UB", UB) != UB
      || adt_insertTable(t, "DIHEDRAL", DIHEDRAL) != DIHEDRAL
      || adt_insertTable(t, "DIHE", DIHEDRAL) != DIHEDRAL
      || adt_insertTable(t, "IMPROPER", IMPROPER) != IMPROPER
      || adt_insertTable(t, "IMPR", IMPROPER) != IMPROPER
      || adt_insertTable(t, "MULTIPLE", MULTIPLE) != MULTIPLE
      || adt_insertTable(t, "MULT", MULTIPLE) != MULTIPLE
      || adt_insertTable(t, "NBFIX", NBFIX) != NBFIX
      || adt_insertTable(t, "NBFI", NBFIX) != NBFIX
      || adt_insertTable(t, "NONBONDED", NONBONDED) != NONBONDED
      || adt_insertTable(t, "NONB", NONBONDED) != NONBONDED
      || adt_insertTable(t, "HBONDED", HBONDED) != HBONDED
      || adt_insertTable(t, "HBON", HBONDED) != HBONDED
      || adt_insertTable(t, "HBONDS", HBONDS) != HBONDS
      || adt_insertTable(t, "NBONDS", NBONDS) != NBONDS
      || adt_insertTable(t, "NBON", NBONDS) != NBONDS
      || adt_insertTable(t, "SET", SET) != SET
      || adt_insertTable(t, "LEARN", LEARN) != LEARN
      || adt_insertTable(t, "LEAR", LEARN) != LEARN
      || adt_insertTable(t, "REDUCE", REDUCE) != REDUCE
      || adt_insertTable(t, "REDU", REDUCE) != REDUCE
      || adt_insertTable(t, "END", END) != END
      || adt_insertTable(t, "RESET", RESET) != RESET
      || adt_insertTable(t, "RESE", RESET) != RESET
      || adt_insertTable(t, "ALL", ALL) != ALL
      || adt_insertTable(t, "TYPE", TYPE) != TYPE
      || adt_insertTable(t, "ATOM", ATOM) != ATOM
      || adt_insertTable(t, "VERBOSE", VERBOSE) != VERBOSE
      || adt_insertTable(t, "VERB", VERBOSE) != VERBOSE) {
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_NOMEM,
       "cannot insert into token table");
    return MDIO_ERROR;
  }
  return 0;
}


/* destructor */
void xplor_done(mdio_Param *prm)
{
  /* cleanup parser data structures */
  free(prm->token);
  free(prm->buffer);
  adt_cleanupTable(&(prm->tokentable));
}


/* convert keyword to upper case and lookup in search table */
int xplor_lookup(mdio_Param *prm, const char *keyword)
{
  /* keywords are all shorter than length 16 */
  char s[16];
  char *c;
  int id;

  strncpy(s, keyword, sizeof(s)-1);
  for (c = s;  *c != '\0';  c++) *c = toupper(*c);
  id = adt_lookupTable(&(prm->tokentable), s);
  return (id != ADT_ERROR ? id : UNKNOWN);
}


/* reset parser */
void xplor_reset(mdio_Param *prm)
{
  prm->tokenflags = 0;
  prm->token[0] = '\0';
}


/* set error flag */
int xplor_error(mdio_Param *prm)
{
  prm->tokenflags |= ERROR;
  return MDIO_ERROR;
}


/*
 * map token ID number back to a string representation
 * (use for error diagnostics)
 */
const char *xplor_keyword(int tokid)
{
  if (tokid == BOND) return "BOND";
  else if (tokid == ANGLE) return "ANGLE";
  else if (tokid == DIHEDRAL) return "DIHEDRAL";
  else if (tokid == IMPROPER) return "IMPROPER";
  else if (tokid == NBFIX) return "NBFIX";
  else if (tokid == NONBONDED) return "NONBONDED";
  else if (tokid == HBONDED) return "HBONDED";
  else if (tokid == HBONDS) return "HBONDS";
  else if (tokid == NBONDS) return "NBONDS";
  else if (tokid == SET) return "SET";
  else if (tokid == LEARN) return "LEARN";
  else if (tokid == REDUCE) return "REDUCE";
  else if (tokid == VERBOSE) return "VERBOSE";
  /* everything else is considered "unknown" */
  return "UNKNOWN";
}


/*
 * read in an entire line (using MDIO_getline())
 * clear out the comments, return line length
 */
int xplor_readline(mdio_Param *prm)
{
  int len, tokenlen;

  ASSERT(prm != NULL);
  tokenlen = prm->bufferlen;

  /* read line */
  len = mdio_readLineFile(&(prm->file), &(prm->buffer), &(prm->bufferlen));

  /* preprocess line */
  if (len > 0) {
    char *p;

    /* make sure token array is same length as buffer */
    if (tokenlen < prm->bufferlen) {
      char *tmp = (char *) realloc(prm->token, prm->bufferlen);
      if (tmp == NULL) {
        ERRMSG("out of memory");
        mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
        return MDIO_ERROR;
      }
      prm->token = tmp;
      /* recreate token format specifier */
      sprintf(prm->tokenfmt, "%%%d%s", prm->bufferlen - 1, TOKENFMT);
    }

    /* remove comments from line */
    p = prm->buffer;
    while (*p != '\0') {
      if (prm->tokenflags & COMMENT) {
        if (*p == '}') prm->tokenflags &= ~COMMENT;
        *p = ' ';
      }
      else if (*p == '{') {
        *p = ' ';
        prm->tokenflags |= COMMENT;
      }
      else if (*p == '!') {
        *p = '\0';
        continue;
      }
      p++;
    }
  }
  if (len == 0 && (prm->tokenflags & COMMENT)) {
    /* did not find closing comment marker */
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_UNXEOF,
        "multi-line comment was not terminated");
    return MDIO_ERROR;
  }
  return len;
}


/*
 * tokenize the char buffer from xplor_readline
 * fill up the token buffer and return the next token
 * callers can treat this function as returning a token stream
 * up to one token can be pushed back onto stream by setting PUSHBACK flag
 * return an empty string in the token buffer for end-of-file
 * return NULL if an error has occurred
 */
const char *xplor_token(mdio_Param *prm)
{
  int len;
  char *token;

  ASSERT(prm != NULL);

  token = prm->token;
  /* check for previous token pushed back to us */
  if (prm->tokenflags & PUSHBACK) {
    prm->tokenflags &= ~PUSHBACK;
    return token;
  }
  /* check for start of file */
  else if (token[0] == '\0') {
    /* first skip over any leading "remark" lines */
    do {
      len = xplor_readline(prm);
      /* need to reset token after readline */
      token = prm->token;
      token[0] = '\0';
      if (len < 0) {
        /* an error occurred */
        return NULL;
      }
      else if (len == 0) {
        /* found the end-of-file marker, return empty token */
        return token;
      }
      prm->startbuf = prm->buffer;
      /* skip white space */
      while (isspace(*(prm->startbuf))) prm->startbuf++;
      /* check for no more chars in buffer */
      if (prm->startbuf[0] == '\0') {
        /* line is empty */
        token[0] = '\0';
      }
      /* otherwise scan for token */
      else if (sscanf(prm->startbuf, prm->tokenfmt, token) == 0) {
        /* matches a 1-char token that is not in the format set */
        token[0] = prm->startbuf[0];
        token[1] = '\0';
      }
    } while (strcasecmp(token, "REMARK") == 0
        || strcasecmp(token, "REMARKS") == 0);
    if (token[0] != '\0') {
      /* found first token */
      return token;
    }
  }
  /* skip length of last token */
  prm->startbuf += strlen(token);
  token[0] = '\0';
  while (token[0] == '\0') {
    /* skip white space */
    while (isspace(*(prm->startbuf))) prm->startbuf++;
    /* check for no more chars in buffer */
    if (prm->startbuf[0] == '\0') {
      /* line is empty - read in new line */
      len = xplor_readline(prm);
      /* need to reset token after readline */
      token = prm->token;
      token[0] = '\0';
      if (len < 0) {
        /* an error occured */
        return NULL;
      }
      else if (len == 0) {
        /* found the end-of-file marker, return empty token */
        return token;
      }
      prm->startbuf = prm->buffer;
    }
    /* otherwise scan for token */
    else if (sscanf(prm->startbuf, prm->tokenfmt, token) == 0) {
      /* matches a 1-char token that is not in the format set */
      token[0] = prm->startbuf[0];
      token[1] = '\0';
    }
  }
  /* return a non-empty token */
  return token;
}


/*
 * read from token stream
 * attempt to parse as a "type name" (up to 4-char string)
 * store value into MD_Name variable
 */
int xplor_parse_name(mdio_Param *prm, MD_Name name)
{
  const char *token;
  char *c;
  int cnt;
  char extra[4];

  token = xplor_token(prm);
  if (token == NULL) return MDIO_ERROR;
  else if (token[0] == '\0') {
    char s[60];
    snprintf(s, sizeof(s), "file ended while parsing %s",
        xplor_keyword(prm->tokenflags & PRIMARY_TOKEN));
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_UNXEOF, s);
    return MDIO_ERROR;
  }
  cnt = sscanf(token, "%4[a-zA-Z0-9*%#+]%1s", name, extra);
  if (cnt != 1) {
    char s[60];
    snprintf(s, sizeof(s), "failed to find type name while parsing %s",
        xplor_keyword(prm->tokenflags & PRIMARY_TOKEN));
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_SYNTAX, s);
    return MDIO_ERROR;
  }
  for (c = name;  *c != '\0';  c++) {
    *c = toupper(*c);
  }
  return 0;
}


/*
 * read from token stream
 * attempt to parse as an integer
 * store value into int variable
 */
int xplor_parse_int(mdio_Param *prm, int *n)
{
  const char *token;
  int cnt;
  char extra[4];
  int m = 0;

  *n = 0;
  if ((token = xplor_token(prm)) == NULL) return MDIO_ERROR;
  else if (token[0] == '\0') {
    char s[60];
    snprintf(s, sizeof(s), "file ended while parsing %s",
        xplor_keyword(prm->tokenflags & PRIMARY_TOKEN));
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_UNXEOF, s);
    return MDIO_ERROR;
  }
  cnt = sscanf(token, "%d%1s", &m, extra);
  if (cnt != 1) {
    char s[60];
    snprintf(s, sizeof(s), "failed to find integer value while parsing %s",
        xplor_keyword(prm->tokenflags & PRIMARY_TOKEN));
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_SYNTAX, s);
    return MDIO_ERROR;
  }
  *n = (int) m;
  return 0;
}


/*
 * read from token stream
 * attempt to parse as a real
 * store value into double variable
 */
int xplor_parse_real(mdio_Param *prm, double *x)
{
  const char *token;
  int cnt;
  char extra[4];
  double r = 0.0;

  *x = 0.0;
  if ((token = xplor_token(prm)) == NULL) return MDIO_ERROR;
  else if (token[0] == '\0') {
    char s[60];
    snprintf(s, sizeof(s), "file ended while parsing %s",
        xplor_keyword(prm->tokenflags & PRIMARY_TOKEN));
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_UNXEOF, s);
    return MDIO_ERROR;
  }
  cnt = sscanf(token, "%lf%1s", &r, extra);
  if (cnt != 1) {
    char s[60];
    snprintf(s, sizeof(s), "failed to find real value while parsing %s",
        xplor_keyword(prm->tokenflags & PRIMARY_TOKEN));
    mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_SYNTAX, s);
    return MDIO_ERROR;
  }
  *x = (double) r;
  return 0;
}


int xplor_parse_bond(mdio_Param *prm)
{
  MD_BondPrm bondp;

  ASSERT(prm->tokenflags & BOND);

  /* parse expected paramters from token stream */
  if (xplor_parse_name(prm, bondp.type[0])
      || xplor_parse_name(prm, bondp.type[1])
      || xplor_parse_real(prm, &bondp.k)
      || xplor_parse_real(prm, &bondp.r0)) return MDIO_ERROR;

  /* append to bondprm array */
  if (adt_appendList(&(prm->bondprm), &bondp)) {
    ERRMSG("cannot add bond params to list");
    mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int xplor_parse_angle(mdio_Param *prm)
{
  MD_AnglePrm anglp;
  const char *token;

  ASSERT(prm->tokenflags & ANGLE);

  /* read expected parameters from token stream */
  if (xplor_parse_name(prm, anglp.type[0])
      || xplor_parse_name(prm, anglp.type[1])
      || xplor_parse_name(prm, anglp.type[2])
      || xplor_parse_real(prm, &anglp.k_theta)
      || xplor_parse_real(prm, &anglp.theta0)) return MDIO_ERROR;

  /* check for UB option */
  if ((token = xplor_token(prm)) == NULL) return MDIO_ERROR;
  else if (xplor_lookup(prm, token) != UB) {
    /* push token back onto stream */
    prm->tokenflags |= PUSHBACK;
    /* init optional parameters */
    anglp.k_ub = 0.0;
    anglp.r_ub = 0.0;
  }
  /* read rest of parameters */
  else if (xplor_parse_real(prm, &anglp.k_ub)
      || xplor_parse_real(prm, &anglp.r_ub)) return MDIO_ERROR;

  /* apply conversion factors */
  anglp.theta0 *= MD_RADIANS;

  /* append to angleprm array */
  if (adt_appendList(&(prm->angleprm), &anglp)) {
    ERRMSG("cannot add angle params to list");
    mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int xplor_parse_dihedral(mdio_Param *prm)
{
  MD_TorsPrm torsp;
  const char *token;
  int mult = 1;  /* set default multiplicity to 1 */
  int i;

  ASSERT(prm->tokenflags & DIHEDRAL);

  /* read expected parameters from token stream */
  if (xplor_parse_name(prm, torsp.type[0])
      || xplor_parse_name(prm, torsp.type[1])
      || xplor_parse_name(prm, torsp.type[2])
      || xplor_parse_name(prm, torsp.type[3])) return MDIO_ERROR;

  /* check for MULTIPLE option */
  if ((token = xplor_token(prm)) == NULL) return MDIO_ERROR;
  else if (xplor_lookup(prm, token) == MULTIPLE) {
    /* check for optional "=" */
    if ((token = xplor_token(prm)) == NULL) return MDIO_ERROR;
    else if (xplor_lookup(prm, token) != EQUALS) {
      /* push back onto stream */
      prm->tokenflags |= PUSHBACK;
    }
    /* read the total multiplicity of this dihedral */
    if (xplor_parse_int(prm, &mult)) return MDIO_ERROR;
    if (mult < 1) {
      mdio_setErrorFile(&(prm->file), MDIO_ERROR_SYNTAX);
      return MDIO_ERROR;
    }
  }
  else {
    /* push back onto stream */
    prm->tokenflags |= PUSHBACK;
  }

  /* read the remaining terms */
  for (i = 0;  i < mult;  i++) {
    torsp.mult = i + 1;
    if (xplor_parse_real(prm, &torsp.k_tor)
        || xplor_parse_int(prm, &torsp.n)
        || xplor_parse_real(prm, &torsp.phi)) return MDIO_ERROR;
    /* apply conversion factors */
    if (torsp.n > 0) {
      /* reverse sign to match CHARMM dihedral function specification */
      torsp.phi *= -MD_RADIANS;
    }
    else {
      torsp.phi *= MD_RADIANS;
    }

    /* append to dihedprm array */
    if (adt_appendList(&(prm->dihedprm), &torsp)) {
      ERRMSG("cannot add dihedral params to list");
      mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
  }
  return 0;
}


int xplor_parse_improper(mdio_Param *prm)
{
  MD_TorsPrm torsp;
  const char *token;
  int mult = 1;  /* set default multiplicity to 1 */
  int i;

  ASSERT(prm->tokenflags & IMPROPER);

  /* read expected parameters from token stream */
  if (xplor_parse_name(prm, torsp.type[0])
      || xplor_parse_name(prm, torsp.type[1])
      || xplor_parse_name(prm, torsp.type[2])
      || xplor_parse_name(prm, torsp.type[3])) return MDIO_ERROR;

  /* check for MULTIPLE option */
  if ((token = xplor_token(prm)) == NULL) return MDIO_ERROR;
  else if (xplor_lookup(prm, token) == MULTIPLE) {
    /* check for optional "=" */
    if ((token = xplor_token(prm)) == NULL) return MDIO_ERROR;
    else if (xplor_lookup(prm, token) != EQUALS) {
      /* push back onto stream */
      prm->tokenflags |= PUSHBACK;
    }
    /* read the multiplicity of this dihedral */
    if (xplor_parse_int(prm, &mult)) return MDIO_ERROR;
    if (mult < 1) {
      mdio_setErrorFile(&(prm->file), MDIO_ERROR_SYNTAX);
      return MDIO_ERROR;
    }
  }
  else {
    /* push back onto stream */
    prm->tokenflags |= PUSHBACK;
  }

  /* read the remaining terms */
  for (i = 0;  i < mult;  i++) {
    torsp.mult = i + 1;
    if (xplor_parse_real(prm, &torsp.k_tor)
        || xplor_parse_int(prm, &torsp.n)
        || xplor_parse_real(prm, &torsp.phi)) return MDIO_ERROR;
    /* apply conversion factors */
    if (torsp.n > 0) {
      /* reverse sign to match CHARMM dihedral function specification */
      torsp.phi *= -MD_RADIANS;
    }
    else {
      torsp.phi *= MD_RADIANS;
    }

    /* append to imprprm array */
    if (adt_appendList(&(prm->imprprm), &torsp)) {
      ERRMSG("cannot add improper params to list");
      mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
      return MDIO_ERROR;
    }
  }
  return 0;
}


int xplor_parse_nbfix(mdio_Param *prm)
{
  MD_NbfixPrm nbfxp;
  double a, b, a14, b14;
  const double one_sixth = 1.0 / 6.0;

  ASSERT(prm->tokenflags & NBFIX);

  /* read expected parameters from token stream */
  if (xplor_parse_name(prm, nbfxp.type[0])
      || xplor_parse_name(prm, nbfxp.type[1])
      || xplor_parse_real(prm, &a)
      || xplor_parse_real(prm, &b)
      || xplor_parse_real(prm, &a14)
      || xplor_parse_real(prm, &b14)) return MDIO_ERROR;

  /* convert units */
  nbfxp.emin = -0.25 * b * b / a;
  nbfxp.rmin = pow(2.0 * a / b, one_sixth);
  nbfxp.emin14 = -0.25 * b14 * b14 / a14;
  nbfxp.rmin14 = pow(2.0 * a14 / b14, one_sixth);

  /* leave other fields uninitialized */
  nbfxp.prm[0] = -1;
  nbfxp.prm[1] = -1;

  /* append to nbfixprm array */
  if (adt_appendList(&(prm->nbfixprm), &nbfxp)) {
    ERRMSG("cannot add nbfix params to list");
    mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int xplor_parse_nonbonded(mdio_Param *prm)
{
  MD_AtomPrm atomp;
  const double sixth_root_of_two = pow(2.0, 1.0/6.0);

  ASSERT(prm->tokenflags & NONBONDED);

  /* read expected parameters from token stream */
  if (xplor_parse_name(prm, atomp.type)
      || xplor_parse_real(prm, &atomp.emin)
      || xplor_parse_real(prm, &atomp.rmin)
      || xplor_parse_real(prm, &atomp.emin14)
      || xplor_parse_real(prm, &atomp.rmin14)) return MDIO_ERROR;

  /* apply conversion factors */
  atomp.emin = -atomp.emin;
  atomp.rmin *= sixth_root_of_two;
  atomp.emin14 = -atomp.emin14;
  atomp.rmin14 *= sixth_root_of_two;

  /* append to nbfixprm array */
  if (adt_appendList(&(prm->atomprm), &atomp)) {
    ERRMSG("cannot add atom params to list");
    mdio_setErrorFile(&(prm->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int xplor_parse_hbonded(mdio_Param *prm)
{
  /*
   * MDAPI does not support explicit hydrogen bonds
   * just read in data, check syntax, discard
   */
  MD_Name t;
  double r;

  ASSERT(prm->tokenflags & HBONDED);

  /* read expected parameters from token stream */
  if (xplor_parse_name(prm, t)
      || xplor_parse_name(prm, t)
      || xplor_parse_real(prm, &r)
      || xplor_parse_real(prm, &r)) return MDIO_ERROR;
  
  return 0;
}


/*
 * main reading routine
 * called at beginning of read or if attempting error recovery
 */
int xplor_read(mdio_Param *prm)
{
  int isnewfile = 1;  /* assume start of brand new file */
  int isdone = 0;
  const char *token;
  int id, tid;

  /* assume filetype is X-Plor */
  prm->filetype = MDIO_PARAM_XPLOR;

  /* if error flag was set, then set recover from previous read error */
  if (prm->tokenflags & ERROR) {
    prm->tokenflags = RECOVER;
    isnewfile = 0;
  }

  /* read loop */
  while (!isdone) {
    if ((token = xplor_token(prm)) == NULL) return xplor_error(prm);
    id = xplor_lookup(prm, token);
    if (id & PRIMARY_TOKEN) {
      /* once we see a primary token, we have finished any error recovery */
      prm->tokenflags &= ~RECOVER;
    }

    switch (id) {
        /*
         * finished parsing file
         */
      case ENDOFILE:
        isdone = 1;
        break;

        /*
         * these tokens we can parse, call appropriate parse function
         */
      case BOND:
        prm->tokenflags |= id;
        if (xplor_parse_bond(prm)) return xplor_error(prm);
        prm->tokenflags &= ~id;
        break;

      case ANGLE:
        prm->tokenflags |= id;
        if (xplor_parse_angle(prm)) return xplor_error(prm);
        prm->tokenflags &= ~id;
        break;

      case DIHEDRAL:
        prm->tokenflags |= id;
        if (xplor_parse_dihedral(prm)) return xplor_error(prm);
        prm->tokenflags &= ~id;
        break;

      case IMPROPER:
        prm->tokenflags |= id;
        if (xplor_parse_improper(prm)) return xplor_error(prm);
        prm->tokenflags &= ~id;
        break;

      case NBFIX:
        prm->tokenflags |= id;
        if (xplor_parse_nbfix(prm)) return xplor_error(prm);
        prm->tokenflags &= ~id;
        break;

      case NONBONDED:
        prm->tokenflags |= id;
        if (xplor_parse_nonbonded(prm)) return xplor_error(prm);
        prm->tokenflags &= ~id;
        break;

      case HBONDED:
        prm->tokenflags |= id;
        if (xplor_parse_hbonded(prm)) return xplor_error(prm);
        prm->tokenflags &= ~id;
        break;

        /*
         * the following tokens are legal, but we can't do anything with them
         * do some syntax checking, but otherwise ignore
         */
      case HBONDS:
      case NBONDS:
      case SET:
      case LEARN:
      case REDUCE:
        prm->tokenflags |= id;
        /* skip over block until END is found */
        if ((token = xplor_token(prm)) == NULL) return xplor_error(prm);
        while ((tid = xplor_lookup(prm, token)) != END) {
          /* make sure it isn't EOF or primary token, ignore others */
          if (tid == ENDOFILE) {
            char s[80];
            snprintf(s, sizeof(s),
                "file ended in middle of %s block before END keyword",
                xplor_keyword(id));
            mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_UNXEOF, s);
            return xplor_error(prm);
          }
          else if (tid & PRIMARY_TOKEN) {
            char s[80];
            snprintf(s, sizeof(s),
                "found %s in middle of %s block while looking for END",
                xplor_keyword(tid), xplor_keyword(id));
            mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_SYNTAX, s);
            return xplor_error(prm);
          }
          if ((token = xplor_token(prm)) == NULL) return xplor_error(prm);
        }
        prm->tokenflags &= ~id;
        break;

      case RESET:
        /* see if option follows */
        if ((token = xplor_token(prm)) == NULL) return xplor_error(prm);
        tid = xplor_lookup(prm, token);
        if (!(tid & RESET_OPTIONS)) {
          /* no option, push back token */
          prm->tokenflags |= PUSHBACK;
        }
        break;

      case VERBOSE:
        /* ignore */
        break;

        /*
         * otherwise something unknown
         * if we are recovering from a previous error, skip over it
         * otherwise return an error
         */
      default:
        /*
         * try to detect CHARMM parameters file
         */
        if (isnewfile && strcmp(token, "*") == 0) {
          /*
           * assume filetype is instead CHARMM parameters file
           */
          prm->filetype = MDIO_PARAM_CHARMM;
          return MDIO_PARAM_CHARMM;
        }
        else if (!(prm->tokenflags & RECOVER)) {
          char s[80];
          snprintf(s, sizeof(s), "found unknown token \"%s\"", token);
          mdio_setErrorMessageFile(&(prm->file), MDIO_ERROR_SYNTAX, s);
          return xplor_error(prm);
        }
        break;
    } /* end switch */

    /*
     * any successful read means this is *not* a CHARMM parameters file
     */
    isnewfile = 0;  

  } /* end while */
  return 0;
}



/*****************************************************************************
 *
 * Down and dirty reader for CHARMM parameter files:
 * - files must start with '*' comment (typically '*' is first char in file)
 * - performs minimal checking for correctness
 * - wildcards are not correctly supported for atom type matching
 *
 *****************************************************************************/

int charmm_read(mdio_Param *p)
{
  int state = COMMENT;  /* skip comment block at beginning of file */
  int skip_prm = 1;
  int skip_cmap = 0;
  int retval;
  char tok[16];

  while ((retval = mdio_readLineFile(&(p->file),
          &(p->buffer), &(p->bufferlen))) > 0) {
    tok[0] = '\0';
    sscanf(p->buffer, "%15s", tok);
    switch (state) {
      case COMMENT:
        if (strncasecmp(tok, "BOND", 4) == 0) {
          state = BOND;
        }
        else if (p->buffer[0] != '*' && tok[0] != '!' && tok[0] != '\0') {
          char s[80];
          snprintf(s, sizeof(s), "found unknown token \"%s\"", tok);
          mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_SYNTAX, s);
          return MDIO_ERROR;
        }
        break;
      case BOND:
        if (strncasecmp(tok, "ANGL", 4) == 0
            || strcasecmp(tok, "THETA") == 0) {
          state = ANGLE;
        }
        else if (tok[0] != '\0' && tok[0] != '!' && charmm_parse_bond(p)) {
          mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_SYNTAX,
              "syntax error in CHARMM bond");
          return MDIO_ERROR;
        }
        break;
      case ANGLE:
        if (strncasecmp(tok, "DIHE", 4) == 0
            || strcasecmp(tok, "PHI") == 0) {
          state = DIHEDRAL;
        }
        else if (tok[0] != '\0' && tok[0] != '!' && charmm_parse_angle(p)) {
          mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_SYNTAX,
              "syntax error in CHARMM angle");
          return MDIO_ERROR;
        }
        break;
      case DIHEDRAL:
        if (strncasecmp(tok, "IMPR", 4) == 0
            || strcasecmp(tok, "IMPHI") == 0) {
          state = IMPROPER;
        }
        else if (tok[0] != '\0' && tok[0] != '!' && charmm_parse_dihedral(p)) {
          mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_SYNTAX,
              "syntax error in CHARMM dihedral");
          return MDIO_ERROR;
        }
        break;
      case IMPROPER:
        if (strncasecmp(tok, "NBON", 4) == 0
            || strncasecmp(tok, "NONB", 4) == 0) {
          state = NONBONDED;
        }
        else if (strcasecmp(tok, "CMAP") == 0) {
          /* if it exists, skip over entire CMAP section */
          skip_cmap = 1;
        }
        else if (!skip_cmap
            && tok[0] != '\0' && tok[0] != '!' && charmm_parse_improper(p)) {
          mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_SYNTAX,
              "syntax error in CHARMM improper");
          return MDIO_ERROR;
        }
        break;
      case NONBONDED:
        if (skip_prm && strcasecmp(tok, "cutnb") == 0) {
          skip_prm = 0;
        }
        else if (strcasecmp(tok, "NBFIX") == 0) {
          state = NBFIX;
        }
        else if (strcasecmp(tok, "HBOND") == 0 || strcasecmp(tok, "END") == 0) {
          return 0;  /* all done! */
        }
        else if (tok[0] != '\0' && tok[0] != '!' && charmm_parse_nonbonded(p)) {
          mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_SYNTAX,
              "syntax error in CHARMM nonbonded");
          return MDIO_ERROR;
        }
        break;
      case NBFIX:
        if (strcasecmp(tok, "HBOND") == 0 || strcasecmp(tok, "END") == 0) {
          return 0;  /* all done! */
        }
        else if (tok[0] != '!' && charmm_parse_nbfix(p)) {
          mdio_setErrorMessageFile(&(p->file), MDIO_ERROR_SYNTAX,
              "syntax error in CHARMM nbfix");
          return MDIO_ERROR;
        }
        break;
      default:
        BUG("invalid state while parsing CHARMM parameter file");
    }
  }
  return retval;
}


int charmm_parse_bond(mdio_Param *p)
{
  MD_BondPrm bondp;
  int cnt;
  char extra[4] = "";

  cnt = sscanf(p->buffer, "%5[a-zA-Z0-9*%#+] %5[a-zA-Z0-9*%#+] %lf %lf%1s",
      bondp.type[0], bondp.type[1], &bondp.k, &bondp.r0, extra);
  if (cnt < 4 || (cnt == 5 && extra[0] != '!')
      || strlen(bondp.type[0]) == 5
      || strlen(bondp.type[1]) == 5) return MDIO_ERROR;

  /* append to bondprm array */
  if (adt_appendList(&(p->bondprm), &bondp)) {
    ERRMSG("cannot add bond params to list");
    mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int charmm_parse_angle(mdio_Param *p)
{
  MD_AnglePrm anglp;
  int cnt;
  char extra[4] = "";

  /* Urey-Bradley constants set to zero if not present */
  anglp.k_ub = 0.0;
  anglp.r_ub = 0.0;

  cnt = sscanf(p->buffer, "%5[a-zA-Z0-9*%#+] %5[a-zA-Z0-9*%#+] "
      "%5[a-zA-Z0-9*%#+] %lf %lf %lf %lf%1s",
      anglp.type[0], anglp.type[1], anglp.type[2],
      &anglp.k_theta, &anglp.theta0, &anglp.k_ub, &anglp.r_ub, extra);
  if ((cnt == 8 && extra[0] != '!')
      || (cnt != 5 && cnt != 7 && cnt != 8)
      || strlen(anglp.type[0]) == 5
      || strlen(anglp.type[1]) == 5
      || strlen(anglp.type[2]) == 5) return MDIO_ERROR;

  /* apply conversion factors */
  anglp.theta0 *= MD_RADIANS;

  /* append to angleprm array */
  if (adt_appendList(&(p->angleprm), &anglp)) {
    ERRMSG("cannot add angle params to list");
    mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int charmm_parse_dihedral(mdio_Param *p)
{
  MD_TorsPrm torsp;
  MD_TorsPrm *tlist;
  int cnt, len;
  char extra[4] = "";

  /* start with assumption that multiplicity is 1 */
  torsp.mult = 1;

  cnt = sscanf(p->buffer, "%5[a-zA-Z0-9*%#+] %5[a-zA-Z0-9*%#+] "
      "%5[a-zA-Z0-9*%#+] %5[a-zA-Z0-9*%#+] %lf %d %lf%1s",
      torsp.type[0], torsp.type[1], torsp.type[2], torsp.type[3],
      &torsp.k_tor, &torsp.n, &torsp.phi, extra);
  if (cnt < 7 || (cnt == 8 && extra[0] != '!')
      || strlen(torsp.type[0]) == 5
      || strlen(torsp.type[1]) == 5
      || strlen(torsp.type[2]) == 5
      || strlen(torsp.type[3]) == 5) return MDIO_ERROR;

  /* apply conversion factors */
  torsp.phi *= MD_RADIANS;

  /*
   * NOTE: according to CHARMM force field, dihedrals should always have
   *   periodicity n > 0, but we won't enforce this restriction.
   */

  /* append to dihedprm array */
  if (adt_appendList(&(p->dihedprm), &torsp)) {
    ERRMSG("cannot add dihedral params to list");
    mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }

  /* determine if multiplicity is > 1 */
  tlist = (MD_TorsPrm *) adt_getDataList(&(p->dihedprm));
  len = adt_getLengthList(&(p->dihedprm));
  if (len > 1
      && strcmp(tlist[len-1].type[0], tlist[len-2].type[0]) == 0
      && strcmp(tlist[len-1].type[1], tlist[len-2].type[1]) == 0
      && strcmp(tlist[len-1].type[2], tlist[len-2].type[2]) == 0
      && strcmp(tlist[len-1].type[3], tlist[len-2].type[3]) == 0) {
    /* for this mult, add 1 to previous mult field having matching entries */
    tlist[len-1].mult = tlist[len-2].mult + 1;
  }
  return 0;
}


int charmm_parse_improper(mdio_Param *p)
{
  MD_TorsPrm torsp;
  MD_TorsPrm *tlist;
  int cnt, len;
  char extra[4] = "";

  /* start with assumption that multiplicity is 1 */
  torsp.mult = 1;

  cnt = sscanf(p->buffer, "%5[a-zA-Z0-9*%#+] %5[a-zA-Z0-9*%#+] "
      "%5[a-zA-Z0-9*%#+] %5[a-zA-Z0-9*%#+] %lf %d %lf%1s",
      torsp.type[0], torsp.type[1], torsp.type[2], torsp.type[3],
      &torsp.k_tor, &torsp.n, &torsp.phi, extra);
  if (cnt < 7 || (cnt == 8 && extra[0] != '!')
      || strlen(torsp.type[0]) == 5
      || strlen(torsp.type[1]) == 5
      || strlen(torsp.type[2]) == 5
      || strlen(torsp.type[3]) == 5) return MDIO_ERROR;

  /* apply conversion factors */
  torsp.phi *= MD_RADIANS;

  /*
   * NOTE: according to CHARMM force field, impropers should always have
   *   periodicity n = 0 with only one term (i.e. always have mult=1),
   *   but we won't enforce this restriction.
   */

  /* append to dihedprm array */
  if (adt_appendList(&(p->imprprm), &torsp)) {
    ERRMSG("cannot add improper params to list");
    mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }

  /* determine if multiplicity is > 1 */
  tlist = (MD_TorsPrm *) adt_getDataList(&(p->imprprm));
  len = adt_getLengthList(&(p->imprprm));
  if (len > 1
      && strcmp(tlist[len-1].type[0], tlist[len-2].type[0]) == 0
      && strcmp(tlist[len-1].type[1], tlist[len-2].type[1]) == 0
      && strcmp(tlist[len-1].type[2], tlist[len-2].type[2]) == 0
      && strcmp(tlist[len-1].type[3], tlist[len-2].type[3]) == 0) {
    /* for this mult, add 1 to previous mult field having matching entries */
    tlist[len-1].mult = tlist[len-2].mult + 1;
  }
  return 0;
}


int charmm_parse_nonbonded(mdio_Param *p)
{
  MD_AtomPrm atomp;
  int cnt;
  char extra[4] = "";

  cnt = sscanf(p->buffer, "%5[a-zA-Z0-9*%#+] %*f %lf %lf %*f %lf %lf%1s",
      atomp.type, &atomp.emin, &atomp.rmin, &atomp.emin14, &atomp.rmin14,
      extra);
  if ((cnt == 6 && extra[0] != '!')
      || (cnt != 3 && cnt != 5 && cnt != 6)
      || strlen(atomp.type) == 5) return MDIO_ERROR;

  /* if not given, set scaled1-4 params to same value */
  if (cnt == 3) {
    atomp.emin14 = atomp.emin;
    atomp.rmin14 = atomp.rmin;
  }

  /* apply conversion factors */
  /* CHARMM parameters given as rmin/2 and rmin14/2 */
  atomp.rmin *= 2.0;
  atomp.rmin14 *= 2.0;

  /* append to atomprm array */
  if (adt_appendList(&(p->atomprm), &atomp)) {
    ERRMSG("cannot add atom params to list");
    mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}


int charmm_parse_nbfix(mdio_Param *p)
{
  MD_NbfixPrm nbfxp;
  int cnt;
  char extra[4] = "";

  cnt = sscanf(p->buffer, "%5[a-zA-Z0-9*%#+] %5[a-zA-Z0-9*%#+] "
      "%lf %lf %lf %lf%1s", nbfxp.type[0], nbfxp.type[1],
      &nbfxp.emin, &nbfxp.rmin, &nbfxp.emin14, &nbfxp.rmin14, extra);
  if ((cnt == 7 && extra[0] != '!')
      || (cnt != 4 && cnt != 6 && cnt != 7)
      || strlen(nbfxp.type[0]) == 5
      || strlen(nbfxp.type[1]) == 5) return MDIO_ERROR;

  /* if not given, set scaled1-4 params to same value */
  if (cnt == 4) {
    nbfxp.emin14 = nbfxp.emin;
    nbfxp.rmin14 = nbfxp.rmin;
  }

  /*
   * NOTE:  unlike NONBONDED, these are listed as rmin and rmin14,
   *   so no conversion factor needed.
   */

  /* append to nbfixprm array */
  if (adt_appendList(&(p->nbfixprm), &nbfxp)) {
    ERRMSG("cannot add nbfix params to list");
    mdio_setErrorFile(&(p->file), MDIO_ERROR_NOMEM);
    return MDIO_ERROR;
  }
  return 0;
}
