/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    file.h
 * @brief   File base class.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c mdio_File class performs the actual I/O along with
 * diagnostic and error handling capabilities.  It is the base
 * class for the specialized reader and writer classes.
 * For accessing supported types of MD files, instead use
 * the higher level "derived" classes.
 *
 * The file I/O routines are a somewhat simplified version of
 * the C standard I/O library function calls.  The fact that
 * the actual I/O calls are hidden behind this layer means that
 * the implementation could depend on the file type, for example,
 * use of @c mmap() for large trajectory files.
 */

#ifndef MDIO_FILE_H
#define MDIO_FILE_H

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif

  enum {
    MDIO_ERROR = -1,          /**< Return value from failed function call. */
    MDIO_LENGTH_ERRMSG = 240  /* length of error message diagnostic string */
  };

  /**@brief File type flags. */
  enum {
    MDIO_FILE_TEXT   = 0x001, /**< Flag indicating text file. */
    MDIO_FILE_BINARY = 0x002, /**< Flag indicating binary file. */
    MDIO_FILE_READ   = 0x004, /**< Flag indicating reading from a file. */
    MDIO_FILE_WRITE  = 0x008  /**< Flag indicating writing to a file. */
  };

  /**@brief Error numbers. */
  enum {
    MDIO_ERROR_NONE,      /**< No error. */
    MDIO_ERROR_WARN,      /**< A warning. */
    MDIO_ERROR_BADVAL,    /**< An illegal value pertaining to file. */
    MDIO_ERROR_NOMEM,     /**< Memory cannot be allocated. */
    MDIO_ERROR_OPEN,      /**< Cannot open a file. */
    MDIO_ERROR_CLOSE,     /**< Cannot close a file. */
    MDIO_ERROR_READ,      /**< Unable to read from file. */
    MDIO_ERROR_WRITE,     /**< Unable to write to file. */
    MDIO_ERROR_SEEK,      /**< Unable to perform a byte seek within file. */
    MDIO_ERROR_SYNTAX,    /**< Syntax error occurred with file. */
    MDIO_ERROR_UNXEOF,    /**< The end-of-file marker occurred unexpectedly. */
    MDIO_LENGTH_ERRORLIST /* number of error constants, marks end of list */
  };

  enum {
    MDIO_EOF = 0x001,          /**< End-of-file constant for MDIO. */
    MDIO_SEEK_SET = SEEK_SET,  /**< Byte offset wrt beginning of file. */
    MDIO_SEEK_CUR = SEEK_CUR,  /**< Byte offset wrt current position. */
    MDIO_SEEK_END = SEEK_END   /**< Byte offset wrt end of file. */
  };

  /**@brief File base class.
   *
   * Members should be treated as private.
   */
  typedef struct mdio_File_t {
    FILE *file;
    const char *name;
    int (*errhandler)(struct mdio_File_t *);
    int filetype;
    int errnum;
    int bytenum;
    int linenum;
    int status;
    char errmsg[MDIO_LENGTH_ERRMSG];
  } mdio_File;


/* constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated file object.
   *
   * @return Pointer to file object or @c NULL on failure.
   */
  mdio_File *mdio_createFile(void);


  /**@brief Destructor.
   *
   * Frees dynamically allocated file object.
   */
  void mdio_destroyFile(mdio_File *);


/* basic file manipulation functions */

  /**@brief Open file.
   *
   * @param[in] name  The file name, nil-terminated string.
   * @param[in] filetype  Bitwise OR'ed combination of file type flags.
   *
   * The @c name string is expected to persist until the file is
   * closed with @c mdio_closeFile().
   *
   * For @c filetype, must choose either
   * @c MDIO_FILE_TEXT or @c MDIO_FILE_BINARY
   * combined (using bitwise OR operator) with either
   * @c MDIO_FILE_READ or @c MDIO_FILE_WRITE.
   *
   * @return 0 on success or @c MDIO_ERROR on failure.
   */
  int mdio_openFile(mdio_File *, const char *name, int filetype);


  /**@brief Close file.
   *
   * @return 0 on success or @c MDIO_ERROR on failure.
   */
  int mdio_closeFile(mdio_File *);


  /**@brief Read entire line from text file.
   *
   * @param[in,out] pbuf  Address of the @c char buffer pointer.
   * @param[in,out] pbuflen  Address of the @c char buffer length variable.
   *
   * Read text file up to and including the next newline character,
   * storing the characters into the buffer.
   * The @c char buffer is nil-terminated.
   * The address @p pbuf needs to point to memory that can
   * be managed using @c realloc() so that the buffer can be expanded
   * arbitrarily to contain the entire line, with the updated length
   * stored back to @p pbuflen.  The caller is responsible for later
   * freeing the memory allocation.
   *
   * @return The number of characters read into the buffer.
   * End-of-file is indicated by returning 0 and storing the
   * empty string into the @p pbuf buffer.
   * A blank line is indicated by returning 1 and storing "\n"
   * into the @p pbuf buffer.
   * An error is indicated by returning @c MDIO_ERROR.
   */
  int mdio_readLineFile(mdio_File *, char **pbuf, int *pbuflen);


  /**@brief Read from text file.
   *
   * @param[in,out] buf  Points to the @c char buffer.
   * @param[in] buflen  The length of the @c char buffer.
   *
   * Read up to and including the next newline character, but no more
   * than @c buflen - 1 characters.  Similar to @c fgets().
   *
   * @return The number of characters read into the buffer.
   * End-of-file is indicated by returning 0 and storing the
   * empty string into the @p pbuf buffer.
   * A blank line is indicated by returning 1 and storing "\n"
   * into the @p pbuf buffer.
   * An error is indicated by returning @c MDIO_ERROR.
   */
  int mdio_readTextFile(mdio_File *, char *buf, int buflen);


  /**@brief Write to text file.
   *
   * @param[in] buf  Points to a nil-terminated string.
   *
   * Write the string, not including the nil-terminator, to the file.
   * Newline characters must be explicitly included in the string.
   * Similar to @c fputs().
   *
   * @return The number of characters written or @c MDIO_ERROR on error.
   */
  int mdio_writeTextFile(mdio_File *, const char *buf);


  /**@brief Read from binary file.
   *
   * @param[out] buffer  Points to memory buffer.
   * @param[in] elemsz  Size in bytes of each element.
   * @param[in] n  The number of elements to be read.
   *
   * Read @p n elements each of size @p elemsz bytes into the memory
   * buffer pointed to by @p buffer.  Similar to @c fread().
   *
   * @return The number of elements read.  If it is less than @p n,
   * then either the end-of-file was reached or and error occurred.
   */
  int mdio_readBinaryFile(mdio_File *, void *buffer, int elemsz, int n);


  /**@brief Write to binary file.
   *
   * @param[in] buffer  Points to memory buffer.
   * @param[in] elemsz  Size in bytes of each element.
   * @param[in] n  The number of elements to be written.
   *
   * Write the @p n @f$ \times @f$ @p elemsz bytes stored in @p buffer
   * to the file.  Similar to @c fwrite().
   *
   * @return The number of elements written.  If it is less than @p n,
   * then an error has occurred.
   */
  int mdio_writeBinaryFile(mdio_File *, const void *buffer, int elemsz, int n);


  /**@brief Seek position within binary file.
   *
   * @param[in] offset  Byte offset relative to some location.
   * @param[in] whence  Flag to indicate the byte offset location.
   *
   * The seek offset in a binary file may be measured from
   * one of the following locations:
   * @li @c MDIO_SEEK_SET  - beginning of file
   * @li @c MDIO_SEEK_CUR  - current location in file
   * @li @c MDIO_SEEK_END  - end of file
   *
   * The current file position is then changed to the indicated location.
   *
   * @return 0 on success or @c MDIO_ERROR on failure.
   */
  int mdio_seekFile(mdio_File *, long offset, int whence);


  /**@brief Return nonzero if end-of-file has been reached,
   * otherwise return 0.
   */
  int mdio_isEndOfFile(mdio_File *);


  /**@brief Set error status on file.
   *
   * @param[in] errnum  MDIO error number.
   *
   * The error status on the file is set to the indicated error number
   * and a corresponding diagnostic message.  The message might include
   * information on the name and location within the file where the
   * error occurred, depending on the type of error.
   *
   * An illegal value for @p errnum will terminate the program.
   *
   * If the error handler has been set (see @c mdio_setErrorHandlerFile() ),
   * then that routine will be invoked before returning.
   *
   * @return The value from the error handler routine or @p errnum
   * if the error handler has been set to @c NULL.
   */
  int mdio_setErrorFile(mdio_File *, int errnum);


  /**@brief Set error status on file with customized message.
   *
   * @param[in] errnum  MDIO error number.
   * @param[in] msg  A nil-terminated string.
   *
   * This routine is identical to @c mdio_setErrorFile() with the
   * addition that the @p msg string will be appended to the diagnostic
   * message.
   */
  int mdio_setErrorMessageFile(mdio_File *, int errnum, const char *msg);


/* field access functions for error handling */

  /**@brief Set error handler routine.
   *
   * @param[in] errhandler  Routine to handle errors.
   *
   * The error handler is invoked whenever an error occurs, as the
   * conclusion of @c mdio_setErrorFile() or @c mdio_setErrorMessageFile().
   * The default error handler prints to @c stderr the error diagnostic
   * message.  It can be replaced by a user-defined error handling
   * routine or resetting to @c NULL.
   *
   * Any of the following routines are available to the error handler:
   * @c mdio_getNameFile(), @c mdio_getTypeFile(), @c mdio_getErrorFile(),
   * @c mdio_getBytenumFile(), @c mdio_getLinenumFile(),
   * @c mdio_getErrorMessageFile().
   */
  void mdio_setErrorHandlerFile(mdio_File *, int (*errhandler)(mdio_File *));


  /**@brief Returns the name of the file. */
  const char *mdio_getNameFile(mdio_File *);


  /**@brief Returns the type of the file. */
  int mdio_getTypeFile(mdio_File *);


  /**@brief Returns the current error number set for the file. */
  int mdio_getErrorFile(mdio_File *);


  /**@brief Returns the current byte position for the file. */
  int mdio_getBytenumFile(mdio_File *);


  /**@brief Returns the current line number for the file. */
  int mdio_getLinenumFile(mdio_File *);


  /**@brief Returns the current diagnostic error message for the file. */
  const char *mdio_getErrorMessageFile(mdio_File *);


  /**@brief Resets the error status for the file. */
  void mdio_resetErrorFile(mdio_File *);


/* constructor and destructor for preallocated mdio_File object */

  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated file object.
   * See @c mdio_createFile() for a description of expected arguments.
   */
  int mdio_initializeFile(mdio_File *);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated file object
   * (i.e. one constructed using @c mdio_initializeFile() ).
   */
  void mdio_cleanupFile(mdio_File *);

#ifdef __cplusplus
}
#endif

#endif /* MDIO_FILE_H */
