/*
 * Copyright (C) 2004-2005 by David J. Hardy.  All rights reserved.
 */

/**@file    bincoord.h
 * @brief   Binary coordinate reader and writer.
 * @author  David J. Hardy
 * @date    2003-2005
 *
 * The @c mdio_Bincoord class reads and writes NAMD-style binary coordinate
 * files.  Reads are stored into an array of @c MD_Dvec coordinates which
 * is then obtained from the @c mdio_Bincoord object.  The @c mdio_Bincoord
 * object can also be provided with a user-defined array of @c MD_Dvec
 * coordinates for writing.
 *
 * The @c mdio_Bincoord class is derived from @c mdio_File so that error
 * handling is performed by typecasting (@c mdio_Bincoord @c *) to
 * (@c mdio_File @c *).
 */

#ifndef MDIO_BINCOORD_H
#define MDIO_BINCOORD_H

#include "adt/array.h"
#include "mdio/file.h"
#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief Binary coordinate reader and writer class.
   *
   * Members should be treated as private.
   */
  typedef struct mdio_Bincoord_t {
    mdio_File file;
    adt_Array dvec;
  } mdio_Bincoord;


/* constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated binary coordinate file reader and
   * writer object.
   *
   * @return Pointer to new object or @c NULL on failure.
   */
  mdio_Bincoord *mdio_createBincoord(void);


  /**@brief Destructor.
   *
   * Frees dynamically allocated binary coordinate file reader and
   * writer object after freeing an allocated array of @c MD_Dvec.
   *
   * Note that an @c MD_Dvec array obtained from reading a file does not
   * persist beyond the destruction of the @c mdio_Bincoord object.
   */
  void mdio_destroyBincoord(mdio_Bincoord *);


/* reading file and obtaining data */

  /**@brief Read binary coordinate file.
   *
   * @param[in] name  A nil-terminated string naming the binary
   *   coordinate file.
   * @param[in] n_expect  The number of coordinates expected in the file.
   *
   * If the number of coordinates is unknown, then @c n_expect should be
   * set to 0.  Otherwise, this value is used to validate the expected
   * number of coordinates.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.  It is considered an
   * error for @c n_expect to not match the number of coordinates
   * contained in the file.
   */
  int mdio_readBincoord(mdio_Bincoord *, const char *name, int n_expect);


  /**@brief Obtain array of coordinates.
   *
   * @param[out] nelems  Length of array returned into indicated variable.
   *
   * If this array resulted from @c mdio_readBincoord(), then it will
   * persist only until destruction of the @c mdio_Bincoord object.
   *
   * @return The @c MD_Dvec coordinate array as a pointer.
   */
  MD_Dvec *mdio_getBincoord(mdio_Bincoord *, int *nelems);


/* providing data and writing file */

  /**@brief Set an array of coordinates.
   *
   * @param[in] dvec  Array of coordinates.
   * @param[in] nelems  Number of elements in array.
   *
   * The @c dvec array will still belong to the user, which means that
   * no attempt will be made to free it, and the burden of any dynamic
   * memory management still belongs to the user.
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_setBincoord(mdio_Bincoord *, MD_Dvec *dvec, int nelems);


  /**@brief Write binary coordinate file.
   *
   * @param[in] name  A nil-terminated string naming the binary
   *   coordinate file.
   *
   * The coordinates written will be whatever is known by the
   * @c mdio_Bincoord object, either from a previous call to
   * @c mdio_setBincoord() or a previous read using @c mdio_readBincoord().
   *
   * @return 0 on success, @c MDIO_ERROR on failure.
   */
  int mdio_writeBincoord(mdio_Bincoord *, const char *name);


/* constructor and destructor for pre-allocated mdio_Bincoord object */

  /**@brief Alternative constructor.
   *
   * Use to construct a preallocated @c mdio_Bincoord object.
   * See @c mdio_createBincoord() for a description of expected arguments.
   */
  int mdio_initializeBincoord(mdio_Bincoord *);


  /**@brief Alternative destructor.
   *
   * Use to destroy a preallocated @c mdio_Bincoord object
   * (i.e. one constructed using @c mdio_initializeBincoord() ).
   */
  void mdio_cleanupBincoord(mdio_Bincoord *);

#ifdef __cplusplus
}
#endif

#endif /* MDIO_BINCOORD_H */
