/*
 * validate safe Buckingham potential
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include "force/force.h"

#define ARRAYLEN(a)  (sizeof(a)/sizeof(a[0]))

#define ERROR(msg)  \
  fprintf(stderr, "FORCE ERROR:  file %s, line %d:  %s\n", \
      __FILE__, __LINE__, msg);

#define CUTOFF      12.0
#define SWITCHDIST  10.0


int main()
{
  MD_AtomPrm atomprm[] = {
    { 0, 0, 0, 0, "SI" },
    { 0, 0, 0, 0, "O" },
  };
  const int natomprms = ARRAYLEN(atomprm);

  const int vdwopts[] = {
    FORCE_VDW_BUCKPRM_BKS  | FORCE_VDW_BUCKSAFE,
    FORCE_VDW_BUCKPRM_TTAM | FORCE_VDW_BUCKSAFE,
    FORCE_VDW_BUCKPRM_FB   | FORCE_VDW_BUCKSAFE,
    FORCE_VDW_BUCKPRM_BKS  | FORCE_VDW_SWITCHBUCKSAFE,
    FORCE_VDW_BUCKPRM_TTAM | FORCE_VDW_SWITCHBUCKSAFE,
    FORCE_VDW_BUCKPRM_FB   | FORCE_VDW_SWITCHBUCKSAFE,
  };
  const int nvdwopts = ARRAYLEN(vdwopts);

  ForceVdwparam *vdwparam;
  ForceBucksafe *bucksafe;

  int i, j, k, n;

  const double cutoff2 = CUTOFF * CUTOFF;
  const double switchdist2 = SWITCHDIST * SWITCHDIST;
  const double denom = 1.0 / ((cutoff2-switchdist2) *
      (cutoff2-switchdist2) * (cutoff2-switchdist2));


  vdwparam = (ForceVdwparam *)
    calloc(natomprms * natomprms, sizeof(ForceVdwparam));

  bucksafe = (ForceBucksafe *)
    calloc(natomprms * natomprms, sizeof(ForceBucksafe));

  for (n = 0;  n < nvdwopts;  n++) {
    double a_si_o, a_o_o, a_si_si;
    double b_si_o, b_o_o, b_si_si;
    double c_si_o, c_o_o, c_si_si;

    printf("# setting Buckingham parameterization\n");
    switch (FORCE_MASK_VDW_BUCKPRM & vdwopts[n]) {
      case FORCE_VDW_BUCKPRM_BKS:
        printf("#   using BKS parameterization for silica\n");
        a_si_o  = 415020.541915277;  /* kcal/mol */
        a_o_o   = 32011.8325889865;  /* kcal/mol */
        a_si_si = 0.0;               /* kcal/mol */
        b_si_o  = 0.205205506984223; /* Ang */
        b_o_o   = 0.362323482844775; /* Ang */
        b_si_si = 0.333333333333333; /* Ang */
        c_si_o  = 3078.35790120536;  /* Ang^6 kcal/mol */
        c_o_o   = 4034.18491523635;  /* Ang^6 kcal/mol */
        c_si_si = 0.0;               /* Ang^6 kcal/mol */
        break;
      case FORCE_VDW_BUCKPRM_TTAM:
        printf("#   using TTAM parameterization for silica\n");
        a_si_o  = 247244.172408168;  /* kcal/mol */
        a_o_o   = 40514.4105526705;  /* kcal/mol */
        a_si_si = 20117023442.1621;  /* kcal/mol */
        b_si_o  = 0.20851;           /* Ang */
        b_o_o   = 0.35132;           /* Ang */
        b_si_si = 0.06570;           /* Ang */
        c_si_o  = 1631.17660000000;  /* Ang^6 kcal/mol */
        c_o_o   = 4951.93690000000;  /* Ang^6 kcal/mol */
        c_si_si = 537.312400000000;  /* Ang^6 kcal/mol */
        break;
      case FORCE_VDW_BUCKPRM_FB:
        printf("#   using Flikkema-Bromley parameterization for silica\n");
        a_si_o  = 241079.522480676;  /* kcal/mol */
        a_o_o   = 32939.8108424280;  /* kcal/mol */
        a_si_si = 1833361.49791679;  /* kcal/mol */
        b_si_o  = 0.208;             /* Ang */
        b_o_o   = 0.358;             /* Ang */
        b_si_si = 0.201;             /* Ang */
        c_si_o  = 1453.89773928600;  /* Ang^6 kcal/mol */
        c_o_o   = 954.106699212000;  /* Ang^6 kcal/mol */
        c_si_si = 10302.9871676400;  /* Ang^6 kcal/mol */
        break;
      default:
        ERROR("no parameterization indicated for silica");
        return FORCE_FAIL;
    }

    /* constructing the table */
    for (i = 0;  i < natomprms;  i++) {
      for (j = i;  j < natomprms;  j++) {
        const int is_i_si = (strcasecmp(atomprm[i].type, "si") == 0);
        const int is_i_o  = (strcasecmp(atomprm[i].type, "o")  == 0);
        const int is_j_si = (strcasecmp(atomprm[j].type, "si") == 0);
        const int is_j_o  = (strcasecmp(atomprm[j].type, "o")  == 0);

        ForceVdwparam *ij_entry = vdwparam + (i * natomprms + j);
        ForceVdwparam *ji_entry = vdwparam + (j * natomprms + i);

        if ((is_i_si || is_i_o) && (is_j_si || is_j_o)) {
          if (is_i_si && is_j_si) {
            ij_entry->a     = ji_entry->a     = a_si_si;
            ij_entry->a14   = ji_entry->a14   = a_si_si;
            ij_entry->rmin2 = ji_entry->rmin2 = b_si_si;
            ij_entry->b     = ji_entry->b     = c_si_si;
            ij_entry->b14   = ji_entry->b14   = c_si_si;
          }
          else if (is_i_o && is_j_o) {
            ij_entry->a     = ji_entry->a     = a_o_o;
            ij_entry->a14   = ji_entry->a14   = a_o_o;
            ij_entry->rmin2 = ji_entry->rmin2 = b_o_o;
            ij_entry->b     = ji_entry->b     = c_o_o;
            ij_entry->b14   = ji_entry->b14   = c_o_o;
          }
          else {
            ij_entry->a     = ji_entry->a     = a_si_o;
            ij_entry->a14   = ji_entry->a14   = a_si_o;
            ij_entry->rmin2 = ji_entry->rmin2 = b_si_o;
            ij_entry->b     = ji_entry->b     = c_si_o;
            ij_entry->b14   = ji_entry->b14   = c_si_o;
          }
        }
        else {
          printf("# WARNING:  no Buckingham parameterization for "
              "%s-%s, setting to zero\n", atomprm[i].type, atomprm[j].type);
          ij_entry->a     = ji_entry->a     = 0.0;
          ij_entry->a14   = ji_entry->a14   = 0.0;
          ij_entry->rmin2 = ji_entry->rmin2 = 1.0;
          ij_entry->b     = ji_entry->b     = 0.0;
          ij_entry->b14   = ji_entry->b14   = 0.0;
        }
      }
    }

    /*
     * Use safe Buckingham potential for energy minimization to
     * remove the unrealistic negative energy well when atoms are too close.
     */
    if ((FORCE_VDW_BUCKSAFE | FORCE_VDW_SWITCHBUCKSAFE) & vdwopts[n]) {
      ForceBucksafe *ij_entry, *ji_entry;

      /* Buckingham constants:  a exp(-r/b) - c/r^6  */
      double a, b, c;

      /* constants for switched extension:  a/r^6 + b  */
      double as, bs;    /* the "a" and "b" for above */
      double rs;        /* switching distance (an inner cutoff) */

      /* reported for diagnostic purposes */
      double urs;       /* U(rs) - energy at switch */
      double rmax;      /* distance at barrier height, Umax = U(rmax) */
      double urmax;     /* U(rmax) - energy at barrier height */

      for (i = 0;  i < natomprms;  i++) {
        for (j = i;  j < natomprms;  j++) {
          const int is_i_si = (strcasecmp(atomprm[i].type, "si") == 0);
          const int is_i_o  = (strcasecmp(atomprm[i].type, "o")  == 0);
          const int is_j_si = (strcasecmp(atomprm[j].type, "si") == 0);
          const int is_j_o  = (strcasecmp(atomprm[j].type, "o")  == 0);

          ij_entry = bucksafe + (i * natomprms + j);
          ji_entry = bucksafe + (j * natomprms + i);

          /* retrieve Buckingham parameters */
          a = vdwparam[i * natomprms + j].a;
          b = vdwparam[i * natomprms + j].rmin2;
          c = vdwparam[i * natomprms + j].b;

          if ((is_i_si || is_i_o) && (is_j_si || is_j_o)) {
            if (force_safe_buckingham_params(&as, &bs, &rs, &urs,
                  &rmax, &urmax, a, b, c)) {
              ERROR("force_safe_buckingham_params()");
              return FORCE_FAIL;
            }
            printf("# safe Buckingham extension for %s-%s interaction:\n",
                atomprm[i].type, atomprm[j].type);
            printf("#   A=%.12g  B=%.12g\n", as, bs);
            printf("# switch to extension:\n"
                "#   R_switch=%.12g Ang  U(R_switch)=%.12g kcal/mol\n",rs,urs);
            printf("# Buckingham energy barrier:\n"
                "#   R_max=%.12g Ang  U(R_max)=%.12g kcal/mol\n", rmax, urmax);
            ij_entry->a       = ji_entry->a       = as;
            ij_entry->b       = ji_entry->b       = bs;
            ij_entry->rinner2 = ji_entry->rinner2 = rs * rs;
          }
          else {
            printf("# WARNING:  no Buckingham parameterization for "
                "%s-%s, setting to zero\n", atomprm[i].type, atomprm[j].type);
            ij_entry->a       = ji_entry->a       = 0.0;
            ij_entry->b       = ji_entry->b       = 0.0;
            ij_entry->rinner2 = ji_entry->rinner2 = 0.0;
          }

          /* test join points */
          printf("### test join for safe extension:\n");
          for (k = -2;  k <= 2;  k++) {
            double u = 0, du_r = 0;
            double rinner2 = rs*rs;
            double r2 = rinner2 * (1.0 + k*DBL_EPSILON);

            if (r2 > 0.0) {
              if (FORCE_VDW_SWITCHBUCKSAFE & vdwopts[n]) {
                force_compute_nbpairs_vdw_switchbucksafe(&u, &du_r, r2, a, b, c,
                    rinner2, as, bs, cutoff2, switchdist2, denom);
              }
              else {
                force_compute_nbpairs_vdw_bucksafe(&u, &du_r, r2, a, b, c,
                    rinner2, as, bs);
              }
            }
            printf("r2=%.15e  u=%+.15e  du_r=%+.15e\n", r2, u, du_r);
          }

          if (FORCE_VDW_SWITCHBUCKSAFE & vdwopts[n]) {
            printf("### test switchdist:\n");
            for (k = -2;  k <= 2;  k++) {
              double u, du_r;
              double rinner2 = rs*rs;
              double r2 = switchdist2 * (1.0 + k*DBL_EPSILON);

              force_compute_nbpairs_vdw_switchbucksafe(&u, &du_r, r2, a, b, c,
                  rinner2, as, bs, cutoff2, switchdist2, denom);
              printf("r2=%.15e  u=%+.15e  du_r=%+.15e\n", r2, u, du_r);
            }
          }

          printf("### test cutoff:\n");
          for (k = -2;  k <= 2;  k++) {
            double u = 0, du_r = 0;
            double rinner2 = rs*rs;
            double r2 = cutoff2 * (1.0 + k*DBL_EPSILON);

            if (r2 < cutoff2) {
              if (FORCE_VDW_SWITCHBUCKSAFE & vdwopts[n]) {
                force_compute_nbpairs_vdw_switchbucksafe(&u, &du_r, r2, a, b, c,
                    rinner2, as, bs, cutoff2, switchdist2, denom);
              }
              else {
                force_compute_nbpairs_vdw_bucksafe(&u, &du_r, r2, a, b, c,
                    rinner2, as, bs);
              }
            }
            printf("r2=%.15e  u=%+.15e  du_r=%+.15e\n", r2, u, du_r);
          }

        }
      }

    } /* end BUCKSAFE parameterization */

  } /* end loop over vdwopts */

  return 0;
}
