/*
 * Copyright (C) 2005-2006 by David J. Hardy.  All rights reserved.
 */

/**@file    fselect.h
 * @brief   Force atom and bond selection.
 * @author  David J. Hardy
 * @date    2005-2006
 *
 * The @c ForceSelect_t class selects which atom and bond
 * terms to compute during force evaluation.  Use of this
 * class is optional.
 * The default selection (i.e. this class is @em not used)
 * is all atoms and bonds evaluated for force computation,
 * as designated by the @c ForceParam_t flags.
 */

#ifndef FSELECT_H
#define FSELECT_H

#include "mdapi/mdtypes.h"
#include "force/fparam.h"

#ifdef __cplusplus
extern "C" {
#endif


  /**@brief Atom and bond selection object.
   *
   * The bonded interaction arrays are given per bond, not per atom.
   * The @c aset_sel and @c bset_sel arrays are either both empty
   * (meaning pairwise interactions are computed normally) or
   * they must both be nonempty disjoint subsets of the total atoms.
   */
  typedef struct ForceSelect_t {
    int32 *bond_sel;
    /**< Select subset of bonds (indicated per bond, not per atom). */
    int32 *angle_sel;
    /**< Select subset of angles (indicated per angle, not per atom). */
    int32 *dihed_sel;
    /**< Select subset of dihedrals (indicated per dihedral, not per atom). */
    int32 *impr_sel;
    /**< Select subset of impropers (indicated per improper, not per atom). */
    int32 *aset_sel;
    /**< Select subset A of atoms for pairwise interactions. */
    int32 *bset_sel;
    /**< Select subset B of atoms for pairwise interactions. */
    int32 *bres_sel;
    /**< Select subset of atoms for boundary restraints. */

    int32 bond_sel_len;
    /**< Length of @c bond_sel. */
    int32 angle_sel_len;
    /**< Length of @c angle_sel. */
    int32 dihed_sel_len;
    /**< Length of @c dihed_sel. */
    int32 impr_sel_len;
    /**< Length of @c impr_sel. */
    int32 aset_sel_len;
    /**< Length of @c aset_sel. */
    int32 bset_sel_len;
    /**< Length of @c bset_sel. */
    int32 bres_sel_len;
    /**< Length of @c bres_sel. */

    int32 self_alloc_sel;
    /**< <i> (internal use) </i> Bit mask indicating which arrays are
     * allocated by constructor. */
  } ForceSelect;


  enum ForceSelectFlags_t {
    FORCE_SELECT_ALL = -1,
    /**< Length value to use with @c NULL to generate array containing all. */
    FORCE_SELECT_NONE = 0,
    /**< Length value to use with @c NULL to select zero. */
    FORCE_SELECT_BOND  = 0x001,
    /**< <i> (internal use) </i> Self-allocated @c ForceSelect::bond_sel. */
    FORCE_SELECT_ANGLE = 0x002,
    /**< <i> (internal use) </i> Self-allocated @c ForceSelect::angle_sel. */
    FORCE_SELECT_DIHED = 0x004,
    /**< <i> (internal use) </i> Self-allocated @c ForceSelect::dihed_sel. */
    FORCE_SELECT_IMPR  = 0x008,
    /**< <i> (internal use) </i> Self-allocated @c ForceSelect::impr_sel. */
    FORCE_SELECT_ASET  = 0x010,
    /**< <i> (internal use) </i> Self-allocated @c ForceSelect::aset_sel. */
    FORCE_SELECT_BSET  = 0x020,
    /**< <i> (internal use) </i> Self-allocated @c ForceSelect::bset_sel. */
    FORCE_SELECT_BRES  = 0x040,
    /**< <i> (internal use) </i> Self-allocated @c ForceSelect::bres_sel. */
    FORCE_SELECT_MASK  = 0x07f
    /**< <i> (internal use) </i> Mask for allocation flags. */
  };


  /**@brief Constructor.
   *
   * @param[in] fprm           @c ForceParam_t object used for configuration.
   * @param[in] bond_sel       Selects terms for spring bonds.
   * @param[in] bond_sel_len   Length of @c bond_sel array.
   * @param[in] angle_sel      Selects terms for angles.
   * @param[in] angle_sel_len  Length of @c angle_sel array.
   * @param[in] dihed_sel      Selects terms for dihedrals.
   * @param[in] dihed_sel_len  Length of @c dihed_sel array.
   * @param[in] impr_sel       Selects terms for impropers.
   * @param[in] impr_sel_len   Length of @c impr_sel array.
   * @param[in] aset_sel   Selects subset A of atoms for pairwise interactions.
   * @param[in] aset_sel_len   Length of @c aset_sel array.
   * @param[in] bset_sel   Selects subset B of atoms for pairwise interactions.
   * @param[in] bset_sel_len   Length of @c bset_sel array.
   * @param[in] bres_sel       Selects atoms for boundary restraints.
   * @param[in] bres_sel_len   Length of @c bres_sel array.
   *
   * Creates dynamically allocated @c ForceSelect_t object
   * used to select atom and bond terms for computation.
   * Each integer array provides a set of indices into a corresponding
   * array of bonds for bonded interactions or of atoms for pairwise
   * interactions and boundary restraints.  These arrays must persist
   * until the @c ForceSelect_t object is destroyed.
   * Choose the entire set of indices by passing
   * @c NULL with @c FORCE_SELECT_ALL for array and length, respectively,
   * or choose no indices by passing
   * @c NULL with @c FORCE_SELECT_NONE.
   *
   * The atom selection for pairwise interactions allows the
   * computation of forces and potentials between two nonempty, disjoint
   * subsets of atoms.  This computes the forces and potentials of
   * a subset A of atoms acting on a subset B of atoms (and vice versa).
   * This computation is enabled by passing the first set of
   * atom selection indices as @c aset_sel with length @c aset_sel_len
   * and the second as @c bset_sel with length @c bset_sel_len.
   * Either subset of atoms selection indices can be given by
   * @c NULL with length @c FORCE_SELECT_ALL to select all remaining
   * atoms, but the other subset must be explicitly designated.
   * The particular pairwise potentials employed and the method of
   * computation is determined by setting @c FORCE_ELEC and/or @c FORCE_VDW
   * flags for ForceParam_t::forcetypes along with their respective options
   * given by @c ForceParam_t::elecopts and @c ForceParam_t::vdwopts.
   * Normal computation of pairwise interactions is accomplished by setting
   * both subsets of atom indices to @c NULL and @c FORCE_SELECT_NONE.
   *
   * The determination of which force and energy terms are computed
   * is specified by @c ForceParam_t::forcetypes.  Separate storage of
   * the different types of forces and potentials is specified by the
   * @c ForceResult object.
   *
   * @return Pointer to @c ForceSelect_t object or @c NULL
   * if memory allocation fails or if arguments are inconsistent.
   */
  ForceSelect *force_select_create(ForceParam *fprm,
      int32 *bond_sel, int32 bond_sel_len,
      int32 *angle_sel, int32 angle_sel_len,
      int32 *dihed_sel, int32 dihed_sel_len,
      int32 *impr_sel, int32 impr_sel_len,
      int32 *aset_sel, int32 aset_sel_len,
      int32 *bset_sel, int32 bset_sel_len,
      int32 *bres_sel, int32 bres_sel_len);

  /**@brief Destructor.
   *
   * Clears memory and destroys the dynamically allocated @c ForceSelect_t
   * object.  Call when finished using the object.
   */
  void force_select_destroy(ForceSelect *);

  /**@brief Alternative constructor.
   *
   * Use this to construct a preallocated @c ForceSelect_t object.
   * Establishes user-provided arrays and allocates other arrays as needed.
   * Same notes apply as for @c force_select_create().
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_select_initialize(ForceSelect *, ForceParam *fprm,
      int32 *bond_sel, int32 bond_sel_len,
      int32 *angle_sel, int32 angle_sel_len,
      int32 *dihed_sel, int32 dihed_sel_len,
      int32 *impr_sel, int32 impr_sel_len,
      int32 *aset_sel, int32 aset_sel_len,
      int32 *bset_sel, int32 bset_sel_len,
      int32 *bres_sel, int32 bres_sel_len);

  /**@brief Alternative destructor.
   *
   * Use this to destroy a preallocated @c ForceSelect_t object.
   * Frees any self-allocated arrays and resets the object memory.
   */
  void force_select_cleanup(ForceSelect *);

  /**@brief Reset selected components of a force array (type @c MD_Dvec). */
  void force_select_reset_force(MD_Dvec f[], const int32 sel[], int32 len);

  /**@brief Reset selected components of a potential array (type @c double). */
  void force_select_reset_potential(double e[], const int32 sel[], int32 len);

#ifdef __cplusplus
}
#endif

#endif /* FDOMAIN_H */
