/*
 * Copyright (C) 2005-2006 by David J. Hardy.  All rights reserved.
 */

#include <stdlib.h>
#include <string.h>
#include "force/force.h"
#include "debug/debug.h"


ForceResult *force_result_create(const ForceParam *fprm, int32 alloc_flags)
{
  ForceResult *fres;

  fres = (ForceResult *) malloc(sizeof(ForceResult));
  if (fres == NULL) {
    return NULL;
  }
  else if (force_result_initialize(fres, fprm, alloc_flags)) {
    force_result_cleanup(fres);
    free(fres);
    return NULL;
  }
  return fres;
}


int force_result_initialize(ForceResult *fres, const ForceParam *fprm,
    int32 alloc_flags)
{
  int32 natoms;

  ASSERT(fres != NULL);
  ASSERT(fprm != NULL);

  /* clear memory */
  memset(fres, 0, sizeof(ForceResult));
  natoms = fprm->atom_len;

  if (fprm->atom_len != 0) {
    fres->f_total = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_total == NULL) return FORCE_FAIL;
  }

  if ((alloc_flags & FORCE_RESULT_F_BOND) && natoms != 0) {
    fres->f_bond = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_bond == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_F_BOND;
  }

  if ((alloc_flags & FORCE_RESULT_F_ANGLE) && natoms != 0) {
    fres->f_angle = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_angle == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_F_ANGLE;
  }

  if ((alloc_flags & FORCE_RESULT_F_DIHED) && natoms != 0) {
    fres->f_dihed = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_dihed == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_F_DIHED;
  }

  if ((alloc_flags & FORCE_RESULT_F_IMPR) && natoms != 0) {
    fres->f_impr = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_impr == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_F_IMPR;
  }

  if ((alloc_flags & FORCE_RESULT_F_ELEC) && natoms != 0) {
    fres->f_elec = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_elec == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_F_ELEC;
  }

  if ((alloc_flags & FORCE_RESULT_F_VDW) && natoms != 0) {
    fres->f_vdw = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_vdw == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_F_VDW;
  }

  if ((alloc_flags & FORCE_RESULT_F_BRES) && natoms != 0) {
    fres->f_bres = (MD_Dvec *) calloc(natoms, sizeof(MD_Dvec));
    if (fres->f_bres == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_F_BRES;
  }

  if ((alloc_flags & FORCE_RESULT_E_BOND) && fprm->bond_len != 0) {
    fres->e_bond = (double *) calloc(fprm->bond_len, sizeof(double));
    if (fres->e_bond == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_BOND;
  }

  if ((alloc_flags & FORCE_RESULT_E_ANGLE) && fprm->angle_len != 0) {
    fres->e_angle = (double *) calloc(fprm->angle_len, sizeof(double));
    if (fres->e_angle == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_ANGLE;
  }

  if ((alloc_flags & FORCE_RESULT_E_DIHED) && fprm->dihed_len != 0) {
    fres->e_dihed = (double *) calloc(fprm->dihed_len, sizeof(double));
    if (fres->e_dihed == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_DIHED;
  }

  if ((alloc_flags & FORCE_RESULT_E_IMPR) && fprm->impr_len != 0) {
    fres->e_impr = (double *) calloc(fprm->impr_len, sizeof(double));
    if (fres->e_impr == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_IMPR;
  }

  if ((alloc_flags & FORCE_RESULT_E_ELEC) && natoms != 0) {
    fres->e_elec = (double *) calloc(natoms, sizeof(double));
    if (fres->e_elec == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_ELEC;
  }

  if ((alloc_flags & FORCE_RESULT_E_VDW) && natoms != 0) {
    fres->e_vdw = (double *) calloc(natoms, sizeof(double));
    if (fres->e_vdw == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_VDW;
  }

  if ((alloc_flags & FORCE_RESULT_E_BRES) && natoms != 0) {
    fres->e_bres = (double *) calloc(natoms, sizeof(double));
    if (fres->e_bres == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_BRES;
  }

  if ((alloc_flags & FORCE_RESULT_E_EPOT) && natoms != 0) {
    fres->e_epot = (double *) calloc(natoms, sizeof(double));
    if (fres->e_epot == NULL) return FORCE_FAIL;
    fres->self_alloc |= FORCE_RESULT_E_EPOT;
  }

  return 0;
}


void force_result_destroy(ForceResult *fres)
{
  force_result_cleanup(fres);
  free(fres);
}


void force_result_cleanup(ForceResult *fres)
{
  /* free allocated array buffer space */
  free(fres->f_total);
  if (fres->self_alloc & FORCE_RESULT_F_BOND) free(fres->f_bond);
  if (fres->self_alloc & FORCE_RESULT_F_ANGLE) free(fres->f_angle);
  if (fres->self_alloc & FORCE_RESULT_F_DIHED) free(fres->f_dihed);
  if (fres->self_alloc & FORCE_RESULT_F_IMPR) free(fres->f_impr);
  if (fres->self_alloc & FORCE_RESULT_F_ELEC) free(fres->f_elec);
  if (fres->self_alloc & FORCE_RESULT_F_VDW) free(fres->f_vdw);
  if (fres->self_alloc & FORCE_RESULT_F_BRES) free(fres->f_bres);
  if (fres->self_alloc & FORCE_RESULT_E_BOND) free(fres->e_bond);
  if (fres->self_alloc & FORCE_RESULT_E_ANGLE) free(fres->e_angle);
  if (fres->self_alloc & FORCE_RESULT_E_DIHED) free(fres->e_dihed);
  if (fres->self_alloc & FORCE_RESULT_E_IMPR) free(fres->e_impr);
  if (fres->self_alloc & FORCE_RESULT_E_ELEC) free(fres->e_elec);
  if (fres->self_alloc & FORCE_RESULT_E_VDW) free(fres->e_vdw);
  if (fres->self_alloc & FORCE_RESULT_E_BRES) free(fres->e_bres);
  if (fres->self_alloc & FORCE_RESULT_E_EPOT) free(fres->e_epot);

  /* reset contents */
  memset(fres, 0, sizeof(ForceResult));
}
