/*
 * Copyright (C) 2004-2006 by David J. Hardy.  All rights reserved.
 */

/**@file    fparam.h
 * @brief   Force field parameters container class.
 * @author  David J. Hardy
 * @date    2004-2006
 *
 * The @c ForceParam_t container class specifies the force field
 * parameters that define how the force will be computed.
 * It is used to construct the @c Force_t object.
 */

#ifndef FPARAM_H
#define FPARAM_H

#include <float.h>
#include "mdapi/mdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief Used to configure @c Force_t class.
   *
   * User sets the internal fields of this class in order to configure
   * the evaluation performed by the @c Force_t class.
   * The contents of the @c ForceParam_t container are intended
   * to be setup once before constructing the @c Force_t object,
   * then treated as read-only data.
   *
   * The user provides and retains ownership of memory buffer space for all
   * arrays.  The force field parameter and topology arrays all follow the
   * MDAPI guidelines as defined by @c mdtypes.h file.  In particular,
   * it is expected that the proper cross-indexing between topology and
   * force field parameter arrays have been established.  The best way to
   * do this is to use the MDIO library for reading the relevant force
   * field parameter and topology files.
   *
   * The parameter @c ForceParam_t::atom_len indicates the number of atoms
   * in the system, providing the expected array length of the
   * @c ForceResult_t force arrays and the @c pos array passed to
   * @c force_create() and @c force_compute().
   *
   * The boundary restraint and nonbonded parameters follow the NAMD
   * conventions, as described in its documentation.
   *
   * The field @c ForceParam_t::forcetype specifies what parts of
   * the force field are to be evaluated.
   * It is assigned a bitwise ORed collection of @c ForceType_t constants.
   *
   * The remaining fields @c ForceParam_t::excltype,
   * @c ForceParam_t::electype, @c ForceParam_t::vdwtype,
   * and @c ForceParam_t::brestype
   * are each assigned a single value from respective enumerated types
   * @c ForceExcltype_t, @c ForceElectype_t, @c ForceVdwtype_t,
   * and @c ForceBrestype_t.
   * These, respectively, indicate which type of interactions
   * are to be excluded, the type and smoothing of electrostatic
   * interactions, the type and switching of van der Waals interactions,
   * and the type of boundary restraints.
   */
  typedef struct ForceParam_t {

    /* Force field parameters as defined by the MDAPI mdtypes.h file. */
    MD_AtomPrm *atomprm;   /**< Array of atom parameters. */
    MD_BondPrm *bondprm;   /**< Array of bond parameters. */
    MD_AnglePrm *angleprm; /**< Array of angle parameters. */
    MD_TorsPrm *dihedprm;  /**< Array of dihedral parameters. */
    MD_TorsPrm *imprprm;   /**< Array of improper parameters. */
    MD_NbfixPrm *nbfixprm; /**< Array of nonbonded parameters to override
                             the combined pairs of atom parameters. */
    int32 atomprm_len;     /**< Length of atom parameters array. */
    int32 bondprm_len;     /**< Length of bond parameters array. */
    int32 angleprm_len;    /**< Length of angle parameters array. */
    int32 dihedprm_len;    /**< Length of dihedral parameters array. */
    int32 imprprm_len;     /**< Length of improper parameters array. */
    int32 nbfixprm_len;    /**< Length of nonbonded parameters array. */

    /* Topology parameters as defined by the MDAPI mdtypes.h file. */
    MD_Atom *atom;         /**< Array of atoms (contains mass and charge). */
    MD_Bond *bond;         /**< Array of bonds. */
    MD_Angle *angle;       /**< Array of angles. */
    MD_Tors *dihed;        /**< Array of dihedrals. */
    MD_Tors *impr;         /**< Array of impropers. */
    MD_Excl *excl;         /**< Array of nonbonded exclusions. */
    int32 atom_len;   /**< Length of atom array (number of atoms in system). */
    int32 bond_len;        /**< Length of bond array. */
    int32 angle_len;       /**< Length of angle array. */
    int32 dihed_len;       /**< Length of dihedral array. */
    int32 impr_len;        /**< Length of improper array. */
    int32 excl_len;        /**< Length of nonbonded exclusion array. */

    /* parameters for pairwise potentials */
    double cutoff;         /**< Cutoff distance used to define both
                             electrostatics and van der Waals interactions. */
    double elec_cutoff;    /**< Cutoff distance for electrostatics. */
    double vdw_cutoff;     /**< Cutoff distance for van der Waals. */
    double switchdist;     /**< Switching distance for van der Waals
                             (needs to be strictly less than the cutoff). */
    double elec_const;     /**< Constant for electrostatic interactions,
                             typically should be defined using @c MD_COULOMB
                             from MDAPI @c mdtypes.h file. */
    double dielectric;     /**< Dielectric constant, typically set to 1.0. */
    double scaling14;      /**< Extra multiplicative constant for
                             electrostatic interactions between 1-4 pairs,
                             applied only if @c FORCE_EXCL_SCAL14 flag is
                             set, typically set to 1.0. */
    double ewald_coef;     /**< Ewald coefficient for real space part. */

    /* boundary restraints defined around center */
    double radius1;        /**< Distance at which first potential boundary
                             takes effect, measured as radius from center. */
    double radius2;        /**< Distance at which second potential boundary
                             takes effect, measured as radius from center. */
    double length1;        /**< For cylindrical restraints: distance at
                             which first potential takes effect along
                             cylinder axis. */
    double length2;        /**< For cylindrical restraints: distance at
                             which second potential takes effect along
                             cylinder axis. */
    double konst1;         /**< Constant for first harmonic potential. */
    double konst2;         /**< Constant for second harmonic potential. */
    int32 exp1;            /**< Exponent for first harmonic potential.
                             Should be an even positive integer. */
    int32 exp2;            /**< Exponent for second harmonic potential.
                             Should be an even positive integer. */

    /* options */
    int32 forcetypes;      /**< Indicates types of forces to be computed. */
    int32 exclpolicy;      /**< Gives nonbonded exclusion policy. */
    int32 elecopts;        /**< Options for electrostatic forces. */
    int32 vdwopts;         /**< Options for van der Waals forces. */
    int32 bresopts;        /**< Options for boundary restraints,
                             generally intended for nonperiodic boundaries. */

    /* algorithmic parameters */
    int32 shell_width;     /**< Thickness of shell of grid cells to evaluate
                             (must be at least 1, max and default is 3). */
    double cutoff_ratio;   /**< Make minimum cell size at least this
                             fraction of cutoff (typically 1/shell_width). */
    double delta_dis;      /**< Maximum allowable distance any atom can
                             travel before pairlists must be regenerated
                             (default is 1.5 Angstroms). */
    double cell_margin;    /**< Add to minimum cell size. */

  } ForceParam;


  /**@brief Types used for setting @c ForceParam_t::forcetype.
   * The individual values are to be bitwise ORed together to
   * select the types of forces to be computed.
   */
  enum ForceTypes_t {
    FORCE_NONE        = 0x0000,
    /**< <b> (The following flags for @c ForceParam_t::forcetypes
     * should be bitwise ORed together.) </b> @n
     * <i> (no forces) </i>
     */
    FORCE_BOND        = 0x0001,
    /**< Indicates evaluation of spring bond interactions. */
    FORCE_ANGLE       = 0x0002,
    /**< Indicates evaluation of angle interactions. */
    FORCE_DIHED       = 0x0004,
    /**< Indicates evaluation of dihedral interactions. */
    FORCE_IMPR        = 0x0008,
    /**< Indicates evaluation of improper interactions. */
    FORCE_ELEC        = 0x0010,
    /**< Indicates evaluation of electrostatic interactions. */
    FORCE_VDW         = 0x0020,
    /**< Indicates evaluation of van der Waals interactions. */
    FORCE_BRES        = 0x0040,
    /**< Indicates evaluation of boundary restraint forces. */
    FORCE_BONDED      = FORCE_BOND | FORCE_ANGLE | FORCE_DIHED | FORCE_IMPR,
    /**< Collection of flags: evaluate all types of bonded interactions. */
    FORCE_PAIRWISE    = FORCE_ELEC | FORCE_VDW,
    /**< Collection of flags: evaluate all types of pairwise interactions. */
    FORCE_NONBONDED   = FORCE_PAIRWISE | FORCE_BRES,
    /**< Collection of flags: evaluate all types of nonbonded interactions. */
    FORCE_ALL         = FORCE_BONDED | FORCE_NONBONDED
    /**< Collection of flags: evaluate all types of bonded and nonbonded
     * interactions. */
  };


  /*@brief Nonbonded exclusion policy, mutually exclusive. */
  enum ForceExcl_t {
    FORCE_EXCL_NONE   = 0x001,
    /**< <b> (The following options for @c ForceParam_t::excltype
     * are mutually exclusive.) </b> @n
     * Do not exclude any nonbonded pairs except for those indicated
     * explicitly within the @c ForceParam_t::excl array. */
    FORCE_EXCL_12,
    /**< Exclude 1-2 interactions (atoms joined by spring bond). */
    FORCE_EXCL_13,
    /**< Exclude 1-3 interactions (atoms joined by pair of spring bonds),
     * which includes all angles. */
    FORCE_EXCL_14,
    /**< Exclude 1-4 interactions (atoms joined by trio of spring bonds),
     * which includes all dihedrals and impropers. */
    FORCE_EXCL_SCAL14,
    /**< Exclude 1-3 interactions but redefine 1-4 interactions using
     * alternate van der Waals parameters and electrostatic 1-4 scaling
     * constant. @n
     * <i> (This is the commonly used standard for modern systems.) </i> */
    FORCE_MARKER_EXCL
    /**< <i> (internal use) </i>
     * Marks end of exclusion policy options. */
  };


  /*@brief Electrostatics options, choose both the method and potential. */
  enum ForceElec_t {

  /* methods */
    FORCE_ELEC_DIRECT    = 0x001,
    /**< Perform direct evaluation of electrostatic interactions. @n
     * <i> (Should be used only on small systems or for testing.) </i> */
    FORCE_ELEC_GRIDCELLS = 0x002,
    /**< Pairs interact only within cutoff distance. */
    FORCE_ELEC_PAIRLISTS = 0x004,
    /**< Use pairlists to evaluate cutoff pairwise potentials. */
    FORCE_ELEC_STNDEXCL  = 0x008,
    /**< Evaluate exclusions using standard potential, even if this differs
     * from the chosen potential.  (This is needed for use with long-range
     * electrostatic solvers, where the short-range part uses a special pair
     * potential, but it is still necessary to subtract exact exclusions.)
     * Note that this can be chosen with another method. */

  /* pair potentials */
    FORCE_ELEC_STANDARD  = 0x010,
    /**< The standard 1/r electrostatic potential. */
    FORCE_ELEC_SHIFTED   = 0x020,
    /**< Multiply 1/r potential by shifting function for smooth cutoff. @n
     * <i> (This is commonly used for cutoff simulations.) </i> */
    FORCE_ELEC_EWALD     = 0x040,
    /**< Compute Ewald real space potential. @n
     * <i> (Use this for Ewald or PME simulations.) </i> */
    FORCE_ELEC_GLASS     = 0x080,

  /* bit masks */
    FORCE_MASK_ELEC_METHOD    = 0x00f,
    /**< <i> (internal use) </i> Mask for electrostatic method. */
    FORCE_MASK_ELEC_POTENTIAL = 0x0f0,
    /**< <i> (internal use) </i> Mask for electrostatic potential. */

  /* typical values for elecopts */
    FORCE_ELECOPTS_DIRECT = FORCE_ELEC_DIRECT | FORCE_ELEC_STANDARD,
    /**< Typical settings for direct evaluation of electrostatics. */
    FORCE_ELECOPTS_CUTOFF = FORCE_ELEC_GRIDCELLS | FORCE_ELEC_SHIFTED,
    /**< Typical settings for cutoff evaluation of electrostatics. */
    FORCE_ELECOPTS_LONGRNG = FORCE_ELEC_STNDEXCL
    /**< Typical settings for subtracting electrostatic exclusions only
     * and evaluating full electrostatics using an external solver. */
  };


  /*@brief Van der Waals options, choose both the method and potential. */
  enum ForceVdw_t {

  /* methods */
    FORCE_VDW_DIRECT     = 0x0001,
    /**< Perform direct evaluation of van der Waals interactions. @n
     * <i> (Should be used only on small systems or for testing.) </i> */
    FORCE_VDW_GRIDCELLS  = 0x0002,
    /**< Pairs interact only within cutoff distance. */
    FORCE_VDW_PAIRLISTS = 0x004,
    /**< Use pairlists to evaluate cutoff pairwise potentials. */
    FORCE_VDW_STNDEXCL   = 0x0008,
    /**< Evaluate exclusions using standard potential, even if this differs
     * from the chosen potential.  (This is needed for use with long-range
     * van der Waals solvers, where the short-range part uses a special pair
     * potential, but it is still necessary to subtract exact exclusions.)
     * Note that this can be chosen with another method. */

  /* pair potentials */
    FORCE_VDW_STANDARD   = 0x0010,
    /**< The standard van der Waals potential. */
    FORCE_VDW_SWITCHED   = 0x0020,
    /**< Apply switching function for continuous force. @n
     * <i> (This is commonly used for cutoff simulations.) </i> */
    FORCE_VDW_BUCK       = 0x0040,
    /**< Use Buckingham potential for silica (instead of van der Waals). */
    FORCE_VDW_SWITCHBUCK = 0x0080,
    /**< Use Buckingham potential for silica along with switching function
     * for continuous force (instead of van der Waals). */
    FORCE_VDW_BUCKND     = 0x0100,
    /**< Use Buckingham without dispersion term potential
     * for silica (instead of van der Waals). */
    FORCE_VDW_SWITCHBUCKND=0x0200,
    /**< Use Buckingham without dispersion term potential
     * for silica along with switching function
     * for continuous force (instead of van der Waals). */
    FORCE_VDW_BUCKSAFE    =0x0400,
    /**< Use this instead of @c FORCE_VDW_BUCK for energy minimization,
     * as it gets rid of the attractive well. */
    FORCE_VDW_SWITCHBUCKSAFE=0x0800,
    /**< Use this instead of @c FORCE_VDW_SWITCHBUCK for energy minimization,
     * as it gets rid of the attractive well. */
    FORCE_VDW_GLASS        =0x1000,

  /* select Buckingham parameterization */
    FORCE_VDW_BUCKPRM_BKS  =0x10000,
    /**< Select BKS parameterization for silica. */
    FORCE_VDW_BUCKPRM_TTAM =0x20000,
    /**< Select TTAM parameterization for silica. */
    FORCE_VDW_BUCKPRM_FB   =0x40000,
    /**< Select FB parameterization for silica. */
    FORCE_VDW_GLASSPRM     =0x80000,

  /* bit masks */
    FORCE_MASK_VDW_METHOD     = 0x000f,
    /**< <i> (internal use) </i> Mask for van der Waals method. */
    FORCE_MASK_VDW_POTENTIAL  = 0x1ff0,
    /**< <i> (internal use) </i> Mask for van der Waals potential. */
    FORCE_MASK_VDW_BUCK       = 0x1fc0,
    /**< <i> (internal use) </i> Mask to check for Buckingham potential. */
    FORCE_MASK_VDW_BUCKPRM    = 0xf0000,
    /**< <i> (internal use) </i> Mask to check for Buckingham potential
     * parameterization. */

  /* typical values for vdwopts */
    FORCE_VDWOPTS_DIRECT = FORCE_VDW_DIRECT | FORCE_VDW_STANDARD,
    /**< Typical settings for direct evaluation of van der Waals. */
    FORCE_VDWOPTS_CUTOFF = FORCE_VDW_GRIDCELLS | FORCE_VDW_SWITCHED,
    /**< Typical settings for cutoff evaluation of van der Waals. */
    FORCE_VDWOPTS_LONGRNG = FORCE_VDW_STNDEXCL
    /**< Typical settings for subtracting van der Waals exclusions only
     * and evaluating full van der Waals using an external solver. */
  };


  /*@brief Boundary restraints, mutually exclusive
   * (generally intended for nonperiodic systems). */
  enum ForceBrestype_t {
    FORCE_BRES_SPHERE = 0x001,
    /**< Use spherical boundary restraints. */
    FORCE_BRES_X_CYLINDER,
    /**< Use cylindrical boundary restraints with axis aligned in
     * x-direction. */
    FORCE_BRES_Y_CYLINDER,
    /**< Use cylindrical boundary restraints with axis aligned in
     * y-direction. */
    FORCE_BRES_Z_CYLINDER,
    /**< Use cylindrical boundary restraints with axis aligned in
     * z-direction. */
    FORCE_MARKER_BRES
    /**< <i> (internal use) </i>
     * Marks end of boundary restraint options. */
  };


  /*@brief Algorithmic parameter defaults.
   *
   * With all parameters present, the extended cutoff should be @n
   *
   *   cutoff + 2*delta_dis <=
   *       shell_width * ( cutoff_ratio * (cutoff + 2*delta_dis) + cell_margin)
   *
   * For grid cell hashing exclusively, set delta_dis = cell_margin = 0.
   * For pair lists generated using grid cells, delta_dis controls how
   * often the pair lists need to be generated.
   *
   * cutoff_ratio tunes the coverage of the (extended) cutoff across
   * the shell width, and cell_margin can be adjusted to make cells a
   * little bigger (so hashing more atoms per cell) independently of
   * the size of the cutoff.
   */
#define FORCE_DEFAULT_SHELL_WIDTH          3
#define FORCE_DEFAULT_CUTOFF_RATIO         (1./FORCE_DEFAULT_SHELL_WIDTH)
#define FORCE_DEFAULT_CELL_MARGIN          FLT_EPSILON
#define FORCE_DEFAULT_PAIRLIST_DELTA_DIS   0.75
#define FORCE_DEFAULT_GRIDCELL_DELTA_DIS   0.0


  /* force param constructor and destructor */

  /**@brief Constructor.
   *
   * Creates dynamically allocated @c ForceParam_t force field parameter
   * container object used to configure the @c Force_t class.
   * The force field parameters memory is cleared.
   *
   * @return Pointer to @c ForceParam_t object or @c NULL
   * if memory allocation fails.
   */
  ForceParam *force_param_create(void);

  /**@brief Destructor.
   *
   * Clears memory and destroys the dynamically allocated @c ForceParam_t
   * object.  Call when finished using @c ForceParam_t object.
   */
  void force_param_destroy(ForceParam *);


  /**@brief Alternative constructor.
   *
   * Use this to construct a preallocated @c ForceParam_t object.
   * Clears the force field parameter memory.
   *
   * @return 0 for success (does not fail)
   */
  int force_param_initialize(ForceParam *);

  /**@brief Alternative destructor.
   *
   * Use this to destroy a preallocated @c ForceParam_t object.
   * Resets the memory of the object.
   */
  void force_param_cleanup(ForceParam *);


#ifdef __cplusplus
}
#endif

#endif /* FPARAM_H */
