/*
 * Copyright (C) 2006 by David J. Hardy.  All rights reserved.
 */

/**@file    compute.h
 * @brief   Lower-level routines for force computation.
 * @author  David J. Hardy
 * @date    2006
 *
 * The high-level @c force_compute() routine is the manager that
 * calls the lower-level routines prototyped in this file.
 * The various force computation routines are called depending on
 * the force types indicated in @c ForceParam_t::forcetypes.
 * The various array and atom/bond selection parameters are determined
 * by the ForceResult_t object passed to @c force_compute() and the
 * ForceSelect_t object provided to the @c force_create() constructor.
 *
 * The medium-level routines:
 * @li @c force_compute_scaled_coords()
 * @li @c force_compute_domain_update()
 * @li @c force_compute_bonds()
 * @li @c force_compute_angles()
 * @li @c force_compute_dihedrals()
 * @li @c force_compute_impropers()
 * @li @c force_compute_bres_sphere()
 * @li @c force_compute_bres_cylinder()
 * @li @c force_compute_nbpairs_direct()
 * @li @c force_compute_nbpairs_geomhash()
 * @li @c force_compute_nbpairs_gridcells()
 * @li @c force_compute_nbpairs_subtexcl()
 *
 * can all still be viewed as methods of class @c Force_t, but with
 * all of the array buffer space and data (from @c ForceResult_t and
 * @c ForceSelect_t objects) explicitly passed as parameters along
 * with any "control" data originally specified as part of
 * @c ForceParam_t::forcetypes, etc.
 *
 * The low-level routines:
 * @li @c force_compute_bond_interaction()
 * @li @c force_compute_angle_interaction()
 * @li @c force_compute_torsion_interaction()
 * @li @c force_compute_nbpairs_elec_standard()
 * @li @c force_compute_nbpairs_elec_shifted()
 * @li @c force_compute_nbpairs_vdw_standard()
 * @li @c force_compute_nbpairs_vdw_switched()
 *
 * compute various types of individual interactions independently
 * of the force and helper classes.
 */

#ifndef COMPUTE_H
#define COMPUTE_H

#ifdef __cplusplus
extern "C" {
#endif

  /**@brief Compute scaled coordinates.
   *
   * @param[in]  pos           Atomic position coordinates.
   * @param[in]  sel           Selects atoms from @c fobj->param->atoms array.
   * @param[in]  sel_len       Length of atom selection array.
   *
   * Compute the scaled atomic coordinates (@c Force_t::trpos) and position
   * wrapping array (@c Force_t::poswrap) for periodic boundaries.  These
   * arrays are held internally by the force object.  For periodic boundaries,
   * it is assumed that the pos[i]+poswrap[i] result from previous computation
   * is no more than one cell image away.  This is ensured by monitoring atom
   * speeds so that an atom travels less than one cell width in a time step.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_scaled_coords(Force *, const MD_Dvec pos[],
      const int32 sel[], int32 sel_len);


  /**@brief Update nonperiodic domain specification.
   *
   * @param[in]  sel           Selects atoms from @c fobj->param->atoms array.
   * @param[in]  sel_len       Length of atom selection array.
   *
   * This routine is called internally to update the nonperiodic domain
   * specification.  The cell basis vectors for a nonperiodic domain are
   * chosen to map most of the atoms into the unit cube (but the choice
   * of these vectors is arbitrary).  During the simulation, the atoms
   * might drift away from the initial domain specification, so this
   * routine is called every @c Force_t::max_steps time intervals.
   * If the center (average position coordinates) has drifted outside
   * of the domain, a new domain specification needs to be determined.
   *
   * If the user previously called @c force_setup_lattice() to fix
   * a lattice in space, then this routine chooses the new center to be
   * some multiple of the lattice spacing away from the old center,
   * and does not rescale the domain basis vector lengths, and so the
   * location of the points of the (infinite) lattice is preserved
   * in space.
   *
   * @return 0 if no change is to be made (the average over all positions
   * is not sufficiently far away from the current center) or if a fixed
   * lattice has been defined (in which case the domain center is updated
   * to modify the translation vector, but the transformation to the unit
   * cube remains unchanged); otherwise, the domain center is updated and 
   * 1 is returned to indicate that @c force_setup_domain() needs to be
   * called at the beginning of the next @c force_compute().
   */
  int force_compute_domain_update(Force *, const int32 sel[], int32 sel_len);


  /**@brief Compute contribution of bond springs to force and potential.
   *
   * @param[out] u_bond        Receives total potential energy.
   * @param[in,out] f_bond     Receives forces on atoms.
   * @param[out] e_bond        Receives interaction potentials.
   * @param[in,out] virial     Accumulates contribution to virial.
   * @param[in]  pos           Atomic position coordinates.
   * @param[in]  bond_sel      Selects bonds from @c fobj->param->bond array.
   * @param[in]  bond_sel_len  Length of bond selection array.
   *
   * Compute force and potentials due to bond springs.  All parameters must
   * exist: array buffers must all be allocated and be of expected length and
   * data arrays must contain the expected information.  The @c bond_sel
   * array contains integer indices into the @c fobj->param->bond array.
   * The output arrays are expected to be in a sensible state upon calling
   * (i.e. these arrays are not zeroed).  In particular, the forces are
   * accumulated into @c f_bond.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_bonds(Force *fobj, double *u_bond, MD_Dvec f_bond[],
      double e_bond[], double virial[], const MD_Dvec pos[],
      const int32 bond_sel[], int32 bond_sel_len);

  /**@brief Compute contribution of bond angles to force and potential.
   *
   * @param[out] u_angle        Receives total potential energy.
   * @param[in,out] f_angle     Receives forces on atoms.
   * @param[out] e_angle        Receives interaction potentials.
   * @param[in,out] virial      Accumulates contribution to virial.
   * @param[in]  pos            Atomic position coordinates.
   * @param[in]  angle_sel      Selects angles from @c fobj->param->angle array.
   * @param[in]  angle_sel_len  Length of angle selection array.
   *
   * Compute force and potentials due to bond angles.  All parameters must
   * exist: array buffers must all be allocated and be of expected length and
   * data arrays must contain the expected information.  The @c angle_sel
   * array contains integer indices into the @c fobj->param->angle array.
   * The output arrays are expected to be in a sensible state upon calling
   * (i.e. these arrays are not zeroed).  In particular, the forces are
   * accumulated into @c f_angle.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_angles(Force *fobj, double *u_angle, MD_Dvec f_angle[],
      double e_angle[], double virial[], const MD_Dvec pos[],
      const int32 angle_sel[], int32 angle_sel_len);

  /**@brief Compute contribution of dihedrals to force and potential.
   *
   * @param[out] u_dihed        Receives total potential energy.
   * @param[in,out] f_dihed     Receives forces on atoms.
   * @param[out] e_dihed        Receives interaction potentials.
   * @param[in,out] virial      Accumulates contribution to virial.
   * @param[in]  pos            Atomic position coordinates.
   * @param[in]  dihed_sel      Selects dihedrals from @c fobj->param->dihed.
   * @param[in]  dihed_sel_len  Length of dihedral selection array.
   *
   * Compute force and potentials due to dihedrals.  All parameters must
   * exist: array buffers must all be allocated and be of expected length and
   * data arrays must contain the expected information.  The @c dihed_sel
   * array contains integer indices into the @c fobj->param->dihed array.
   * The output arrays are expected to be in a sensible state upon calling
   * (i.e. these arrays are not zeroed).  In particular, the forces are
   * accumulated into @c f_dihed.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_dihedrals(Force *fobj, double *u_dihed, MD_Dvec f_dihed[],
      double e_dihed[], double virial[], const MD_Dvec pos[],
      const int32 dihed_sel[], int32 dihed_sel_len);

  /**@brief Compute contribution of impropers to force and potential.
   *
   * @param[out] u_impr        Receives total potential energy.
   * @param[in,out] f_impr     Receives forces on atoms.
   * @param[out] e_impr        Receives interaction potentials.
   * @param[in,out] virial     Accumulates contribution to virial.
   * @param[in]  pos           Atomic position coordinates.
   * @param[in]  impr_sel      Selects impropers from @c fobj->param->impr.
   * @param[in]  impr_sel_len  Length of improper selection array.
   *
   * Compute force and potentials due to impropers.  All parameters must
   * exist: array buffers must all be allocated and be of expected length and
   * data arrays must contain the expected information.  The @c impr_sel
   * array contains integer indices into the @c fobj->param->impr array.
   * The output arrays are expected to be in a sensible state upon calling
   * (i.e. these arrays are not zeroed).  In particular, the forces are
   * accumulated into @c f_impr.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_impropers(Force *fobj, double *u_impr, MD_Dvec f_impr[],
      double e_impr[], double virial[], const MD_Dvec pos[],
      const int32 impr_sel[], int32 impr_sel_len);


  /**@brief Compute an individual bond spring interaction.
   *
   * @param[in,out] f_bond  Receives forces on atoms.
   * @param[in,out] virial  Accumulates contribution to virial.
   * @param[in]  pos        Atomic position coordinates.
   * @param[in]  bond       Contains the bond.
   * @param[in]  prm        Contains the parameters for this bond.
   *
   * Compute force and potential for this individual spring bond.
   * The @c bond contains atom indices for the participating atoms,
   * so the entire @c f_bond and @c pos arrays are passed.  The @c prm
   * contains the force constants needed for this bond.
   *
   * @return the potential for this interaction.
   */
  double force_compute_bond_interaction(MD_Dvec f_bond[], double virial[],
      const MD_Dvec pos[], const MD_Bond *bond, const MD_BondPrm *prm);

  /**@brief Compute an individual bond angle interaction.
   *
   * @param[in,out] f_angle  Receives forces on atoms.
   * @param[in,out] virial   Accumulates contribution to virial.
   * @param[in]  pos         Atomic position coordinates.
   * @param[in]  bond        Contains the angle.
   * @param[in]  prm         Contains the parameters for this angle.
   *
   * Compute force and potential for this individual bond angle.
   * The @c angle contains atom indices for the participating atoms,
   * so the entire @c f_angle and @c pos arrays are passed.  The @c prm
   * contains the force constants needed for this bond angle.
   *
   * @return the potential for this interaction.
   */
  double force_compute_angle_interaction(MD_Dvec f_angle[], double virial[],
      const MD_Dvec pos[], const MD_Angle *angle, const MD_AnglePrm *prm);

  /**@brief Compute an individual torsion angle interaction.
   *
   * @param[in,out] f_tors  Receives forces on atoms.
   * @param[in,out] virial  Accumulates contribution to virial.
   * @param[in]  pos        Atomic position coordinates.
   * @param[in]  tors       Contains the torsion.
   * @param[in]  prm        Contains the parameters for this torsion.
   *
   * Compute force and potential for this individual torsion angle.
   * (This is used for contributions of both dihedrals and impropers.)
   * The @c tors contains atom indices for the participating atoms,
   * so the entire @c f_tors and @c pos arrays are passed.  The @c prm
   * contains the force constants needed for this torsion angle.
   *
   * @return the potential for this interaction.
   */
  double force_compute_torsion_interaction(MD_Dvec f_tors[], double virial[],
      const MD_Dvec pos[], const MD_Tors *tors, const MD_TorsPrm *prm);


  /**@brief Compute spherical boundary restraint.
   *
   * @param[out] u_bres        Receives total potential energy.
   * @param[in,out] f_bres     Receives forces on atoms.
   * @param[out] e_bres        Receives interaction potentials.
   * @param[in]  pos           Atomic position coordinates.
   * @param[in]  bres_sel      Selects atoms from @c fobj->param->atom.
   * @param[in]  bres_sel_len  Length of atom selection array.
   *
   * Compute force and potentials due to spherical boundary restraint.
   * All arguments must exist: array buffers must all be allocated and be
   * of the expected length and data arrays must contain the expected
   * information.  The @c bres_sel array contains integer indices into the
   * @c fobj->param->atom array.  The output arrays are expected to be in a
   * sensible state upon calling (i.e. these arrays are not zeroed).
   * In particular, the forces are accumulated into @c f_bres.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_bres_sphere(Force *fobj, double *u_bres,
      MD_Dvec f_bres[], double e_bres[], const MD_Dvec *pos,
      const int32 bres_sel[], int32 bres_sel_len);

  /**@brief Compute cylindrical boundary restraint.
   *
   * @param[out] u_bres        Receives total potential energy.
   * @param[in,out] f_bres     Receives forces on atoms.
   * @param[out] e_bres        Receives interaction potentials.
   * @param[in]  pos           Atomic position coordinates.
   * @param[in]  bres_sel      Selects atoms from @c fobj->param->atom.
   * @param[in]  bres_sel_len  Length of atom selection array.
   * @param[in]  bresopts      Gives orientation of cylinder.
   *
   * Compute force and potentials due to cylindrical boundary restraint.
   * All arguments must exist: array buffers must all be allocated and be
   * of the expected length and data arrays must contain the expected
   * information.  The @c bres_sel array contains integer indices into the
   * @c fobj->param->atom array.  The output arrays are expected to be in a
   * sensible state upon calling (i.e. these arrays are not zeroed).
   * In particular, the forces are accumulated into @c f_bres.  The
   * @c bresopts parameter gives orientation of cylinder, with value
   * one of @c FORCE_BRES_X_CYLINDER, @c FORCE_BRES_Y_CYLINDER, or
   * @c FORCE_BRES_Z_CYLINDER.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_bres_cylinder(Force *fobj, double *u_bres,
      MD_Dvec f_bres[], double e_bres[], const MD_Dvec *pos,
      const int32 *bres_sel, int32 bres_sel_len, int32 bresopts);


  /**@brief Use direct all pairs algorithm to compute nonbonded interactions.
   *
   * @param[in,out] virial       Accumulates nonbonded contribution to virial.
   * @param[out] u_elec          Receives total electrostatic potential energy.
   * @param[in,out] f_elec       Receives electrostatic forces on atoms.
   * @param[in,out] e_elec       Receives electrostatic interaction energies.
   * @param[in,out] e_epot       Receives electrostatic potentials.
   * @param[in]  is_elec_direct  Gives on/off status for electrostatics.
   * @param[in]  elec_pair_potential  Gives electrostatic pairwise potential.
   * @param[out] u_vdw           Receives total van der Waals potential energy.
   * @param[in,out] f_vdw        Receives van der Waals forces on atoms.
   * @param[in,out] e_vdw        Receives van der Waals interaction energies.
   * @param[in]  is_vdw_direct   Gives on/off status for van der Waals.
   * @param[in]  vdw_pair_potential   Gives van der Waals pairwise potential.
   * @param[in]  pos             Atomic position coordinates.
   * @param[in]  aset_sel        Set A selection of atoms.
   * @param[in]  aset_sel_len    Length of set A atom selection array.
   * @param[in]  bset_sel        Set B selection of atoms.
   * @param[in]  bset_sel_len    Length of set B atom selection array.
   *
   * Compute nonbonded electrostatics and/or van der Waals using simple
   * direct all-pairs algorithm.  All arguments must exist: array buffers
   * must all be allocated and be of the expected length and data arrays
   * must contain the expected information.  The two selection sets should
   * either point to the same array or to two nonempty arrays containing
   * no overlapping indices.
   *
   * Set @c is_elec_direct to nonzero to compute electrostatic interactions.
   * Set @c is_vdw_direct to nonzero to compute van der Waals interactions.
   * The @c elec_pair_potential parameter tells specifically what
   * electrostatic pairwise potential to use (e.g. @c FORCE_ELEC_STANDARD
   * is standard 1/r potential, @c FORCE_ELEC_SHIFTED is the 1/r potential
   * multiplied by a shifting function for smooth cutoff).  Similarly,
   * the @c vdw_pair_potential parameter tells specifically what
   * van der Waals pairwise potential to use (e.g. @c FORCE_VDW_STANDARD
   * is the standard potential, @c FORCE_VDW_SWITCHED is uses a switching
   * function to smoothly truncate the potential at the cutoff).
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_nbpairs_direct(Force *fobj, double virial[],
      double *u_elec, MD_Dvec f_elec[], double e_elec[], double e_epot[],
      int32 is_elec_direct, int32 elec_pair_potential,
      double *u_vdw, MD_Dvec f_vdw[], double e_vdw[],
      int32 is_vdw_direct, int32 vdw_pair_potential,
      const MD_Dvec pos[],
      const int32 aset_sel[], int32 aset_sel_len,
      const int32 bset_sel[], int32 bset_sel_len);

  /**@brief Apply geometric hashing of atomic positions into grid cells.
   *
   * @param[in] trpos    Scaled atomic position coordinates.
   * @param[in] sel      Select atom indices into @c fobj->param->atom.
   * @param[in] sel_len  Length of atom selection array.
   *
   * Atoms are hashed into grid cell linked lists using scaled atomic
   * coordinates.  Must be called before calling
   * @c force_compute_nbpairs_gridcells().  All arguments must
   * exist: data arrays must contain the expected information.
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_nbpairs_geomhash(Force *fobj,
      const MD_Dvec trpos[], const int32 sel[], int32 sel_len);

  /**@brief Compute cutoff nonbonded interactions using grid cells.
   *
   * @param[in,out] virial       Accumulates nonbonded contribution to virial.
   * @param[out] u_elec          Receives total electrostatic potential energy.
   * @param[in,out] f_elec       Receives electrostatic forces on atoms.
   * @param[in,out] e_elec       Receives electrostatic interaction energies.
   * @param[in,out] e_epot       Receives electrostatic potentials.
   * @param[in]  is_elec_gridcells    Gives on/off status for electrostatics.
   * @param[in]  elec_pair_potential  Gives electrostatic pairwise potential.
   * @param[out] u_vdw           Receives total van der Waals potential energy.
   * @param[in,out] f_vdw        Receives van der Waals forces on atoms.
   * @param[in,out] e_vdw        Receives van der Waals interaction energies.
   * @param[in]  is_vdw_gridcells     Gives on/off status for van der Waals.
   * @param[in]  vdw_pair_potential   Gives van der Waals pairwise potential.
   * @param[in]  pos             Atomic position coordinates.
   * @param[in]  is_subtracted   Tells to add (zero) or subtract (nonzero)
   *                             interactions to force and potential arrays.
   *
   * Compute cutoff nonbonded electrostatics and/or van der Waals using
   * grid cell hashing, with cost linear in the number of atoms.  Requires
   * calling force_compute_nbpairs_geomhash() first to fill the grid cells.
   * All arguments must exist: array buffers must all be allocated and be of
   * the expected length and data arrays must contain the expected information.
   * The selection of positions was done during the preceding hashing.
   *
   * Set @c is_elec_gridcells to nonzero to compute electrostatic interactions.
   * Set @c is_vdw_gridcells to nonzero to compute van der Waals interactions.
   * The @c elec_pair_potential parameter tells specifically what
   * electrostatic pairwise potential to use (e.g. @c FORCE_ELEC_STANDARD
   * is standard 1/r potential, @c FORCE_ELEC_SHIFTED is the 1/r potential
   * multiplied by a shifting function for smooth cutoff).  Similarly,
   * the @c vdw_pair_potential parameter tells specifically what
   * van der Waals pairwise potential to use (e.g. @c FORCE_VDW_STANDARD
   * is the standard potential, @c FORCE_VDW_SWITCHED is uses a switching
   * function to smoothly truncate the potential at the cutoff).
   *
   * The @c is_subtracted parameter indicates whether interactions should
   * be added (value: zero) or subtracted (value: nonzero) to the force and
   * potential arrays.  (Note: this is a kludge so that grid cell hashing can
   * be applied conveniently to compute potential differences between disjoint
   * sets of atoms.  The idea is to add interactions involving combined sets,
   * then subtract out separate contributions each set individually.)
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_nbpairs_gridcells(Force *fobj, double virial[],
      double *u_elec, MD_Dvec f_elec[], double e_elec[], double e_epot[],
      int32 is_elec_gridcells, int32 elec_pair_potential,
      double *u_vdw, MD_Dvec f_vdw[], double e_vdw[],
      int32 is_vdw_gridcells, int32 vdw_pair_potential,
      const MD_Dvec pos[], int32 is_subtracted);

  /**@brief Subtract excluded interactions for nonbonded pairwise interactions.
   *
   * @param[in,out] virial       Accumulates nonbonded contribution to virial.
   * @param[out] u_elec          Receives total electrostatic potential energy.
   * @param[in,out] f_elec       Receives electrostatic forces on atoms.
   * @param[in,out] e_elec       Receives electrostatic interaction energies.
   * @param[in,out] e_epot       Receives electrostatic potentials.
   * @param[in]  is_elec_subtexcl     Gives on/off status for electrostatics.
   * @param[in]  elec_pair_potential  Gives electrostatic pairwise potential.
   * @param[out] u_vdw           Receives total van der Waals potential energy.
   * @param[in,out] f_vdw        Receives van der Waals forces on atoms.
   * @param[in,out] e_vdw        Receives van der Waals interaction energies.
   * @param[in]  is_vdw_subtexcl      Gives on/off status for van der Waals.
   * @param[in]  vdw_pair_potential   Gives van der Waals pairwise potential.
   * @param[in]  pos             Atomic position coordinates.
   * @param[in]  sel             Select atom indices into @c fobj->param->atom.
   * @param[in]  sel_len         Length of atom selection arrays.
   * @param[in]  mapnb           Per atom map indicating set ownership.
   * @param[in]  map_id          ID of set to match against for exclusion.
   *
   * Subtract excluded interactions for nonbonded electrostatics and/or
   * van der Waals.  All arguments must exist: array buffers must all be
   * allocated and be of the expected length and data arrays must contain
   * the expected information.
   *
   * The atom selection array @c sel gives the indices of a set of atoms in
   * which to consider exclusions.  The second atom in the excluded interaction
   * pair is found by considering the exclusion list for each atom in @c sel,
   * and matching these against set ownership given by @c mapnb.  Atoms
   * matching @c map_id are taken to be the second atom in the exclusion.
   *
   * Here is how the above is applied in practice: for standard computation
   * of nonbonded interactions involving all atoms, then @c sel selects all
   * atoms (the identity map, 0..N-1) and @c mapnb contains
   * (@c FORCE_SELECT_ASET | @c FORCE_SELECT_BSET) meaning that every atom
   * belongs to both sets, so that is the value to assign to @c map_id.
   * When computing potential differences between two disjoint subsets,
   * the excluded interactions should span the two sets, so @c sel should
   * take on one set, say, @c ForceSelect_t::aset_sel and @c map_id should
   * be assigned the other set @c FORCE_SELECT_BSET.
   *
   * Set @c is_elec_subtexcl to nonzero to compute electrostatic interactions.
   * Set @c is_vdw_subtexcl to nonzero to compute van der Waals interactions.
   * The @c elec_pair_potential parameter tells specifically what
   * electrostatic pairwise potential to use (e.g. @c FORCE_ELEC_STANDARD
   * is standard 1/r potential, @c FORCE_ELEC_SHIFTED is the 1/r potential
   * multiplied by a shifting function for smooth cutoff).  Similarly,
   * the @c vdw_pair_potential parameter tells specifically what
   * van der Waals pairwise potential to use (e.g. @c FORCE_VDW_STANDARD
   * is the standard potential, @c FORCE_VDW_SWITCHED is uses a switching
   * function to smoothly truncate the potential at the cutoff).
   *
   * @return 0 for success or @c FORCE_FAIL on failure.
   */
  int force_compute_nbpairs_subtexcl(Force *fobj, double virial[],
      double *u_elec, MD_Dvec f_elec[], double e_elec[], double e_epot[],
      int32 is_elec_subtexcl, int32 elec_pair_potential,
      double *u_vdw, MD_Dvec f_vdw[], double e_vdw[],
      int32 is_vdw_subtexcl, int32 vdw_pair_potential,
      const MD_Dvec pos[], const int32 sel[], int32 sel_len,
      const int32 mapnb[], int32 map_id);


  /**@brief Compute a standard electrostatic interaction.
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  c     Multiplicative constant.
   *
   * Computes the standard c/r potential and the force scaling given
   * by D(c/r)/r.
   */
  void force_compute_nbpairs_elec_standard(double *u, double *du_r,
      double r2, double c);

  /**@brief Compute a shifted electrostatic interaction.
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  c     Multiplicative constant.
   * @param[in]  inv_elec_cutoff2  Inverse of square of cutoff distance.
   *
   * Computes the shifted electrostatic potential and force scaling.  This can
   * be expressed in the form (1/r) s(r), where s(r) is the shifting function
   * that is parameterized by @c inv_elec_cutoff2.  The exact function is
   * documented in the NAMD User Guide, as well as the X-Plor and CHARMM
   * literature.
   */
  void force_compute_nbpairs_elec_shifted(double *u, double *du_r,
      double r2, double c, double inv_elec_cutoff2);

  void force_compute_nbpairs_elec_ewald(double *u, double *du_r,
      double r2, double c, double ewald_coef, double grad_coef);

  /**@brief Compute a standard van der Waals interaction.
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  a     Multiplicative constant for r^{-12} term.
   * @param[in]  b     Multiplicative constant for r^{-6} term.
   *
   * Computes the standard van der Waals potential and force scaling.
   */
  void force_compute_nbpairs_vdw_standard(double *u, double *du_r,
      double r2, double a, double b);

  /**@brief Compute a switched van der Waals interaction.
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  a     Multiplicative constant for r^{-12} term.
   * @param[in]  b     Multiplicative constant for r^{-6} term.
   * @param[in]  vdw_cutoff2       Square of cutoff distance.
   * @param[in]  switch_dist2      Square of switching distance.
   * @param[in]  inv_denom_switch  Inverse of switching function denominator.
   *
   * Computes the switched van der Waals potential and force scaling.
   * This is piecewise defined, where it is the standard potential for
   * pairwise distance less than switching distance, or the standard
   * potential times the switching function for pairwise distance between
   * the switching distance and cutoff, and zero beyond the cutoff.
   * The switching function gives C^1 joins and is documented in the
   * NAMD User Guide, as well as the X-Plor and CHARMM literature.
   */
  void force_compute_nbpairs_vdw_switched(double *u, double *du_r,
      double r2, double a, double b,
      double vdw_cutoff2, double switch_dist2, double inv_denom_switch);

  /**@brief Compute Buckingham potential (replaces van der Waals).
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  a     constant
   * @param[in]  b     constant
   * @param[in]  c     constant
   *
   * Compute Buckingham form:  a exp(-r/b) - c/r^6
   *
   * See Flikkema & Bromley, Chem Phys Lett 378 (2003) 622-629,
   * for parameterization for silica.
   */
  void force_compute_nbpairs_vdw_buck(double *u, double *du_r,
      double r2, double a, double b, double c);

  /**@brief Compute switched Buckingham potential (replaces van der Waals).
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  a     constant
   * @param[in]  b     constant
   * @param[in]  c     constant
   * @param[in]  vdw_cutoff2       Square of cutoff distance.
   * @param[in]  switch_dist2      Square of switching distance.
   * @param[in]  inv_denom_switch  Inverse of switching function denominator.
   *
   * Compute Buckingham form:  (a exp(-r/b) - c/r^6) s(r),
   * where s(r) is same switching function as used for van der Waals.
   *
   * See Flikkema & Bromley, Chem Phys Lett 378 (2003) 622-629,
   * for parameterization for silica.
   */
  void force_compute_nbpairs_vdw_switchbuck(double *u, double *du_r,
      double r2, double a, double b, double c,
      double vdw_cutoff2, double switch_dist2, double inv_denom_switch);

  /**@brief Compute Buckingham without dispersion term potential
   * (replaces van der Waals).
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  a     constant
   * @param[in]  b     constant
   *
   * Compute Buckingham without dispersion form:  a exp(-r/b)
   */
  void force_compute_nbpairs_vdw_bucknd(double *u, double *du_r,
      double r2, double a, double b);

  /**@brief Compute switched Buckingham without dispersion term potential
   * (replaces van der Waals).
   *
   * @param[out] u     Receives potential energy.
   * @param[out] du_r  Receives 1/r times derivative of potential energy.
   * @param[in]  r2    Square of pairwise distance.
   * @param[in]  a     constant
   * @param[in]  b     constant
   * @param[in]  vdw_cutoff2       Square of cutoff distance.
   * @param[in]  switch_dist2      Square of switching distance.
   * @param[in]  inv_denom_switch  Inverse of switching function denominator.
   *
   * Compute Buckingham form:  a exp(-r/b) s(r),
   * where s(r) is same switching function as used for van der Waals.
   */
  void force_compute_nbpairs_vdw_switchbucknd(double *u, double *du_r,
      double r2, double a, double b,
      double vdw_cutoff2, double switch_dist2, double inv_denom_switch);

  void force_compute_nbpairs_vdw_bucksafe(double *u, double *du_r,
      double r2, double a, double b, double c,
      double rn2, double an, double bn);

  void force_compute_nbpairs_vdw_switchbucksafe(double *u, double *du_r,
      double r2, double a, double b, double c,
      double rn2, double an, double bn,
      double roff2, double ron2, double denom);

  int force_compute_nbpairs_isregen_pairlists(Force *fobj,
      double delta2, const MD_Dvec initpos[], const MD_Dvec pos[],
      const int32 sel[], int32 sel_len);
  int force_compute_nbpairs_regen_pairlists(Force *fobj,
      MD_Dvec initpos[], const MD_Dvec pos[],
      const int32 aset_sel[], int32 aset_sel_len,
      const int32 mapnb[], int32 aset_id, int32 bset_id);
  int force_compute_nbpairs_eval_pairlists(Force *fobj, double virial[],
      double *u_elec, MD_Dvec f_elec[], double e_elec[], double e_epot[],
      int32 is_elec_gridcells, int32 elec_pair_potential,
      double *u_vdw, MD_Dvec f_vdw[], double e_vdw[],
      int32 is_vdw_gridcells, int32 vdw_pair_potential,
      const MD_Dvec pos[], const int32 sel[], int32 sel_len,
      const int32 mapnb[], int32 map_id);

#ifdef __cplusplus
}
#endif

#endif /* COMPUTE_H */
