/*
 * Copyright (C) 2006 by David J. Hardy.  All rights reserved.
 *
 * compute.c
 *
 * Compute atomic forces and potentials.
 */

#include <stdlib.h>
#include <math.h>
#include "force/force.h"
#undef DEBUG_WATCH
#include "debug/debug.h"


/*
 * This routine manages calls to the medium level routines
 * that perform the actual computation.
 *
 * The complication is having to multiplex all of the possible
 * force types and algorithmic options for nonbonded forces
 * with allocated buffers in ForceResult object and atom/bond
 * selections in ForceSelect object.
 *
 * The ForceParam::forcetypes value is strictly followed, where
 * a force type is evaluated only if set on.  Beyond that, forces
 * and potentials are placed in array space as provided by
 * ForceResult and atom/bond selections are followed as provided
 * by ForceSelect.  No more work is done than is necessary to
 * compute the indicated selection, meaning that if the atom/bond
 * selection is K << N, then the work done is O(K) array elements
 * touched and no more than O(K^2) if FORCE_ELEC_DIRECT or
 * FORCE_VDW_DIRECT has been set.
 */
int force_compute(Force *fobj, ForceResult *fres, const MD_Dvec pos[])
{
  ForceParam *fprm = fobj->param;
  ForceSelect *fsel = fobj->select;

  const int32 forcetypes = fprm->forcetypes;
  const int32 elecopts = fprm->elecopts;
  const int32 vdwopts = fprm->vdwopts;
  const int32 bresopts = fprm->bresopts;

  const int32 *bond_sel = (fsel ? fsel->bond_sel : fobj->idmap);
  const int32 *angle_sel = (fsel ? fsel->angle_sel : fobj->idmap);
  const int32 *dihed_sel = (fsel ? fsel->dihed_sel : fobj->idmap);
  const int32 *impr_sel = (fsel ? fsel->impr_sel : fobj->idmap);
  const int32 *aset_sel = (fsel && fsel->aset_sel ?
      fsel->aset_sel : fobj->idmap);
  const int32 *bset_sel = (fsel && fsel->bset_sel ?
      fsel->bset_sel : fobj->idmap);
  const int32 *bres_sel = (fsel ? fsel->bres_sel : fobj->idmap);

  const int32 bond_sel_len = (fsel ? fsel->bond_sel_len : fprm->bond_len);
  const int32 angle_sel_len = (fsel ? fsel->angle_sel_len : fprm->angle_len);
  const int32 dihed_sel_len = (fsel ? fsel->dihed_sel_len : fprm->dihed_len);
  const int32 impr_sel_len = (fsel ? fsel->impr_sel_len : fprm->impr_len);
  const int32 aset_sel_len = (fsel && fsel->aset_sel ?
      fsel->aset_sel_len : fprm->atom_len);
  const int32 bset_sel_len = (fsel && fsel->bset_sel ?
      fsel->bset_sel_len : fprm->atom_len);
  const int32 bres_sel_len = (fsel ? fsel->bres_sel_len : fprm->atom_len);

  const int32 *atom_bond_sel = fobj->atom_bond_sel;
  const int32 *atom_angle_sel = fobj->atom_angle_sel;
  const int32 *atom_dihed_sel = fobj->atom_dihed_sel;
  const int32 *atom_impr_sel = fobj->atom_impr_sel;
  const int32 *abset_sel = fobj->abset_sel;
  const int32 *total_sel = fobj->total_sel;

  const int32 atom_bond_sel_len = fobj->atom_bond_sel_len;
  const int32 atom_angle_sel_len = fobj->atom_angle_sel_len;
  const int32 atom_dihed_sel_len = fobj->atom_dihed_sel_len;
  const int32 atom_impr_sel_len = fobj->atom_impr_sel_len;
  const int32 abset_sel_len = fobj->abset_sel_len;
  const int32 total_sel_len = fobj->total_sel_len;

  double *virial = fres->virial;
  MD_Dvec *f_total = fres->f_total;
  MD_Dvec *f_buffer = fobj->f_buffer;
  double *e_buffer = fobj->e_buffer;

  int32 i, j;

  TEXT("starting force_compute");

  /*
   * set force, potential, and virial array elements to zero
   */
  virial[FORCE_VIRIAL_XX] = 0.0;
  virial[FORCE_VIRIAL_XY] = 0.0;
  virial[FORCE_VIRIAL_XZ] = 0.0;
  virial[FORCE_VIRIAL_YX] = 0.0;
  virial[FORCE_VIRIAL_YY] = 0.0;
  virial[FORCE_VIRIAL_YZ] = 0.0;
  virial[FORCE_VIRIAL_ZX] = 0.0;
  virial[FORCE_VIRIAL_ZY] = 0.0;
  virial[FORCE_VIRIAL_ZZ] = 0.0;

  fres->u_total = 0.0;
  fres->u_bond = 0.0;
  fres->u_angle = 0.0;
  fres->u_dihed = 0.0;
  fres->u_impr = 0.0;
  fres->u_elec = 0.0;
  fres->u_vdw = 0.0;
  fres->u_bres = 0.0;

  force_select_reset_force(f_total, total_sel, total_sel_len);
  if (fres->f_bond) {
    force_select_reset_force(fres->f_bond, atom_bond_sel, atom_bond_sel_len);
  }
  if (fres->f_angle) {
    force_select_reset_force(fres->f_angle, atom_angle_sel, atom_angle_sel_len);
  }
  if (fres->f_dihed) {
    force_select_reset_force(fres->f_dihed, atom_dihed_sel, atom_dihed_sel_len);
  }
  if (fres->f_impr) {
    force_select_reset_force(fres->f_impr, atom_impr_sel, atom_impr_sel_len);
  }
  if (fres->f_elec) {
    force_select_reset_force(fres->f_elec, abset_sel, abset_sel_len);
  }
  if (fres->f_vdw) {
    force_select_reset_force(fres->f_vdw, abset_sel, abset_sel_len);
  }
  if (fres->f_bres) {
    force_select_reset_force(fres->f_bres, abset_sel, abset_sel_len);
  }

  if (fres->e_bond) {
    force_select_reset_potential(fres->e_bond, bond_sel, bond_sel_len);
  }
  if (fres->e_angle) {
    force_select_reset_potential(fres->e_angle, angle_sel, angle_sel_len);
  }
  if (fres->e_dihed) {
    force_select_reset_potential(fres->e_dihed, dihed_sel, dihed_sel_len);
  }
  if (fres->e_impr) {
    force_select_reset_potential(fres->e_impr, impr_sel, impr_sel_len);
  }
  if (fres->e_elec) {
    force_select_reset_potential(fres->e_elec, abset_sel, abset_sel_len);
  }
  if (fres->e_vdw) {
    force_select_reset_potential(fres->e_vdw, abset_sel, abset_sel_len);
  }
  if (fres->e_bres) {
    force_select_reset_potential(fres->e_bres, abset_sel, abset_sel_len);
  }
  if (fres->e_epot) {
    force_select_reset_potential(fres->e_epot, abset_sel, abset_sel_len);
  }

  /*
   * compute scaled coordinates
   *
   * performed either as a side effect of setting up a new domain
   * or explicitly
   */
  if (fobj->is_domain_update) {
    if (force_setup_domain(fobj, fobj->domain, pos,
          total_sel, total_sel_len)) {
      return FORCE_FAIL;
    }
    fobj->is_domain_update = 0;
  }
  else {
    TEXT("calling force_compute_scaled_coords");
    if (force_compute_scaled_coords(fobj, pos, total_sel, total_sel_len)) {
      return FORCE_FAIL;
    }
    TEXT("done force_compute_scaled_coords");
    if (fobj->is_periodic != FORCE_PERIODIC &&
        ++(fobj->cnt_steps) == fobj->max_steps) {
      fobj->is_domain_update = force_compute_domain_update(fobj,
          total_sel, total_sel_len);
      fobj->cnt_steps = 0;
    }
  }

  /*
   * compute force and potential
   */
  if (forcetypes & FORCE_BOND) {
    double *u_bond = &(fres->u_bond);
    MD_Dvec *f_bond = (fres->f_bond ? fres->f_bond : f_total);
    double *e_bond = (fres->e_bond ? fres->e_bond : e_buffer);
    if (force_compute_bonds(fobj, u_bond, f_bond, e_bond, virial,
          pos, bond_sel, bond_sel_len)) {
      return FORCE_FAIL;
    }
    if (fres->f_bond) {
      for (i = 0;  i < atom_bond_sel_len;  i++) {
        j = atom_bond_sel[i];
        f_total[j].x += f_bond[j].x;
        f_total[j].y += f_bond[j].y;
        f_total[j].z += f_bond[j].z;
      }
    }
  }
  TEXT("done bonds");

  if (forcetypes & FORCE_ANGLE) {
    double *u_angle = &(fres->u_angle);
    MD_Dvec *f_angle = (fres->f_angle ? fres->f_angle : f_total);
    double *e_angle = (fres->e_angle ? fres->e_angle : e_buffer);
    if (force_compute_angles(fobj, u_angle, f_angle, e_angle, virial,
          pos, angle_sel, angle_sel_len)) {
      return FORCE_FAIL;
    }
    if (fres->f_angle) {
      for (i = 0;  i < atom_angle_sel_len;  i++) {
        j = atom_angle_sel[i];
        f_total[j].x += f_angle[j].x;
        f_total[j].y += f_angle[j].y;
        f_total[j].z += f_angle[j].z;
      }
    }
  }
  TEXT("done angles");

  if (forcetypes & FORCE_DIHED) {
    double *u_dihed = &(fres->u_dihed);
    MD_Dvec *f_dihed = (fres->f_dihed ? fres->f_dihed : f_total);
    double *e_dihed = (fres->e_dihed ? fres->e_dihed : e_buffer);
    if (force_compute_dihedrals(fobj, u_dihed, f_dihed, e_dihed, virial,
          pos, dihed_sel, dihed_sel_len)) {
      return FORCE_FAIL;
    }
    if (fres->f_dihed) {
      for (i = 0;  i < atom_dihed_sel_len;  i++) {
        j = atom_dihed_sel[i];
        f_total[j].x += f_dihed[j].x;
        f_total[j].y += f_dihed[j].y;
        f_total[j].z += f_dihed[j].z;
      }
    }
  }
  TEXT("done diheds");

  if (forcetypes & FORCE_IMPR) {
    double *u_impr = &(fres->u_impr);
    MD_Dvec *f_impr = (fres->f_impr ? fres->f_impr : f_total);
    double *e_impr = (fres->e_impr ? fres->e_impr : e_buffer);
    if (force_compute_impropers(fobj, u_impr, f_impr, e_impr, virial,
          pos, impr_sel, impr_sel_len)) {
      return FORCE_FAIL;
    }
    if (fres->f_impr) {
      for (i = 0;  i < atom_impr_sel_len;  i++) {
        j = atom_impr_sel[i];
        f_total[j].x += f_impr[j].x;
        f_total[j].y += f_impr[j].y;
        f_total[j].z += f_impr[j].z;
      }
    }
  }
  TEXT("done imprs");

  if (forcetypes & FORCE_PAIRWISE) { /* begin nbpairs */
    double u_buffer;
    int32 is_elec = (forcetypes & FORCE_ELEC);
    int32 is_vdw = (forcetypes & FORCE_VDW);

    double *u_elec = (is_elec ? &(fres->u_elec) : &u_buffer);
    MD_Dvec *f_elec = (is_elec ?
        (fres->f_elec ? fres->f_elec : f_total) : f_buffer);
    double *e_elec = (is_elec && fres->e_elec ? fres->e_elec : e_buffer);
    double *e_epot = (is_elec && fres->e_epot ? fres->e_epot : e_buffer);

    double *u_vdw = (is_vdw ? &(fres->u_vdw) : &u_buffer);
    MD_Dvec *f_vdw = (is_vdw ?
        (fres->f_vdw ? fres->f_vdw : f_total) : f_buffer);
    double *e_vdw = (is_vdw && fres->e_vdw ? fres->e_vdw : e_buffer);

    int32 elec_pair_potential = (elecopts & FORCE_MASK_ELEC_POTENTIAL);
    int32 elec_excl_pair_potential = ((elecopts & FORCE_ELEC_STNDEXCL) ?
        FORCE_ELEC_STANDARD : elec_pair_potential);

    int32 is_elec_direct = (is_elec && (elecopts & FORCE_ELEC_DIRECT));
    int32 is_elec_gridcells = (is_elec && (elecopts & FORCE_ELEC_GRIDCELLS));
    int32 is_elec_pairlists = (is_elec && (elecopts & FORCE_ELEC_PAIRLISTS));
    int32 is_elec_subtexcl = (is_elec && !is_elec_pairlists
        && (elecopts & FORCE_MASK_ELEC_METHOD));

    int32 vdw_pair_potential = (vdwopts & FORCE_MASK_VDW_POTENTIAL);
    int32 vdw_excl_pair_potential = ((vdwopts & FORCE_VDW_STNDEXCL) ?
        FORCE_VDW_STANDARD : vdw_pair_potential);

    int32 is_vdw_direct = (is_vdw && (vdwopts & FORCE_VDW_DIRECT));
    int32 is_vdw_gridcells = (is_vdw && (vdwopts & FORCE_VDW_GRIDCELLS));
    int32 is_vdw_pairlists = (is_vdw && (vdwopts & FORCE_VDW_PAIRLISTS));
    int32 is_vdw_subtexcl = (is_vdw && !is_vdw_pairlists
        && (vdwopts & FORCE_MASK_VDW_METHOD));

    const int32 *mapnb = fobj->mapnb;
    int32 is_subsets = (aset_sel != bset_sel);

    if (is_elec_direct || is_vdw_direct) {
      if (force_compute_nbpairs_direct(fobj, virial,
            u_elec, f_elec, e_elec, e_epot,
            is_elec_direct, elec_pair_potential,
            u_vdw, f_vdw, e_vdw, is_vdw_direct, vdw_pair_potential,
            pos, aset_sel, aset_sel_len, bset_sel, bset_sel_len)) {
        return FORCE_FAIL;
      }
    } /* end direct */

    if (is_elec_pairlists || is_vdw_pairlists) {
      TEXT("computing pairlists");
      if (force_compute_nbpairs_isregen_pairlists(fobj, fobj->delta_dis2,
            fobj->pos_init, pos, abset_sel, abset_sel_len)
          && (force_compute_nbpairs_geomhash(fobj, fobj->trpos,
              abset_sel, abset_sel_len)
            || force_compute_nbpairs_regen_pairlists(fobj,
              fobj->pos_init, pos, aset_sel, aset_sel_len, mapnb,
              FORCE_INDEX_ASET, FORCE_INDEX_BSET))) {
        return FORCE_FAIL;
      }
      if (force_compute_nbpairs_eval_pairlists(fobj, virial,
            u_elec, f_elec, e_elec, e_epot,
            is_elec_pairlists, elec_pair_potential,
            u_vdw, f_vdw, e_vdw,
            is_vdw_pairlists, vdw_pair_potential,
            pos, aset_sel, aset_sel_len,
            mapnb, FORCE_INDEX_BSET)) {
        return FORCE_FAIL;
      }
    } /* end pairlists */

    if (is_elec_gridcells || is_vdw_gridcells) {
      TEXT("computing gridcells");
      TEXT("calling _geomhash");
      if (force_compute_nbpairs_geomhash(fobj, fobj->trpos,
            abset_sel, abset_sel_len)) {
        return FORCE_FAIL;
      }
      TEXT("done _geomhash");
      TEXT("calling _gridcells");
      if (force_compute_nbpairs_gridcells(fobj, virial,
            u_elec, f_elec, e_elec, e_epot,
            is_elec_gridcells, elec_pair_potential,
            u_vdw, f_vdw, e_vdw,
            is_vdw_gridcells, vdw_pair_potential, pos,
            0 /* default is to add interactions */)) {
        return FORCE_FAIL;
      }
      TEXT("done _gridcells");
      if (is_subsets) {
        TEXT("subsets with gridcells");
        /* must subtract out contributions of each set A and B alone */
        if (force_compute_nbpairs_geomhash(fobj, fobj->trpos,
              aset_sel, aset_sel_len)) {
          return FORCE_FAIL;
        }
        if (force_compute_nbpairs_gridcells(fobj, virial,
              u_elec, f_elec, e_elec, e_epot,
              is_elec_gridcells, elec_pair_potential,
              u_vdw, f_vdw, e_vdw,
              is_vdw_gridcells, vdw_pair_potential, pos,
              1 /* indicates subtracting interactions */)) {
          return FORCE_FAIL;
        }
        /* now subtract out contributions of set B alone */
        if (force_compute_nbpairs_geomhash(fobj, fobj->trpos,
              bset_sel, bset_sel_len)) {
          return FORCE_FAIL;
        }
        if (force_compute_nbpairs_gridcells(fobj, virial,
              u_elec, f_elec, e_elec, e_epot,
              is_elec_gridcells, elec_pair_potential,
              u_vdw, f_vdw, e_vdw,
              is_vdw_gridcells, vdw_pair_potential, pos,
              1 /* indicates subtracting interactions */)) {
          return FORCE_FAIL;
        }
        /* (when subt excl later, loop over, say, set A subt excl to set B) */
      }
    } /* end gridcells */

    if (is_elec_subtexcl || is_vdw_subtexcl) {
      if ( ! is_subsets ) {
        TEXT("subtr excl");
        /* compute this standard way, subtracting all exclusions */
        if (force_compute_nbpairs_subtexcl(fobj, virial,
              u_elec, f_elec, e_elec, e_epot,
              is_elec_subtexcl, elec_excl_pair_potential,
              u_vdw, f_vdw, e_vdw,
              is_vdw_subtexcl, vdw_excl_pair_potential,
              pos, abset_sel, abset_sel_len,
              mapnb, (FORCE_INDEX_ASET | FORCE_INDEX_BSET))) {
          return FORCE_FAIL;
        }
      }
      else {
        /* iterate over smallest set of atoms */
        const int32 *sel = (aset_sel_len < bset_sel_len ? aset_sel : bset_sel);
        int32 sel_len = (aset_sel_len < bset_sel_len ?
            aset_sel_len : bset_sel_len);
        /* match against opposite set ID */
        int32 map_id = (aset_sel_len < bset_sel_len ?
            FORCE_INDEX_BSET : FORCE_INDEX_ASET);

        /* subtract excluded interactions between sets */
        if (force_compute_nbpairs_subtexcl(fobj, virial,
              u_elec, f_elec, e_elec, e_epot,
              is_elec_subtexcl, elec_excl_pair_potential,
              u_vdw, f_vdw, e_vdw,
              is_vdw_subtexcl, vdw_excl_pair_potential,
              pos, sel, sel_len, mapnb, map_id)) {
          return FORCE_FAIL;
        }
      }
    } /* end subtexcl */

    if (fres->f_elec) {
      for (i = 0;  i < abset_sel_len;  i++) {
        j = abset_sel[i];
        f_total[j].x += f_elec[j].x;
        f_total[j].y += f_elec[j].y;
        f_total[j].z += f_elec[j].z;
      }
    }
    if (fres->f_vdw) {
      for (i = 0;  i < abset_sel_len;  i++) {
        j = abset_sel[i];
        f_total[j].x += f_vdw[j].x;
        f_total[j].y += f_vdw[j].y;
        f_total[j].z += f_vdw[j].z;
      }
    }

  } /* end nbpairs */
  TEXT("done nbpairs");

  if (forcetypes & FORCE_BRES) {
    double *u_bres = &(fres->u_bres);
    MD_Dvec *f_bres = (fres->f_bres ? fres->f_bres : f_total);
    double *e_bres = (fres->e_bres ? fres->e_bres : e_buffer);
    if (bresopts == FORCE_BRES_SPHERE) {
      if (force_compute_bres_sphere(fobj, u_bres, f_bres, e_bres, pos,
            bres_sel, bres_sel_len)) {
        return FORCE_FAIL;
      }
    }
    else {  /* cylinder */
      if (force_compute_bres_cylinder(fobj, u_bres, f_bres, e_bres, pos,
            bres_sel, bres_sel_len, bresopts)) {
        return FORCE_FAIL;
      }
    }
    if (fres->f_bres) {
      for (i = 0;  i < abset_sel_len;  i++) {
        j = abset_sel[i];
        f_total[j].x += f_bres[j].x;
        f_total[j].y += f_bres[j].y;
        f_total[j].z += f_bres[j].z;
      }
    }
  }

  /*
   * sum total potential energy
   */
  fres->u_total = fres->u_bond + fres->u_angle + fres->u_dihed
    + fres->u_impr + fres->u_elec + fres->u_vdw + fres->u_bres;

  return 0;
}


int force_compute_scaled_coords(Force *fobj, const MD_Dvec pos[],
    const int32 sel[], int32 sel_len)
{
  MD_Dvec *wrap = fobj->poswrap;  /* position wrapping vector */
  MD_Dvec *trpos = fobj->trpos;   /* stores scaled coordinates */

  const MD_Dvec ta1 = fobj->ta1;  /* row vectors of transformation matrix */
  const MD_Dvec ta2 = fobj->ta2;
  const MD_Dvec ta3 = fobj->ta3;

  const MD_Dvec v1 = fobj->v1;    /* cell basis vectors */
  const MD_Dvec v2 = fobj->v2;
  const MD_Dvec v3 = fobj->v3;

  const MD_Dvec lowerc = fobj->lowerc;  /* lower corner of domain */

  MD_Dvec smin = { 1.0, 1.0, 1.0 };     /* initial min/max values */
  MD_Dvec smax = { 0.0, 0.0, 0.0 };

  MD_Dvec p;

  const int32 is_minmax = fobj->is_minmax;

  const int32 is_x_periodic = fobj->is_periodic & FORCE_X_PERIODIC; 
  const int32 is_y_periodic = fobj->is_periodic & FORCE_Y_PERIODIC; 
  const int32 is_z_periodic = fobj->is_periodic & FORCE_Z_PERIODIC; 

  int32 i, j;

  /*
   * here orthogonal means transformation is diagonal matrix,
   * i.e. cell basis vectors v1, v2, v3 are in direction of
   * x, y, z axes, respectively
   */
  if (fobj->is_orthogonal) {

    /* compute transformation using only diagonal elements of matrix */
    for (i = 0;  i < sel_len;  i++) {
      j = sel[i];
      p.x = pos[j].x - lowerc.x + wrap[j].x;
      p.y = pos[j].y - lowerc.y + wrap[j].y;
      p.z = pos[j].z - lowerc.z + wrap[j].z;
      trpos[j].x = ta1.x * p.x;
      trpos[j].y = ta2.y * p.y;
      trpos[j].z = ta3.z * p.z;

      /* adjust coordinate wraping back into periodic domain */
      if (is_x_periodic) {
        if (trpos[j].x < 0.0) {
          trpos[j].x += 1.0;
          wrap[j].x += v1.x;
        }
        else if (trpos[j].x >= 1.0) {
          trpos[j].x -= 1.0;
          wrap[j].x -= v1.x;
        }
      }
      else if (is_minmax) {
        if (trpos[j].x < smin.x) smin.x = trpos[j].x;
        else if (trpos[j].x > smax.x) smax.x = trpos[j].x;
      }

      if (is_y_periodic) {
        if (trpos[j].y < 0.0) {
          trpos[j].y += 1.0;
          wrap[j].y += v2.y;
        }
        else if (trpos[j].y >= 1.0) {
          trpos[j].y -= 1.0;
          wrap[j].y -= v2.y;
        }
      }
      else if (is_minmax) {
        if (trpos[j].y < smin.y) smin.y = trpos[j].y;
        else if (trpos[j].y > smax.y) smax.y = trpos[j].y;
      }

      if (is_z_periodic) {
        if (trpos[j].z < 0.0) {
          trpos[j].z += 1.0;
          wrap[j].z += v3.z;
        }
        else if (trpos[j].z >= 1.0) {
          trpos[j].z -= 1.0;
          wrap[j].z -= v3.z;
        }
      }
      else if (is_minmax) {
        if (trpos[j].z < smin.z) smin.z = trpos[j].z;
        else if (trpos[j].z > smax.z) smax.z = trpos[j].z;
      }

    } /* end for loop */

  } /* end orthogonal case */

  else { /* nonorthogonal case */

    /* compute transformation */
    for (i = 0;  i < sel_len;  i++) {
      j = sel[i];
      p.x = pos[j].x - lowerc.x + wrap[j].x;
      p.y = pos[j].y - lowerc.y + wrap[j].y;
      p.z = pos[j].z - lowerc.z + wrap[j].z;
      trpos[j].x = ta1.x * p.x + ta1.y * p.y + ta1.z * p.z;
      trpos[j].y = ta2.x * p.x + ta2.y * p.y + ta2.z * p.z;
      trpos[j].z = ta3.x * p.x + ta3.y * p.y + ta3.z * p.z;

      /* adjust coordinate wraping back into periodic domain */
      if (is_x_periodic) {
        if (trpos[j].x < 0.0) {
          trpos[j].x += 1.0;
          wrap[j].x += v1.x;
          wrap[j].y += v1.y;
          wrap[j].z += v1.z;
        }
        else if (trpos[j].x >= 1.0) {
          trpos[j].x -= 1.0;
          wrap[j].x -= v1.x;
          wrap[j].y -= v1.y;
          wrap[j].z -= v1.z;
        }
      }
      else if (is_minmax) {
        if (trpos[j].x < smin.x) smin.x = trpos[j].x;
        else if (trpos[j].x > smax.x) smax.x = trpos[j].x;
      }

      if (is_y_periodic) {
        if (trpos[j].y < 0.0) {
          trpos[j].y += 1.0;
          wrap[j].x += v2.x;
          wrap[j].y += v2.y;
          wrap[j].z += v2.z;
        }
        else if (trpos[j].y >= 1.0) {
          trpos[j].y -= 1.0;
          wrap[j].x -= v2.x;
          wrap[j].y -= v2.y;
          wrap[j].z -= v2.z;
        }
      }
      else if (is_minmax) {
        if (trpos[j].y < smin.y) smin.y = trpos[j].y;
        else if (trpos[j].y > smax.y) smax.y = trpos[j].y;
      }

      if (is_z_periodic) {
        if (trpos[j].z < 0.0) {
          trpos[j].z += 1.0;
          wrap[j].x += v3.x;
          wrap[j].y += v3.y;
          wrap[j].z += v3.z;
        }
        else if (trpos[j].z >= 1.0) {
          trpos[j].z -= 1.0;
          wrap[j].x -= v3.x;
          wrap[j].y -= v3.y;
          wrap[j].z -= v3.z;
        }
      }
      else if (is_minmax) {
        if (trpos[j].z < smin.z) smin.z = trpos[j].z;
        else if (trpos[j].z > smax.z) smax.z = trpos[j].z;
      }

    } /* end for loop */

  } /* end of nonorthogonal case */

  /* retain min/max values for scaled coordinates */
  if (is_minmax) {
    fobj->smin = smin;
    fobj->smax = smax;
  }

  return 0;
}


int force_compute_domain_update(Force *f, const int32 sel[], int32 sel_len)
{
  const MD_Dvec v1 = f->v1;
  const MD_Dvec v2 = f->v2;
  const MD_Dvec v3 = f->v3;

  const MD_Dvec lowerc = f->lowerc;
  const MD_Dvec center = f->center;

  MD_Dvec scn = { 0.0, 0.0, 0.0 };
  MD_Dvec cn;

  double inv_natoms;

  const MD_Dvec *s = f->trpos;

  int32 need_update = 0;
  int32 i, j;

  /* compute new scaled center as average over scaled coordinates */
  for (i = 0;  i < sel_len;  i++) {
    j = sel[i];
    scn.x += s[j].x;
    scn.y += s[j].y;
    scn.z += s[j].z;
  }
  inv_natoms = 1.0 / (double) f->param->atom_len;
  scn.x *= inv_natoms;
  scn.y *= inv_natoms;
  scn.z *= inv_natoms;

  if (scn.x < 0.0 || scn.x > 1.0
      || scn.y < 0.0 || scn.y > 1.0
      || scn.z < 0.0 || scn.z > 1.0) {
    TEXT("*** updating domain ***");

    cn.x = scn.x*v1.x + scn.y*v2.x + scn.z*v3.x + lowerc.x;
    cn.y = scn.x*v1.y + scn.y*v2.y + scn.z*v3.y + lowerc.y;
    cn.z = scn.x*v1.z + scn.y*v2.z + scn.z*v3.z + lowerc.z;

    if (f->is_fixed_lattice) {
      const MD_Dvec dv1 = f->dv1;
      const MD_Dvec dv2 = f->dv2;
      const MD_Dvec dv3 = f->dv3;
      const MD_Dvec tb1 = f->tb1;
      const MD_Dvec tb2 = f->tb2;
      const MD_Dvec tb3 = f->tb3;
      MD_Dvec diff;
      MD_Dvec nu;
      MD_Dvec kappa;

      TEXT("*** move center while leaving transformation fixed ***");

      diff.x = cn.x - center.x;
      diff.y = cn.y - center.y;
      diff.z = cn.z - center.z;
      nu.x = tb1.x*diff.x + tb1.y*diff.y + tb1.z*diff.z;
      nu.y = tb2.x*diff.x + tb2.y*diff.y + tb2.z*diff.z;
      nu.z = tb3.x*diff.x + tb3.y*diff.y + tb3.z*diff.z;
      kappa.x = floor(nu.x + 0.5);
      kappa.y = floor(nu.y + 0.5);
      kappa.z = floor(nu.z + 0.5);

      VEC(kappa);

      cn.x = dv1.x*kappa.x + dv2.x*kappa.y + dv3.x*kappa.z + center.x;
      cn.y = dv1.y*kappa.x + dv2.y*kappa.y + dv3.y*kappa.z + center.y;
      cn.z = dv1.z*kappa.x + dv2.z*kappa.y + dv3.z*kappa.z + center.z;
      f->center = cn;

      f->lowerc.x = cn.x - 0.5 * (v1.x + v2.x + v3.x);
      f->lowerc.y = cn.y - 0.5 * (v1.y + v2.y + v3.y);
      f->lowerc.z = cn.z - 0.5 * (v1.z + v2.z + v3.z);

      need_update = 0;
    }
    else {
      need_update = 1;
    }

    f->domain->center = cn;  /* save new center */
  }

  return need_update;
}
